/*
 * Decompiled with CFR 0.152.
 */
package org.chromium.media;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.media.DeniedByServerException;
import android.media.MediaCrypto;
import android.media.MediaCryptoException;
import android.media.MediaDrm;
import android.media.MediaDrmException;
import android.media.NotProvisionedException;
import android.media.UnsupportedSchemeException;
import android.os.Build;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;
import org.chromium.base.Log;
import org.chromium.base.annotations.CalledByNative;
import org.chromium.base.annotations.JNINamespace;
import org.chromium.base.annotations.MainDex;

@JNINamespace(value="media")
@SuppressLint(value={"WrongConstant"})
@TargetApi(value=19)
public class MediaDrmBridge {
    private static final String TAG = "cr_media";
    private static final String SECURITY_LEVEL = "securityLevel";
    private static final String SERVER_CERTIFICATE = "serviceCertificate";
    private static final String PRIVACY_MODE = "privacyMode";
    private static final String SESSION_SHARING = "sessionSharing";
    private static final String ENABLE = "enable";
    private static final char[] HEX_CHAR_LOOKUP = "0123456789ABCDEF".toCharArray();
    private static final long INVALID_NATIVE_MEDIA_DRM_BRIDGE = 0L;
    private static final UUID WIDEVINE_UUID = UUID.fromString("edef8ba9-79d6-4ace-a3c8-27dcd51d21ed");
    private static final byte[] DUMMY_KEY_ID = new byte[]{0};
    private MediaDrm mMediaDrm;
    private long mNativeMediaDrmBridge;
    private UUID mSchemeUUID;
    private byte[] mMediaCryptoSession;
    private HashMap<ByteBuffer, String> mSessionIds;
    private ArrayDeque<PendingCreateSessionData> mPendingCreateSessionDataQueue;
    private boolean mResetDeviceCredentialsPending;
    private boolean mProvisioningPending;

    private static List<KeyStatus> getDummyKeysInfo(int statusCode) {
        ArrayList<KeyStatus> keysInfo = new ArrayList<KeyStatus>();
        keysInfo.add(new KeyStatus(DUMMY_KEY_ID, statusCode));
        return keysInfo;
    }

    private static UUID getUUIDFromBytes(byte[] data) {
        int i;
        if (data.length != 16) {
            return null;
        }
        long mostSigBits = 0L;
        long leastSigBits = 0L;
        for (i = 0; i < 8; ++i) {
            mostSigBits = mostSigBits << 8 | (long)(data[i] & 0xFF);
        }
        for (i = 8; i < 16; ++i) {
            leastSigBits = leastSigBits << 8 | (long)(data[i] & 0xFF);
        }
        return new UUID(mostSigBits, leastSigBits);
    }

    private static String bytesToHexString(byte[] bytes) {
        StringBuilder hexString = new StringBuilder();
        for (int i = 0; i < bytes.length; ++i) {
            hexString.append(HEX_CHAR_LOOKUP[bytes[i] >>> 4]);
            hexString.append(HEX_CHAR_LOOKUP[bytes[i] & 0xF]);
        }
        return hexString.toString();
    }

    private boolean isNativeMediaDrmBridgeValid() {
        return this.mNativeMediaDrmBridge != 0L;
    }

    private boolean isWidevine() {
        return this.mSchemeUUID.equals(WIDEVINE_UUID);
    }

    @TargetApi(value=23)
    private MediaDrmBridge(UUID schemeUUID, long nativeMediaDrmBridge) throws UnsupportedSchemeException {
        this.mSchemeUUID = schemeUUID;
        this.mMediaDrm = new MediaDrm(schemeUUID);
        this.mNativeMediaDrmBridge = nativeMediaDrmBridge;
        assert (this.isNativeMediaDrmBridgeValid());
        this.mSessionIds = new HashMap();
        this.mPendingCreateSessionDataQueue = new ArrayDeque();
        this.mResetDeviceCredentialsPending = false;
        this.mProvisioningPending = false;
        this.mMediaDrm.setOnEventListener((MediaDrm.OnEventListener)new EventListener());
        if (Build.VERSION.SDK_INT >= 23) {
            this.mMediaDrm.setOnExpirationUpdateListener((MediaDrm.OnExpirationUpdateListener)new ExpirationUpdateListener(), null);
            this.mMediaDrm.setOnKeyStatusChangeListener((MediaDrm.OnKeyStatusChangeListener)new KeyStatusChangeListener(), null);
        }
        if (this.isWidevine()) {
            this.mMediaDrm.setPropertyString(PRIVACY_MODE, ENABLE);
            this.mMediaDrm.setPropertyString(SESSION_SHARING, ENABLE);
        }
    }

    private boolean createMediaCrypto() {
        assert (this.mMediaDrm != null);
        assert (!this.mProvisioningPending);
        assert (this.mMediaCryptoSession == null);
        try {
            this.mMediaCryptoSession = this.openSession();
        }
        catch (NotProvisionedException e) {
            Log.d(TAG, "Device not provisioned", (Object)e);
            this.startProvisioning();
            return true;
        }
        if (this.mMediaCryptoSession == null) {
            Log.e(TAG, "Cannot create MediaCrypto Session.", new Object[0]);
            return false;
        }
        Log.d(TAG, "MediaCrypto Session created: %s", MediaDrmBridge.bytesToHexString(this.mMediaCryptoSession));
        try {
            if (MediaCrypto.isCryptoSchemeSupported((UUID)this.mSchemeUUID)) {
                MediaCrypto mediaCrypto = new MediaCrypto(this.mSchemeUUID, this.mMediaCryptoSession);
                Log.d(TAG, "MediaCrypto successfully created!");
                this.onMediaCryptoReady(mediaCrypto);
                return true;
            }
            Log.e(TAG, "Cannot create MediaCrypto for unsupported scheme.", new Object[0]);
        }
        catch (MediaCryptoException e) {
            Log.e(TAG, "Cannot create MediaCrypto", new Object[]{e});
        }
        try {
            this.mMediaDrm.closeSession(this.mMediaCryptoSession);
        }
        catch (Exception e) {
            Log.e(TAG, "closeSession failed: ", e);
        }
        this.mMediaCryptoSession = null;
        return false;
    }

    private byte[] openSession() throws NotProvisionedException {
        assert (this.mMediaDrm != null);
        try {
            byte[] sessionId = this.mMediaDrm.openSession();
            return (byte[])sessionId.clone();
        }
        catch (RuntimeException e) {
            Log.e(TAG, "Cannot open a new session", e);
            this.release();
            return null;
        }
        catch (NotProvisionedException e) {
            throw e;
        }
        catch (MediaDrmException e) {
            Log.e(TAG, "Cannot open a new session", new Object[]{e});
            this.release();
            return null;
        }
    }

    @CalledByNative
    private static boolean isCryptoSchemeSupported(byte[] schemeUUID, String containerMimeType) {
        UUID cryptoScheme = MediaDrmBridge.getUUIDFromBytes(schemeUUID);
        if (containerMimeType.isEmpty()) {
            return MediaDrm.isCryptoSchemeSupported((UUID)cryptoScheme);
        }
        return MediaDrm.isCryptoSchemeSupported((UUID)cryptoScheme, (String)containerMimeType);
    }

    @CalledByNative
    private static MediaDrmBridge create(byte[] schemeUUID, String securityLevel, long nativeMediaDrmBridge) {
        UUID cryptoScheme = MediaDrmBridge.getUUIDFromBytes(schemeUUID);
        if (cryptoScheme == null || !MediaDrm.isCryptoSchemeSupported((UUID)cryptoScheme)) {
            return null;
        }
        MediaDrmBridge mediaDrmBridge = null;
        try {
            mediaDrmBridge = new MediaDrmBridge(cryptoScheme, nativeMediaDrmBridge);
            Log.d(TAG, "MediaDrmBridge successfully created.");
        }
        catch (UnsupportedSchemeException e) {
            Log.e(TAG, "Unsupported DRM scheme", new Object[]{e});
            return null;
        }
        catch (IllegalArgumentException e) {
            Log.e(TAG, "Failed to create MediaDrmBridge", e);
            return null;
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to create MediaDrmBridge", e);
            return null;
        }
        if (!securityLevel.isEmpty() && !mediaDrmBridge.setSecurityLevel(securityLevel)) {
            return null;
        }
        if (!mediaDrmBridge.createMediaCrypto()) {
            return null;
        }
        return mediaDrmBridge;
    }

    private boolean setSecurityLevel(String securityLevel) {
        if (!this.isWidevine()) {
            Log.d(TAG, "Security level is not supported.");
            return true;
        }
        assert (this.mMediaDrm != null);
        assert (!securityLevel.isEmpty());
        String currentSecurityLevel = this.mMediaDrm.getPropertyString(SECURITY_LEVEL);
        Log.e(TAG, "Security level: current %s, new %s", currentSecurityLevel, securityLevel);
        if (securityLevel.equals(currentSecurityLevel)) {
            return true;
        }
        try {
            this.mMediaDrm.setPropertyString(SECURITY_LEVEL, securityLevel);
            return true;
        }
        catch (IllegalArgumentException e) {
            Log.e(TAG, "Failed to set security level %s", securityLevel, e);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to set security level %s", securityLevel, e);
        }
        Log.e(TAG, "Security level %s not supported!", securityLevel);
        return false;
    }

    @CalledByNative
    private boolean setServerCertificate(byte[] certificate) {
        if (!this.isWidevine()) {
            Log.d(TAG, "Setting server certificate is not supported.");
            return true;
        }
        try {
            this.mMediaDrm.setPropertyByteArray(SERVER_CERTIFICATE, certificate);
            return true;
        }
        catch (IllegalArgumentException e) {
            Log.e(TAG, "Failed to set server certificate", e);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to set server certificate", e);
        }
        return false;
    }

    @CalledByNative
    private void resetDeviceCredentials() {
        if (this.mMediaDrm == null) {
            this.onResetDeviceCredentialsCompleted(false);
            return;
        }
        this.mResetDeviceCredentialsPending = true;
        this.startProvisioning();
    }

    @CalledByNative
    private void destroy() {
        this.mNativeMediaDrmBridge = 0L;
        if (this.mMediaDrm != null) {
            this.release();
        }
    }

    private void release() {
        assert (this.mMediaDrm != null);
        for (PendingCreateSessionData data : this.mPendingCreateSessionDataQueue) {
            this.onPromiseRejected(data.promiseId(), "Create session aborted.");
        }
        this.mPendingCreateSessionDataQueue.clear();
        this.mPendingCreateSessionDataQueue = null;
        for (ByteBuffer sessionId : this.mSessionIds.keySet()) {
            try {
                this.mMediaDrm.removeKeys(sessionId.array());
            }
            catch (Exception e) {
                Log.e(TAG, "removeKeys failed: ", e);
            }
            try {
                this.mMediaDrm.closeSession(sessionId.array());
            }
            catch (Exception e) {
                Log.e(TAG, "closeSession failed: ", e);
            }
            this.onSessionClosed(sessionId.array());
        }
        this.mSessionIds.clear();
        this.mSessionIds = null;
        if (this.mMediaCryptoSession == null) {
            this.onMediaCryptoReady(null);
        } else {
            try {
                this.mMediaDrm.closeSession(this.mMediaCryptoSession);
            }
            catch (Exception e) {
                Log.e(TAG, "closeSession failed: ", e);
            }
            this.mMediaCryptoSession = null;
        }
        if (this.mResetDeviceCredentialsPending) {
            this.mResetDeviceCredentialsPending = false;
            this.onResetDeviceCredentialsCompleted(false);
        }
        if (this.mMediaDrm != null) {
            this.mMediaDrm.release();
            this.mMediaDrm = null;
        }
    }

    private MediaDrm.KeyRequest getKeyRequest(byte[] sessionId, byte[] data, String mime, HashMap<String, String> optionalParameters) throws NotProvisionedException {
        MediaDrm.KeyRequest request;
        block6: {
            assert (this.mMediaDrm != null);
            assert (this.mMediaCryptoSession != null);
            assert (!this.mProvisioningPending);
            if (optionalParameters == null) {
                optionalParameters = new HashMap();
            }
            request = null;
            try {
                request = this.mMediaDrm.getKeyRequest(sessionId, data, mime, 1, optionalParameters);
            }
            catch (IllegalStateException e) {
                if (Build.VERSION.SDK_INT < 21 || !(e instanceof MediaDrm.MediaDrmStateException)) break block6;
                Log.e(TAG, "MediaDrmStateException fired during getKeyRequest().", e);
            }
        }
        String result = request != null ? "successed" : "failed";
        Log.d(TAG, "getKeyRequest %s!", result);
        return request;
    }

    private void savePendingCreateSessionData(byte[] initData, String mime, HashMap<String, String> optionalParameters, long promiseId) {
        Log.d(TAG, "savePendingCreateSessionData()");
        this.mPendingCreateSessionDataQueue.offer(new PendingCreateSessionData(initData, mime, optionalParameters, promiseId));
    }

    private void processPendingCreateSessionData() {
        Log.d(TAG, "processPendingCreateSessionData()");
        assert (this.mMediaDrm != null);
        while (this.mMediaDrm != null && !this.mProvisioningPending && !this.mPendingCreateSessionDataQueue.isEmpty()) {
            PendingCreateSessionData pendingData = this.mPendingCreateSessionDataQueue.poll();
            byte[] initData = pendingData.initData();
            String mime = pendingData.mimeType();
            HashMap optionalParameters = pendingData.optionalParameters();
            long promiseId = pendingData.promiseId();
            this.createSession(initData, mime, optionalParameters, promiseId);
        }
    }

    @CalledByNative
    private void createSessionFromNative(byte[] initData, String mime, String[] optionalParamsArray, long promiseId) {
        HashMap<String, String> optionalParameters = new HashMap<String, String>();
        if (optionalParamsArray != null) {
            if (optionalParamsArray.length % 2 != 0) {
                throw new IllegalArgumentException("Additional data array doesn't have equal keys/values");
            }
            for (int i = 0; i < optionalParamsArray.length; i += 2) {
                optionalParameters.put(optionalParamsArray[i], optionalParamsArray[i + 1]);
            }
        }
        this.createSession(initData, mime, optionalParameters, promiseId);
    }

    private void createSession(byte[] initData, String mime, HashMap<String, String> optionalParameters, long promiseId) {
        Log.d(TAG, "createSession()");
        if (this.mMediaDrm == null) {
            Log.e(TAG, "createSession() called when MediaDrm is null.", new Object[0]);
            this.onPromiseRejected(promiseId, "MediaDrm released previously.");
            return;
        }
        if (this.mProvisioningPending) {
            this.savePendingCreateSessionData(initData, mime, optionalParameters, promiseId);
            return;
        }
        assert (this.mMediaCryptoSession != null);
        boolean newSessionOpened = false;
        byte[] sessionId = null;
        try {
            sessionId = this.openSession();
            if (sessionId == null) {
                this.onPromiseRejected(promiseId, "Open session failed.");
                return;
            }
            newSessionOpened = true;
            assert (!this.sessionExists(sessionId));
            MediaDrm.KeyRequest request = null;
            request = this.getKeyRequest(sessionId, initData, mime, optionalParameters);
            if (request == null) {
                try {
                    this.mMediaDrm.closeSession(sessionId);
                }
                catch (Exception e) {
                    Log.e(TAG, "closeSession failed", e);
                }
                this.onPromiseRejected(promiseId, "Generate request failed.");
                return;
            }
            Log.d(TAG, "createSession(): Session (%s) created.", MediaDrmBridge.bytesToHexString(sessionId));
            this.onPromiseResolvedWithSession(promiseId, sessionId);
            this.onSessionMessage(sessionId, request);
            this.mSessionIds.put(ByteBuffer.wrap(sessionId), mime);
        }
        catch (NotProvisionedException e) {
            Log.e(TAG, "Device not provisioned", new Object[]{e});
            if (newSessionOpened) {
                try {
                    this.mMediaDrm.closeSession(sessionId);
                }
                catch (Exception ex) {
                    Log.e(TAG, "closeSession failed", ex);
                }
            }
            this.savePendingCreateSessionData(initData, mime, optionalParameters, promiseId);
            this.startProvisioning();
        }
    }

    private boolean sessionExists(byte[] sessionId) {
        if (this.mMediaCryptoSession == null) {
            assert (this.mSessionIds.isEmpty());
            Log.e(TAG, "Session doesn't exist because media crypto session is not created.", new Object[0]);
            return false;
        }
        return !Arrays.equals(sessionId, this.mMediaCryptoSession) && this.mSessionIds.containsKey(ByteBuffer.wrap(sessionId));
    }

    @CalledByNative
    private void closeSession(byte[] sessionId, long promiseId) {
        Log.d(TAG, "closeSession()");
        if (this.mMediaDrm == null) {
            this.onPromiseRejected(promiseId, "closeSession() called when MediaDrm is null.");
            return;
        }
        if (!this.sessionExists(sessionId)) {
            this.onPromiseRejected(promiseId, "Invalid sessionId in closeSession(): " + MediaDrmBridge.bytesToHexString(sessionId));
            return;
        }
        try {
            this.mMediaDrm.removeKeys(sessionId);
        }
        catch (Exception e) {
            Log.e(TAG, "removeKeys failed: ", e);
        }
        try {
            this.mMediaDrm.closeSession(sessionId);
        }
        catch (Exception e) {
            Log.e(TAG, "closeSession failed: ", e);
        }
        this.mSessionIds.remove(ByteBuffer.wrap(sessionId));
        this.onPromiseResolved(promiseId);
        this.onSessionClosed(sessionId);
        Log.d(TAG, "Session %s closed", MediaDrmBridge.bytesToHexString(sessionId));
    }

    @CalledByNative
    private void updateSession(byte[] sessionId, byte[] response, long promiseId) {
        Log.d(TAG, "updateSession()");
        if (this.mMediaDrm == null) {
            this.onPromiseRejected(promiseId, "updateSession() called when MediaDrm is null.");
            return;
        }
        if (!this.sessionExists(sessionId)) {
            this.onPromiseRejected(promiseId, "Invalid session in updateSession: " + MediaDrmBridge.bytesToHexString(sessionId));
            return;
        }
        try {
            try {
                this.mMediaDrm.provideKeyResponse(sessionId, response);
            }
            catch (IllegalStateException e) {
                Log.e(TAG, "Exception intentionally caught when calling provideKeyResponse()", e);
            }
            Log.d(TAG, "Key successfully added for session %s", MediaDrmBridge.bytesToHexString(sessionId));
            this.onPromiseResolved(promiseId);
            if (Build.VERSION.SDK_INT < 23) {
                this.onSessionKeysChange(sessionId, MediaDrmBridge.getDummyKeysInfo(0).toArray(), true);
            }
            return;
        }
        catch (NotProvisionedException e) {
            Log.e(TAG, "failed to provide key response", new Object[]{e});
        }
        catch (DeniedByServerException e) {
            Log.e(TAG, "failed to provide key response", new Object[]{e});
        }
        this.onPromiseRejected(promiseId, "Update session failed.");
        this.release();
    }

    @CalledByNative
    private String getSecurityLevel() {
        if (this.mMediaDrm == null || !this.isWidevine()) {
            Log.e(TAG, "getSecurityLevel(): MediaDrm is null or security level is not supported.", new Object[0]);
            return null;
        }
        return this.mMediaDrm.getPropertyString(SECURITY_LEVEL);
    }

    private void startProvisioning() {
        if (this.mProvisioningPending) {
            Log.d(TAG, "startProvisioning: another provisioning is in progress, returning");
            return;
        }
        Log.d(TAG, "startProvisioning");
        this.mProvisioningPending = true;
        assert (this.mMediaDrm != null);
        MediaDrm.ProvisionRequest request = this.mMediaDrm.getProvisionRequest();
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnStartProvisioning(this.mNativeMediaDrmBridge, request.getDefaultUrl(), request.getData());
        }
    }

    @CalledByNative
    private void processProvisionResponse(boolean isResponseReceived, byte[] response) {
        boolean success;
        Log.d(TAG, "processProvisionResponse()");
        if (this.mMediaDrm == null) {
            return;
        }
        assert (this.mProvisioningPending);
        this.mProvisioningPending = false;
        boolean bl = success = isResponseReceived ? this.provideProvisionResponse(response) : false;
        if (this.mResetDeviceCredentialsPending) {
            this.onResetDeviceCredentialsCompleted(success);
            this.mResetDeviceCredentialsPending = false;
        }
        if (!success || this.mMediaCryptoSession == null && !this.createMediaCrypto()) {
            this.release();
            return;
        }
        this.processPendingCreateSessionData();
    }

    boolean provideProvisionResponse(byte[] response) {
        if (response == null || response.length == 0) {
            Log.e(TAG, "Invalid provision response.", new Object[0]);
            return false;
        }
        try {
            this.mMediaDrm.provideProvisionResponse(response);
            return true;
        }
        catch (DeniedByServerException e) {
            Log.e(TAG, "failed to provide provision response", new Object[]{e});
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "failed to provide provision response", e);
        }
        return false;
    }

    private void onMediaCryptoReady(MediaCrypto mediaCrypto) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnMediaCryptoReady(this.mNativeMediaDrmBridge, mediaCrypto);
        }
    }

    private void onPromiseResolved(long promiseId) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnPromiseResolved(this.mNativeMediaDrmBridge, promiseId);
        }
    }

    private void onPromiseResolvedWithSession(long promiseId, byte[] sessionId) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnPromiseResolvedWithSession(this.mNativeMediaDrmBridge, promiseId, sessionId);
        }
    }

    private void onPromiseRejected(long promiseId, String errorMessage) {
        Log.e(TAG, "onPromiseRejected: %s", errorMessage);
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnPromiseRejected(this.mNativeMediaDrmBridge, promiseId, errorMessage);
        }
    }

    @TargetApi(value=23)
    private void onSessionMessage(byte[] sessionId, MediaDrm.KeyRequest request) {
        if (!this.isNativeMediaDrmBridgeValid()) {
            return;
        }
        int requestType = 0;
        requestType = Build.VERSION.SDK_INT >= 23 ? request.getRequestType() : (request.getDefaultUrl().isEmpty() ? 0 : 1);
        this.nativeOnSessionMessage(this.mNativeMediaDrmBridge, sessionId, requestType, request.getData(), request.getDefaultUrl());
    }

    private void onSessionClosed(byte[] sessionId) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnSessionClosed(this.mNativeMediaDrmBridge, sessionId);
        }
    }

    private void onSessionKeysChange(byte[] sessionId, Object[] keysInfo, boolean hasAdditionalUsableKey) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnSessionKeysChange(this.mNativeMediaDrmBridge, sessionId, keysInfo, hasAdditionalUsableKey);
        }
    }

    private void onSessionExpirationUpdate(byte[] sessionId, long expirationTime) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnSessionExpirationUpdate(this.mNativeMediaDrmBridge, sessionId, expirationTime);
        }
    }

    private void onLegacySessionError(byte[] sessionId, String errorMessage) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnLegacySessionError(this.mNativeMediaDrmBridge, sessionId, errorMessage);
        }
    }

    private void onResetDeviceCredentialsCompleted(boolean success) {
        if (this.isNativeMediaDrmBridgeValid()) {
            this.nativeOnResetDeviceCredentialsCompleted(this.mNativeMediaDrmBridge, success);
        }
    }

    private native void nativeOnMediaCryptoReady(long var1, MediaCrypto var3);

    private native void nativeOnStartProvisioning(long var1, String var3, byte[] var4);

    private native void nativeOnPromiseResolved(long var1, long var3);

    private native void nativeOnPromiseResolvedWithSession(long var1, long var3, byte[] var5);

    private native void nativeOnPromiseRejected(long var1, long var3, String var5);

    private native void nativeOnSessionMessage(long var1, byte[] var3, int var4, byte[] var5, String var6);

    private native void nativeOnSessionClosed(long var1, byte[] var3);

    private native void nativeOnSessionKeysChange(long var1, byte[] var3, Object[] var4, boolean var5);

    private native void nativeOnSessionExpirationUpdate(long var1, byte[] var3, long var4);

    private native void nativeOnLegacySessionError(long var1, byte[] var3, String var4);

    private native void nativeOnResetDeviceCredentialsCompleted(long var1, boolean var3);

    @TargetApi(value=23)
    @MainDex
    private class ExpirationUpdateListener
    implements MediaDrm.OnExpirationUpdateListener {
        private ExpirationUpdateListener() {
        }

        public void onExpirationUpdate(MediaDrm md, byte[] sessionId, long expirationTime) {
            Log.d(MediaDrmBridge.TAG, "ExpirationUpdate: " + MediaDrmBridge.bytesToHexString(sessionId) + ", " + expirationTime);
            MediaDrmBridge.this.onSessionExpirationUpdate(sessionId, expirationTime);
        }
    }

    @TargetApi(value=23)
    @MainDex
    private class KeyStatusChangeListener
    implements MediaDrm.OnKeyStatusChangeListener {
        private KeyStatusChangeListener() {
        }

        private List<KeyStatus> getKeysInfo(List<MediaDrm.KeyStatus> keyInformation) {
            ArrayList<KeyStatus> keysInfo = new ArrayList<KeyStatus>();
            for (MediaDrm.KeyStatus keyStatus : keyInformation) {
                keysInfo.add(new KeyStatus(keyStatus.getKeyId(), keyStatus.getStatusCode()));
            }
            return keysInfo;
        }

        public void onKeyStatusChange(MediaDrm md, byte[] sessionId, List<MediaDrm.KeyStatus> keyInformation, boolean hasNewUsableKey) {
            Log.d(MediaDrmBridge.TAG, "KeysStatusChange: " + MediaDrmBridge.bytesToHexString(sessionId) + ", " + hasNewUsableKey);
            MediaDrmBridge.this.onSessionKeysChange(sessionId, this.getKeysInfo(keyInformation).toArray(), hasNewUsableKey);
        }
    }

    @MainDex
    private class EventListener
    implements MediaDrm.OnEventListener {
        private EventListener() {
        }

        public void onEvent(MediaDrm mediaDrm, byte[] sessionId, int event, int extra, byte[] data) {
            if (sessionId == null) {
                Log.e(MediaDrmBridge.TAG, "EventListener: Null session.", new Object[0]);
                return;
            }
            if (!MediaDrmBridge.this.sessionExists(sessionId)) {
                Log.e(MediaDrmBridge.TAG, "EventListener: Invalid session %s", MediaDrmBridge.bytesToHexString(sessionId));
                return;
            }
            switch (event) {
                case 2: {
                    Log.d(MediaDrmBridge.TAG, "MediaDrm.EVENT_KEY_REQUIRED");
                    if (MediaDrmBridge.this.mProvisioningPending) {
                        return;
                    }
                    String mime = (String)MediaDrmBridge.this.mSessionIds.get(ByteBuffer.wrap(sessionId));
                    MediaDrm.KeyRequest request = null;
                    try {
                        request = MediaDrmBridge.this.getKeyRequest(sessionId, data, mime, null);
                    }
                    catch (NotProvisionedException e) {
                        Log.e(MediaDrmBridge.TAG, "Device not provisioned", new Object[]{e});
                        MediaDrmBridge.this.startProvisioning();
                        return;
                    }
                    if (request != null) {
                        MediaDrmBridge.this.onSessionMessage(sessionId, request);
                        break;
                    }
                    MediaDrmBridge.this.onLegacySessionError(sessionId, "MediaDrm EVENT_KEY_REQUIRED: Failed to generate request.");
                    if (Build.VERSION.SDK_INT < 23) {
                        MediaDrmBridge.this.onSessionKeysChange(sessionId, MediaDrmBridge.getDummyKeysInfo(4).toArray(), false);
                    }
                    Log.e(MediaDrmBridge.TAG, "EventListener: getKeyRequest failed.", new Object[0]);
                    return;
                }
                case 3: {
                    Log.d(MediaDrmBridge.TAG, "MediaDrm.EVENT_KEY_EXPIRED");
                    MediaDrmBridge.this.onLegacySessionError(sessionId, "MediaDrm EVENT_KEY_EXPIRED.");
                    if (Build.VERSION.SDK_INT >= 23) break;
                    MediaDrmBridge.this.onSessionKeysChange(sessionId, MediaDrmBridge.getDummyKeysInfo(1).toArray(), false);
                    break;
                }
                case 4: {
                    Log.d(MediaDrmBridge.TAG, "MediaDrm.EVENT_VENDOR_DEFINED");
                    assert (false);
                    break;
                }
                default: {
                    Log.e(MediaDrmBridge.TAG, "Invalid DRM event " + event, new Object[0]);
                    return;
                }
            }
        }
    }

    @MainDex
    private static class PendingCreateSessionData {
        private final byte[] mInitData;
        private final String mMimeType;
        private final HashMap<String, String> mOptionalParameters;
        private final long mPromiseId;

        private PendingCreateSessionData(byte[] initData, String mimeType, HashMap<String, String> optionalParameters, long promiseId) {
            this.mInitData = initData;
            this.mMimeType = mimeType;
            this.mOptionalParameters = optionalParameters;
            this.mPromiseId = promiseId;
        }

        private byte[] initData() {
            return this.mInitData;
        }

        private String mimeType() {
            return this.mMimeType;
        }

        private HashMap<String, String> optionalParameters() {
            return this.mOptionalParameters;
        }

        private long promiseId() {
            return this.mPromiseId;
        }
    }

    @MainDex
    private static class KeyStatus {
        private final byte[] mKeyId;
        private final int mStatusCode;

        private KeyStatus(byte[] keyId, int statusCode) {
            this.mKeyId = keyId;
            this.mStatusCode = statusCode;
        }

        @CalledByNative(value="KeyStatus")
        private byte[] getKeyId() {
            return this.mKeyId;
        }

        @CalledByNative(value="KeyStatus")
        private int getStatusCode() {
            return this.mStatusCode;
        }
    }
}

