package jp.agentec.abook.abv.ui.common.activity;

import android.app.Dialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Color;
import android.net.ConnectivityManager;
import android.os.Bundle;
import android.os.Handler;
import android.text.InputType;
import android.view.KeyEvent;
import android.view.View;
import android.view.Window;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.RadioGroup;
import android.widget.TextView;
import android.widget.Toast;

import com.google.firebase.iid.FirebaseInstanceId;

import java.io.FileNotFoundException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import jp.agentec.abook.abv.bl.acms.client.json.DownloadedContentInfoJSON;
import jp.agentec.abook.abv.bl.acms.client.json.content.ContentJSON;
import jp.agentec.abook.abv.bl.acms.type.DownloadStatusType;
import jp.agentec.abook.abv.bl.common.ABVEnvironment;
import jp.agentec.abook.abv.bl.common.CommonExecutor;
import jp.agentec.abook.abv.bl.common.Constant;
import jp.agentec.abook.abv.bl.common.Constant.AlertMessageLevel;
import jp.agentec.abook.abv.bl.common.constant.ABookKeys;
import jp.agentec.abook.abv.bl.common.exception.ABVException;
import jp.agentec.abook.abv.bl.common.exception.ABVExceptionCode;
import jp.agentec.abook.abv.bl.common.exception.ABVRuntimeException;
import jp.agentec.abook.abv.bl.common.exception.AcmsException;
import jp.agentec.abook.abv.bl.common.exception.NetworkDisconnectedException;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.common.util.ContentFileUtil;
import jp.agentec.abook.abv.bl.data.ABVDataCache;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.ContentDao;
import jp.agentec.abook.abv.bl.data.dao.OperationDao;
import jp.agentec.abook.abv.bl.download.ContentDownloadListener;
import jp.agentec.abook.abv.bl.download.ContentDownloader;
import jp.agentec.abook.abv.bl.download.ContentRefresher;
import jp.agentec.abook.abv.bl.download.ContentZipDownloadNotification;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.abook.abv.bl.dto.FixPushMessageDto;
import jp.agentec.abook.abv.bl.dto.OperationDto;
import jp.agentec.abook.abv.bl.dto.PushMessageDto;
import jp.agentec.abook.abv.bl.logic.AbstractLogic;
import jp.agentec.abook.abv.bl.logic.BookmarkLogic;
import jp.agentec.abook.abv.bl.logic.ContractLogic;
import jp.agentec.abook.abv.bl.logic.MemoLogic;
import jp.agentec.abook.abv.bl.logic.PushMessageLogic;
import jp.agentec.abook.abv.bl.logic.UserAuthenticateLogic;
import jp.agentec.abook.abv.cl.helper.ABVUncaughtExceptionHandler;
import jp.agentec.abook.abv.cl.util.PreferenceUtil;
import jp.agentec.abook.abv.cl.util.StorageUtil;
import jp.agentec.abook.abv.launcher.android.ABVApplication;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType.DefPrefKey;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType.UserPrefKey;
import jp.agentec.abook.abv.ui.common.constant.ErrorCode;
import jp.agentec.abook.abv.ui.common.constant.ErrorMessage;
import jp.agentec.abook.abv.ui.common.constant.NaviConsts;
import jp.agentec.abook.abv.ui.common.dialog.ABookAlertDialog;
import jp.agentec.abook.abv.ui.common.helper.ProgressDialogHelper;
import jp.agentec.abook.abv.ui.common.util.ABVToastUtil;
import jp.agentec.abook.abv.ui.common.util.AlertDialogUtil;
import jp.agentec.abook.abv.ui.common.util.DisplayUtil;
import jp.agentec.abook.abv.ui.common.util.Initializer;
import jp.agentec.abook.abv.ui.common.util.KeyboardUtils;
import jp.agentec.abook.abv.ui.common.util.PatternStringUtil;
import jp.agentec.abook.abv.ui.common.view.ABVBatchSyncView;
import jp.agentec.abook.abv.ui.common.vo.Size;
import jp.agentec.abook.abv.ui.home.activity.HelpActivity;
import jp.agentec.abook.abv.ui.home.activity.LoginActivity;
import jp.agentec.abook.abv.ui.home.activity.OperationListActivity;
import jp.agentec.abook.abv.ui.home.adapter.FixPushMessageAdapter;
import jp.agentec.abook.abv.ui.home.adapter.OperationSelectAdapter;
import jp.agentec.abook.abv.ui.home.adapter.PushMessageListAdapter;
import jp.agentec.abook.abv.ui.home.helper.ActivityHandlingHelper;
import jp.agentec.abook.abv.ui.home.helper.ContentViewHelper;
import jp.agentec.adf.net.http.HttpDownloadSimpleNotification;
import jp.agentec.adf.util.DateTimeFormat;
import jp.agentec.adf.util.DateTimeUtil;
import jp.agentec.adf.util.StringUtil;

public abstract class ABVAuthenticatedActivity extends ABVActivity implements ContentDownloadListener {
	private final static String TAG = "ABVAuthenticatedActivity";

	public static final String FILEPATH = "FILEPATH";

	//連続タップ防止用のボタン活性化するタイム
	protected static final int BUTTON_ENABLE_DELAY_MILLIS = 500;

	protected ContentDao contentDao = AbstractDao.getDao(ContentDao.class);

	protected MemoLogic memoLogic = AbstractLogic.getLogic(MemoLogic.class);
	protected BookmarkLogic bookmarkLogic = AbstractLogic.getLogic(BookmarkLogic.class);
	protected ContentDownloader contentDownloader = ContentDownloader.getInstance();
	protected ContentRefresher contentRefresher = ContentRefresher.getInstance();

	private ExecutorService initilizeExecutor = Executors.newFixedThreadPool(2); // DL後の初期化専用

	protected ActivityHandlingHelper activityHandlingHelper;
	protected Size mDisplaySize;


	// 一括同期ビュー（コントロール）
	protected ABVBatchSyncView batchSyncView;

	protected ABVUIActivity abvuiActivity;

	protected Dialog mCommunicationMenuDialog;
	protected Dialog mOperationSelectDialog;
	protected ListView mOperationSelectListView;
	protected Dialog mPushMessageListDialog;
	protected ListView mPushMessageListView;
	protected OperationSelectAdapter mOperationSelectAdapter;
	protected PushMessageListAdapter mPushMessageListAdapter;
	protected ListView mFixPushMessageListView;
	protected Dialog mPushMessageSendDialog;
	protected ImageButton communicationButton;                    // コミュニケーションボタン
	protected int mSelectedFixPuchMessagePosition;
	protected int mSendType;
	protected PushMessageLogic pushMessageLogic = AbstractLogic.getLogic(PushMessageLogic.class);

	/**
	 * メッセージ表示タイプ
	 */
	public interface MessageType {
		int TYPE_SHOW_NONE = 0;
		int TYPE_SHOW_TOAST = 1;
		int TYPE_SHOW_ALERT = 2;
	}

	public interface ContentCheckResultType {
		int NETWORK_ERROR = -2;
		int ERROR = -1;
		int SUCCESS = 0;
		int NO_AUTH_DELETE = 1;
		int CHANGED_META_INFO = 2;
		int PAYMENT = 3;
	}

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		sendRegistrationIdToCMS();
		activityHandlingHelper = ActivityHandlingHelper.getInstance();
		activityHandlingHelper.setCurrentActivity(this);
	}

	private void sendRegistrationIdToCMS() {
		// Tokenが更新されたか確認
		if (!PreferenceUtil.getUserPref(this, UserPrefKey.NEED_SEND_TOKEN, false)){
			// 更新されてない
			return;
		}
		String fcmToken = FirebaseInstanceId.getInstance().getToken();
		boolean isSendSuccess = false;
		// CMSに送信
		if (getRInteger(R.integer.push_message) == 1 && fcmToken != null) {
			try {
				// 既にMacAddress使用に同意している場合はMacAddressを使う
				// 認証なしの場合はUUIDを使用する
				//noinspection IfStatementWithNegatedCondition
				if (ABVEnvironment.getInstance().deviceIdType != Constant.DeviceIdType.MAC_ADDRESS) {
					AbstractLogic.getLogic(UserAuthenticateLogic.class).updateDeviceTokenByUUID(fcmToken);
				} else {
					AbstractLogic.getLogic(UserAuthenticateLogic.class).updateDeviceTokenByMacAdress(fcmToken);
				}
				isSendSuccess = true;
			} catch (NetworkDisconnectedException e) {
				Logger.w(TAG, "[sendRegistrationIdToCMS] NetworkDisconnectedException");
			} catch (Exception e) {
				Logger.e(TAG, "[sendRegistrationIdToCMS]", e);
			}
		}
		// CMSへ更新した結果をセットする
		PreferenceUtil.putUserPref(this, UserPrefKey.NEED_SEND_TOKEN, !isSendSuccess);
	}


	@Override
	protected void onStart() {
		super.onStart();
		// 前回起動時にアプリが強制終了した場合、オプションで送信オンになっている場合は例外トレース情報を送信する（設定によりメールかCMSへ送信）
		ABVUncaughtExceptionHandler.getInstancer().sendBugReport(this);
	}

	@Override
	protected void onResume() {
		super.onResume();
		if (!ABVEnvironment.getInstance().isInitialized()) {
			Initializer.getInstance().init(getABVApplication());
		}
		validateLoginStatus();

		if (ABVEnvironment.getInstance().acmsAddress == null) {
			loadAddressIfSaasGeneral();
		}

		contentDownloader.addContentDownloadListener(this);
	}

	@Override
	protected void onPause() {
		super.onPause();
		if (!(this instanceof OperationListActivity)) {
			// 作業一覧ではない場合、ダウンロードリスナーを削除する
			contentDownloader.removeContentDownloadListener(this);
		}
	}

	@Override
	protected void onDestroy() {
		super.onDestroy();
		activityHandlingHelper.removeCurrentActivity(this);
	}

	@Override
	protected void onApplicationBroughtFromBackground() {
		if (isShowingBatchSync()) {
			// 一括同期中は以下の処理を行わないようにする
			return;
		}
		new Handler().post(new Runnable() {
			@Override
			public void run() {
				if (checkForceOfflineLogout()) {
					finish();
				} else {
					if (checkValidAuthTime()) {
						finish();
					} else {
						checkRequiredChangePassword(null);
					}
				}
			}
		});
	}

	@Override
	protected void onPasswordChanged() {
	}

	private void validateLoginStatus() {
		boolean isLoggedIn = true;

		try {
			isLoggedIn = (ABVDataCache.getInstance().getMemberInfo() != null);
		} catch (ABVRuntimeException e) {
			Logger.e("onResume failed", e);

			if (e.getCode() == ABVExceptionCode.C_E_SECURITY_1004) {
				isLoggedIn = false;
			}
		} catch (Exception e) {
			Logger.e("onResume failed", e);
		}

		if (!isLoggedIn) {
			ErrorMessage.showErrorMessageToast(this, ErrorCode.C_E_SECURITY_1004);

			Intent intent = new Intent();
			intent.setClassName(getApplicationContext().getPackageName(), LoginActivity.class.getName()).setFlags(Intent.FLAG_ACTIVITY_NO_HISTORY);
			startActivity(intent);
			finish();
		}
	}

	/**
	 * Viewを開始します。
	 */
	public void startContentViewActivity(Intent intent, long contentId) {
		// 該当コンテンツが360コンテンツ＆直前にある場合、startActivityで移動せず、履歴で残っているActivityをそのまま表示する。
		if (checkPanoContent(contentId) && ActivityHandlingHelper.getInstance().hasPreviousPanoContentId(contentId)) {
			finish();
			return;
		}

		// 直前のActivityにはないが、同じ360コンテンツが存在している場合、それをfinishさせる。
		ActivityHandlingHelper.getInstance().deleteExistPanoContentId(contentId);

		// コンテンツアラートのサービスオプションがYの時はアラートを表示
		if (AbstractLogic.getLogic(ContractLogic.class).getContentAlert()) {
			contentViewActivityMoveWithContentAlert(intent, contentId, NaviConsts.Right);
		} else {
			contentViewActivityMoveWithAccessLocationAlert(intent, contentId, NaviConsts.Right);
		}
	}

	/**
	 * 360コンテンツ使用可能かのチェック
	 * バーチャルツアー：OS4.4以上
	 * 360動画：OS5.0以上
	 * @param contentId
	 * @return
	 */
	private boolean checkPanoContent(long contentId) {
		int ret = ContentViewHelper.getInstance().check360Content(contentId);
		return ret == 0 ? false : true;
	}

	private void contentViewActivityMoveWithContentAlert(final Intent intent, final long contentId, final NaviConsts ABVNavi) {
		ContentDto contentDto = contentDao.getContent(contentId);
		// コンテンツ表示制限チェック
		switch (contentDto.alertMessageLevel) {
			case AlertMessageLevel.NONE :
				contentViewActivityMoveWithAccessLocationAlert(intent, contentId, ABVNavi);
				break;
			case AlertMessageLevel.MESSAGE :
				showContentMessageDialog(intent, contentId, contentDto.alertMessage, ABVNavi);
				break;
			case AlertMessageLevel.PASSWORD :
				showPasswordCheckDialog(intent, contentId, null, ABVNavi);
				break;
		}
	}

	private void contentViewActivityMoveWithAccessLocationAlert(final Intent intent, final long contentId, final NaviConsts ABVNavi) {
		if (!getABVUIDataCache().isShowedPermissionAccessLocationAlert() && ( AbstractLogic.getLogic(ContractLogic.class).getUsableReadinglogGps())) {
			// 位置情報取得許可アラートを一度も表示していない場合表示する
			putUserPref(UserPrefKey.SHOWED_PERMISSION_ACCESS_LOCATION_ALERT, true);
			showPermissionAccessLocationAlert(intent, contentId, ABVNavi);
		} else {
			contentViewActivityMove(intent, contentId, ABVNavi);
		}
	}

	private void showPermissionAccessLocationAlert(final Intent intent, final long contentId, final NaviConsts ABVNavi) {
		ABookAlertDialog dialog = AlertDialogUtil.createAlertDialog(this, R.string.app_name);
		dialog.setMessage(String.format(getString(R.string.access_location_message), getString(R.string.app_name)));
		dialog.setPositiveButton(R.string.yes, new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int which) {
				putUserPref(DefPrefKey.PERMISSION_ACCESS_LOCATION, true);
				contentViewActivityMove(intent, contentId, ABVNavi);
			}
		});
		dialog.setNegativeButton(R.string.no, new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int which) {
				putUserPref(DefPrefKey.PERMISSION_ACCESS_LOCATION, false);
				contentViewActivityMove(intent, contentId, ABVNavi);
			}
		});
		showAlertDialog(dialog);
	}

	private void contentViewActivityMove(final Intent intent, final long contentId, final NaviConsts ABVNavi) {
		Logger.v(TAG, "contentViewActivityMove. contentId=%s", contentId);
		final Context context = this;
		if (isRotatable()) {
			setFixedOrientation(true);
		}
		showProgressPopup();
		CommonExecutor.execute(new Runnable() {
			@Override
			public void run() {
				try {
					Thread.sleep(500);
					activityHandlingHelper.startContentActivity(context, intent, ABVNavi, null, contentId);
				} catch (FileNotFoundException e) {
					Logger.e(TAG, "startContentActivity " + contentId, e);
					handleErrorMessageToast(ErrorCode.E105);
				} catch (ActivityNotFoundException e) {
					Logger.e(TAG, "startContentActivity " + contentId, e);
					handleErrorMessageToast(ErrorCode.E112);
				} catch (Exception e) {
					Logger.e(TAG, "startContentActivity " + contentId, e);
					handleErrorMessageToast(ErrorCode.E107);
				} finally {
					closeProgressPopup();
					if (ABVAuthenticatedActivity.this instanceof ShowPushMessageDailogActivity) {
						if (getABVApplication().isAutoOpenAfterDownloadContent()) {
							//コンテンツダウンロードして自動的にコンテンツを開く場合：自動的に開くフラグ更新
							((ABVApplication)getApplication()).setAutoOpenAfterDownloadContent(false);
						} else {
							//ダウンロード済みコンテンツを開く場合：プッシュ―アクティビティを終了
							finish();
						}
					}
					if (isRotatable()) {
						setFixedOrientation(false);
					}
					finish();
				}
			}
		});
	}

	public void showContentMessageDialog(final Intent intent, final long contentId, String message, final NaviConsts ABVNavi) {
		ABookAlertDialog dialog = AlertDialogUtil.createAlertDialog(this, R.string.app_name);
		dialog.setMessage(message);
		dialog.setPositiveButton(R.string.confirm, new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int which) {
				contentViewActivityMoveWithAccessLocationAlert(intent, contentId, ABVNavi);
			}
		});
		dialog.setNegativeButton(R.string.cancel, null);
		showAlertDialog(dialog);
	}

	public void showPasswordCheckDialog(final Intent intent, final long contentId, String additionalMessage, final NaviConsts ABVNavi) {
		ABookAlertDialog dialog = AlertDialogUtil.createAlertDialog(this, R.string.app_name);

		LinearLayout layout = new LinearLayout(this);
		layout.setOrientation(LinearLayout.VERTICAL);
		layout.setPadding(10, 0, 10, 0);

		TextView messageView = new TextView(this);
		messageView.setText(getString(R.string.L123));
		layout.addView(messageView);
		final EditText edtPassword = new EditText(this);
		edtPassword.setSingleLine(true);
		edtPassword.setInputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_VARIATION_PASSWORD);
		layout.addView(edtPassword);

		if (additionalMessage != null) {
			TextView additionalMessageView = new TextView(this);
			additionalMessageView.setText(additionalMessage);
			additionalMessageView.setTextColor(Color.RED);
			layout.addView(additionalMessageView);
		}

		edtPassword.setOnEditorActionListener(new TextView.OnEditorActionListener() {
			@Override
			public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
				passwordCheck(intent, contentId, ABVNavi, edtPassword.getText().toString());
				return true;
			}
		});
		dialog.setView(layout);

		dialog.setPositiveButton(R.string.ok, new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int which) {
				passwordCheck(intent, contentId, ABVNavi, edtPassword.getText().toString());
			}
		});

		dialog.setNegativeButton(R.string.cancel, null);
		showAlertDialog(dialog);
	}

	private void passwordCheck(final Intent intent, final long contentId, final NaviConsts ABVNavi, String password) {
		if (StringUtil.isNullOrEmpty(password)) {
			showPasswordCheckDialog(intent, contentId, getString(R.string.E002), ABVNavi);
		} else {
			UserAuthenticateLogic userAuthenticateLogic;
			userAuthenticateLogic = AbstractLogic.getLogic(UserAuthenticateLogic.class);
			if (userAuthenticateLogic.checkPassword(password)) {
				contentViewActivityMoveWithAccessLocationAlert(intent, contentId, ABVNavi);
			} else {
				showPasswordCheckDialog(intent, contentId, getString(R.string.E129), ABVNavi);
			}
		}

	}

	@Override
	public void onAuthenticationFailed() {
		handleErrorMessageToast(ErrorCode.S_E_ACMS_1403);
		callLoginActivity(true, false, false);
	}

	@Override
	public void onDownloadedContentDetail(HttpDownloadSimpleNotification notification) {
	}

	@Override
	public void onDownloadingContentZip(final ContentZipDownloadNotification notification) {
		if (notification.getError() != null) {
			runOnUiThread(new Runnable() {
				@Override
				public void run() {
					handleError(notification.getError());
				}
			});
		}
	}

	@Override
	public void onRefreshedContent(boolean result, long contentId, final Exception e) {
		if (e != null) {
			Logger.e(TAG, "onRefreshedContent failed. contentId=" + contentId, e);
		}
		runOnUiThread(new Runnable() {
			@Override
			public void run() {
				if (!contentRefresher.isRefreshing()) {
					configureKeepScreen();
				}
				if (e != null) {
					//新着更新時にログアウトすることで、SIDが取得できなく、例外が発生したときにはトースト非表示
					if (ABVDataCache.getInstance().getMemberInfo() != null) {
						handleError(e);
					}
				}
			}
		});
	}

	/**
	 * メッセージを表示しないコンテンツチェック
	 * 呼び出し元でチェック結果の処理を行う
	 * @param contentId
	 * @return
	 */
	//CustomUrlでダウンロードまたは権限のないコンテンツをダウンロードする前にコンテンツの有効性をチェックするため、(ActivityHandlingHelperで使う)protectedからpublicに変更
	public int contentValidCheck(long contentId) {
		return contentValidCheckAndShowMessage(contentId, 0);
	}

	/**
	 * コンテンツの有効性のチェックを行う
	 * @param contentId
	 * @param showMessageType {@link MessageType}
	 * @return チェック結果 int {@link ContentCheckResultType}
	 */
	protected int contentValidCheckAndShowMessage(long contentId, int showMessageType) {
		int checkStatus;
		try {
			ContentDto dto = contentDao.getContent(contentId);
			if (dto == null) {
				return ContentCheckResultType.ERROR;
			}

			ContractLogic contractLogic = AbstractLogic.getLogic(ContractLogic.class);
			checkStatus = contentLogic.getContentCheckDeliverable(contentId, dto.resourceVersion);
			if (checkStatus == ContentCheckResultType.NO_AUTH_DELETE) {
				// サーバ側でコンテンツがダウンロード出来ない状態(非公開、削除など)の時、該当コンテンツを削除する
				if (contractLogic.isDeleteNoAuthContent()) {
					// 強制削除の場合、権限喪失コンテンツを削除する
					if (contentDao.isExist(contentId)) {
						deleteContent(contentDao.getContent(contentId), true);
					}
				} else {
					// 強制削除ではない場合、
					// 削除されているものはアップデートフラグをfalseにする
					ContentDto contentDto = contentDao.getContent(contentId); // FIXME: なぜここで再度コンテンツを取得している？？
					if (contentDto != null && contentDto.downloadedFlg && contentDto.updatedFlg) {
						contentDto.updatedFlg = false;
						contentDao.updateContent(contentDto, false);
					}
				}
				if (showMessageType == MessageType.TYPE_SHOW_TOAST) {
					handleErrorMessageToast(ErrorCode.M002);
				} else if (showMessageType ==  MessageType.TYPE_SHOW_ALERT) {
					handleErrorMessageDialog(R.string.app_name, ErrorCode.M002);
				}
			} else if (checkStatus == ContentCheckResultType.CHANGED_META_INFO) {
				// メタバージョンの違い
				if (showMessageType ==  MessageType.TYPE_SHOW_TOAST) {
					handleErrorMessageToast(ErrorCode.M005);
				} else if (showMessageType ==  MessageType.TYPE_SHOW_ALERT) {
					handleErrorMessageDialog(R.string.app_name, ErrorCode.M005);
				}
			} else {
				DownloadedContentInfoJSON json = ContentFileUtil.getDownloadedContentInfoJSON(contentId);
				// アンケートと試験コンテンツの場合、回答済みの場合ダウンロードさせない
				if(ContentJSON.KEY_ENQUETE_TYPE.equals(json.contentType) || ContentJSON.KEY_EXAM_TYPE.equals(json.contentType)){
					if (ActivityHandlingHelper.getInstance().isAnsweredEnquete(json)) {
						checkStatus = ContentCheckResultType.ERROR;
					}
				}
			}
		} catch (NetworkDisconnectedException e) {
			Logger.e(TAG, "NetworkDisconnectedException" + e);
			handleErrorMessageToast(ErrorCode.NETWORK);
			checkStatus = ContentCheckResultType.NETWORK_ERROR;
		} catch (ABVException e) {
			Logger.e(TAG, "contentCheckDeliverable ", e);

			if (e.getCode() == ABVExceptionCode.S_E_ACMS_1403) {
				onAuthenticationFailed();
			} else {
				handleErrorMessageToast(ErrorMessage.getErrorCode(e));
			}
			checkStatus = ContentCheckResultType.ERROR;
		} catch (Exception e) {
			Logger.e(TAG, "contentCheckDeliverable failed.", e);
			handleErrorMessageToast(ErrorCode.E107);
			checkStatus = ContentCheckResultType.ERROR;
		}
		return checkStatus;
	}

	protected void deleteContent(ContentDto dto, boolean deletePhysical) {
		contentLogic.deleteContent(dto, deletePhysical);
	}

	/**
	 * ダウンロード処理（wifiチェック/トーストメッセージ表示）
	 * @param contentId
	 * @return
	 */
	protected boolean contentDownload(long contentId) {
		return contentDownload(contentId, true, true);
	}

	/**
	 * ダウンロード処理
	 * @param contentId
	 * @param needCheckWifiFlag
	 * @param isShowToast
	 * @return
	 */
	protected boolean contentDownload(final long contentId, boolean needCheckWifiFlag, final boolean isShowToast) {
		boolean result = true;
		try {
			if (needCheckWifiFlag) {
				// アラートを表示する設定かつ3G回線がアクティブの場合警告を表示
				showWifiDisconnectAlert(R.string.C_E_SYSTEM_0005, new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
						contentDownload(contentId, false, isShowToast);
					}
				}, new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
						ContentZipDownloadNotification notification = new ContentZipDownloadNotification();
						notification.contentId = contentId;
						notification.downloadStatus = DownloadStatusType.Canceled;
						onDownloadingContentZip(notification);
					}
				});
			} else {
				// 空き容量が少ない場合警告を表示
				if (!StorageUtil.isFreeSpaceEnough(this)) {
					Logger.w(TAG, "[storage free space enough]");
					if (isShowToast) {
						handleErrorMessageToast(ErrorCode.STORAGE_WARNING);
					}
				}
				contentDownloader.download(contentId);
			}
		} catch (NetworkDisconnectedException e) {
			Logger.e(TAG, "NetworkDisconnectedException" + e);
			if (isShowToast) {
				handleErrorMessageToast(ErrorCode.NETWORK);
			}
			result = false;
		} catch (ABVException e) {
			Logger.e("ABVException", e.toString());
			result = false;
		} catch (Exception e) {
			Logger.e(TAG, "Exception " + e);
			if (isShowToast) {
				handleErrorMessageToast(ErrorCode.E107);
			}
			result = false;
		}
		return result;
	}



	// Wifi非接続時のアラート表示
	public void showWifiDisconnectAlert(final int messageId, final DialogInterface.OnClickListener positive, final DialogInterface.OnClickListener negative) throws NetworkDisconnectedException {
		showWifiDisconnectAlert(messageId, R.string.download, positive, negative);
	}

	public void showWifiDisconnectAlert(final int messageId, final int titleId, final DialogInterface.OnClickListener positive, final DialogInterface.OnClickListener negative) throws NetworkDisconnectedException {
		if(ABVEnvironment.getInstance().networkAdapter.getActiveNetworkType() == ConnectivityManager.TYPE_MOBILE && PreferenceUtil.get(this, DefPrefKey.WIFI_ALERT_ENABLE, true)){
			handler.post(new Runnable() {
				@Override
				public void run() {
					final ABookAlertDialog alertDialog = AlertDialogUtil.createABookAlertDialog(ABVAuthenticatedActivity.this);
					alertDialog.setTitle(titleId);
					View checkBoxAlert = getLayoutInflater().inflate(R.layout.item_check_box_alert, null);
					alertDialog.setView(checkBoxAlert);
					TextView message = (TextView) checkBoxAlert.findViewById(R.id.message);
					message.setText(getString(messageId));

					final CheckBox checkBox = (CheckBox) checkBoxAlert.findViewById(R.id.check_box);
					alertDialog.setPositiveButton(R.string.yes, new DialogInterface.OnClickListener() {
						@Override
						public void onClick(DialogInterface dialog, int which) {
							PreferenceUtil.put(ABVAuthenticatedActivity.this, DefPrefKey.WIFI_ALERT_ENABLE, !checkBox.isChecked());
							positive.onClick(dialog, which);
						}
					});

					alertDialog.setNegativeButton(R.string.no, new DialogInterface.OnClickListener() {
						@Override
						public void onClick(DialogInterface dialog, int which) {
							negative.onClick(alertDialog, which);
						}
					});

					TextView checkBoxMessage = (TextView) checkBoxAlert.findViewById(R.id.check_box_message);
					checkBoxMessage.setText(getString(R.string.not_show_nexttime));
					checkBoxMessage.setOnClickListener(new View.OnClickListener() {
						@Override
						public void onClick(View v) {
							checkBox.setChecked(!checkBox.isChecked());
						}
					});
					showAlertDialog(alertDialog);
				}
			});
		} else {
			positive.onClick(null, -1);
		}
	}

	public void startContentViewActivity(long contentId) {
		startContentViewActivity(contentId, 0);
	}

	public void startContentViewActivity(long contentId, int pageNum) {
		Intent intent = new Intent();
		//		intent.putExtra("className", getClass().getName());
		intent.putExtra("page", pageNum);
		//		intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
		startContentViewActivity(intent, contentId);
	}

	protected void setDisplaySize() {
		mDisplaySize = DisplayUtil.getDisplaySize2(getApplicationContext());

		Logger.v(TAG, "[Size] display width=%s, height=%s", mDisplaySize.width, mDisplaySize.height);
	}

	private void handleError(final Exception e) {
		ErrorCode error;
		if (e instanceof AcmsException && ((AcmsException)e).getCode() == ABVExceptionCode.S_E_ACMS_1403) { //	再ログインが必要
			onAuthenticationFailed();
		} else {
			error = ErrorCode.L120;
			ErrorMessage.showErrorMessageToast(ABVAuthenticatedActivity.this, error);
		}
	}

	public void showUnAuthorizedContentWarningDialog(String msg) {
		ABookAlertDialog alertDialog = AlertDialogUtil.createAlertDialog(this, getString(R.string.app_name), msg);
		alertDialog.setPositiveButton(R.string.ok, null);
		alertDialog.show();
	}

	/**
	 * ヘルプ画面を表示
	 * @param helpViewType ヘルプ表示種類
	 */
	public void showHelpViewDialog(int helpViewType) {
		Intent intent = new Intent();
		String className = HelpActivity.class.getName();
		if (!isNormalSize()) {
			className += "Dialog";
		}
		intent.putExtra(ABookKeys.HELP_VIEW_TYPE, helpViewType);
		intent.setClassName(getApplicationContext().getPackageName(), className);
		startActivity(intent);
	}

	/**
	 * 一括同期処理中であるか確認
	 * @return
	 */
	public boolean isShowingBatchSync() {
		return batchSyncView != null && batchSyncView.isShowing();
	}

	/**
	 * @version 1.2.300
	 * 新着情報更新処理を中止
	 */
	public void stopContentRefresher() {
		contentRefresher.stopRefresh();
	}

	/**
	 * @version 1.2.300
	 * ダブルタップ制御用
	 * @param button ダブルタップ防止ボタン
	 */
	protected void buttonDoubleTapControl(Button button) {
		final Button finalButton = button;
		finalButton.setEnabled(false);
		handler.postDelayed(new Runnable() {
			@Override
			public void run() {
				finalButton.setEnabled(true);
			}
		}, BUTTON_ENABLE_DELAY_MILLIS);
	}

	/**
	 * コミュニケーションメニューダイアログ
	 */
	public void showCommunicationMenuDialog() {
		mCommunicationMenuDialog = new Dialog(this);
		mCommunicationMenuDialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
		mCommunicationMenuDialog.setCanceledOnTouchOutside(false);
		mCommunicationMenuDialog.setContentView(R.layout.communication_menu_dialog);

		List<PushMessageDto> pushMessageDtoList = pushMessageLogic.getAllPushMessageList();

		boolean existUnreadFlg = false;
		for (PushMessageDto pushMessageDto : pushMessageDtoList) {
			if (!pushMessageDto.readingFlg) {
				existUnreadFlg = true;
				break;
			}
		}

		ImageView ivIcon = (ImageView)mCommunicationMenuDialog.findViewById(R.id.btn_push_message_list);

		if (existUnreadFlg) {
			ivIcon.setBackgroundResource(R.drawable.ic_communication_push_message_list_with_badge);
		} else {
			ivIcon.setBackgroundResource(R.drawable.ic_communication_push_message_list);
		}

		LinearLayout ll_menuItem_meeting = (LinearLayout)mCommunicationMenuDialog.findViewById(R.id.ll_item_meeting);
		ll_menuItem_meeting.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View view) {
				ActivityHandlingHelper.getInstance().startMeetingActivity();
				mCommunicationMenuDialog.dismiss();
			}
		});

		LinearLayout ll_menuItem_push_message = (LinearLayout)mCommunicationMenuDialog.findViewById(R.id.ll_item_push_message);
		ll_menuItem_push_message.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View view) {
				showOperationSelectDialog();
				mCommunicationMenuDialog.dismiss();
			}
		});

		LinearLayout ll_menuItem_push_message_list = (LinearLayout)mCommunicationMenuDialog.findViewById(R.id.ll_item_push_message_list);
		ll_menuItem_push_message_list.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View view) {
				showPushMessageListDialog();
				mCommunicationMenuDialog.dismiss();
			}
		});

		// chat
		LinearLayout ll_menuItem_chat = (LinearLayout)mCommunicationMenuDialog.findViewById(R.id.ll_item_chat);
		if (ABVDataCache.getInstance().serviceOption.isChat()) {
			ll_menuItem_chat.setOnClickListener(new View.OnClickListener() {
				@Override
				public void onClick(View view) {
					ActivityHandlingHelper.getInstance().startChatWebViewActivity();
					mCommunicationMenuDialog.dismiss();
				}
			});
		} else {
			ll_menuItem_chat.setVisibility(View.GONE);
		}

		mCommunicationMenuDialog.findViewById(R.id.close_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				mCommunicationMenuDialog.dismiss();
			}
		});

		mCommunicationMenuDialog.show();
	}

	private void showOperationSelectDialog() {
		OperationDao operationDao = AbstractDao.getDao(OperationDao.class);
		mOperationSelectDialog = new Dialog(this);
		mOperationSelectDialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
		mOperationSelectDialog.setCanceledOnTouchOutside(false);
		mOperationSelectDialog.setContentView(R.layout.operation_select_dialog);
		mOperationSelectListView = (ListView) mOperationSelectDialog.findViewById(R.id.lv_operation_select);
		List<OperationDto> operationDtoList = operationDao.getAllOperation();
		mOperationSelectAdapter = new OperationSelectAdapter(this, operationDtoList);
		mOperationSelectListView.setAdapter(mOperationSelectAdapter);

		mOperationSelectListView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
			@Override
			public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
				OperationDto operationDto = mOperationSelectAdapter.getItem(position);
				showPushMessageSendDialog(operationDto);
				mOperationSelectDialog.dismiss();
			}
		});

		// リソースパターンを適用
		TextView operationSelect = (TextView) mOperationSelectDialog.findViewById(R.id.tv_toolbar_title);
		operationSelect.setText(PatternStringUtil.patternToInt(getApplicationContext(),
				R.string.operation_select,
				getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)));

		mOperationSelectDialog.findViewById(R.id.close_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				mOperationSelectDialog.dismiss();
			}
		});

		mOperationSelectDialog.show();
	}

	/**
	 * プッシュメッセージダイアログ表示
	 */
	private void showPushMessageListDialog() {
		mPushMessageListDialog = new Dialog(this);
		mPushMessageListDialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
		mPushMessageListDialog.setCanceledOnTouchOutside(false);
		mPushMessageListDialog.setContentView(R.layout.push_message_list_dialog);
		mPushMessageListView = (ListView) mPushMessageListDialog.findViewById(R.id.lv_push_message);

		final List<PushMessageDto> pushMessageDtoList = pushMessageLogic.getAllPushMessageList();
		mPushMessageListAdapter = new PushMessageListAdapter(this, pushMessageDtoList);
		mPushMessageListView.setAdapter(mPushMessageListAdapter);

		mPushMessageListAdapter.setAdapterListener(new PushMessageListAdapter.PushMessageListAdapterListener() {
			@Override
			public void onDetailView(final PushMessageDto dto) {

				if(dto.roomName != null && dto.roomId != 0)
				{
					pushMessageLogic.updateReadingFlg(dto.pushMessageId);
					ActivityHandlingHelper.getInstance().startChatWebViewActivityWithPushMessage(dto);
					mPushMessageListDialog.dismiss();
				} else {
					showPushMessageDetailView(dto);
				}
				// プッシュメッセージで未読がなければ、下辺ツールバーのコミュニケーションボタンのバッジを外す
				List<PushMessageDto> checkPushMessageList = mPushMessageListAdapter.getItems();
				if (!checkUnReadCommunication(checkPushMessageList)) {
					communicationButton.setImageResource(R.drawable.ic_communication_menu);
				}
			}
		});
		mPushMessageListView.invalidate();
		mPushMessageListView.setClickable(false);

		mPushMessageListDialog.findViewById(R.id.back_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				List<PushMessageDto> pushMessageDtoList = pushMessageLogic.getAllPushMessageList();
				mPushMessageListAdapter.setItem(pushMessageDtoList);
				showPushMessgeListView(true);
			}
		});

		mPushMessageListDialog.findViewById(R.id.close_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				mPushMessageListDialog.dismiss();
			}
		});

		showPushMessgeListView(true);
		mPushMessageListDialog.show();
	}

	// プッシュメッセージの詳細表示
	private void showPushMessageDetailView(final PushMessageDto dto) {
		dto.readingFlg = true;

		TextView messageTextView = (TextView)mPushMessageListDialog.findViewById(R.id.tv_message);
		TextView sendDateTextView = (TextView)mPushMessageListDialog.findViewById(R.id.tv_send_date);
		TextView sendOperationNameTextView = (TextView)mPushMessageListDialog.findViewById(R.id.tv_operation_name);
		TextView sendUserTextView = (TextView)mPushMessageListDialog.findViewById(R.id.tv_send_user);
		messageTextView.setText(dto.pushMessage);
		sendDateTextView.setText(DateTimeUtil.toString(dto.pushSendDate, DateTimeFormat.yyyyMMddHHmm_slash));
		sendOperationNameTextView.setText(dto.operationName);
		sendUserTextView.setText(dto.pushSendLoginId);

		pushMessageLogic.updateReadingFlg(dto.pushMessageId);
		showPushMessgeListView(false);
	}

	// プッシュメッセージの一覧表示
	private void showPushMessgeListView(boolean pushMessageListViewFlg) {
		TextView titleTextView = (TextView)mPushMessageListDialog.findViewById(R.id.tv_toolbar_title);
		if (pushMessageListViewFlg) {
			mPushMessageListView.setVisibility(View.VISIBLE);
			mPushMessageListDialog.findViewById(R.id.close_btn).setVisibility(View.VISIBLE);
			mPushMessageListDialog.findViewById(R.id.back_btn).setVisibility(View.GONE);
			mPushMessageListDialog.findViewById(R.id.ll_detail).setVisibility(View.GONE);
			// リソースパターンの適用
			titleTextView.setText(PatternStringUtil.patternToString(getApplicationContext(),
					R.string.push_message_list,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)));
		} else {
			mPushMessageListView.setVisibility(View.GONE);
			mPushMessageListDialog.findViewById(R.id.close_btn).setVisibility(View.VISIBLE);
			mPushMessageListDialog.findViewById(R.id.back_btn).setVisibility(View.VISIBLE);
			mPushMessageListDialog.findViewById(R.id.ll_detail).setVisibility(View.VISIBLE);
			// リソースパターンの適用
			titleTextView.setText(PatternStringUtil.patternToString(getApplicationContext(),
					R.string.title_message_detail,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)));
		}
	}

	/**
	 * プッシュメッセージ送信ダイヤログ表示
	 */
	private void showPushMessageSendDialog(final OperationDto operationDto) {
		mPushMessageSendDialog = new Dialog(this);
		mPushMessageSendDialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
		mPushMessageSendDialog.setCanceledOnTouchOutside(false);
		mPushMessageSendDialog.setContentView(R.layout.push_message_send_dialog);
		mFixPushMessageListView = (ListView) mPushMessageSendDialog.findViewById(R.id.listView);


		mPushMessageSendDialog.findViewById(R.id.back_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				showOperationSelectDialog();
				mPushMessageSendDialog.dismiss();
			}
		});

		mPushMessageSendDialog.findViewById(R.id.close_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				mPushMessageSendDialog.dismiss();
			}
		});

		mPushMessageSendDialog.findViewById(R.id.send_btn).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				sendPushMessageRequest(operationDto.operationId);
			}
		});

		RadioGroup searchTarget = (RadioGroup) mPushMessageSendDialog.findViewById(R.id.send_target_group);
		searchTarget.setOnCheckedChangeListener(new RadioGroup.OnCheckedChangeListener() {
			@Override
			public void onCheckedChanged(RadioGroup group, int checkedId) {
				if (checkedId == R.id.in_gourp) {
					mSendType = Constant.PushMessageSendType.InGroup;
				} else {
					mSendType = Constant.PushMessageSendType.AllOperation;
				}
			}
		});

		mPushMessageSendDialog.findViewById(R.id.fix_push_message_layout).setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				showFixPushMessageView();
			}
		});

		TextView messageTitleTextView = (TextView) mPushMessageSendDialog.findViewById(R.id.message_title);
		// リソースパターンの適用
		messageTitleTextView.setText(PatternStringUtil.patternToString(getApplicationContext(),
				R.string.free_input,
				getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)) + " > ");

		TextView tvOperationName = (TextView) mPushMessageSendDialog.findViewById(R.id.operation_name);
		tvOperationName.setText(operationDto.operationName);

		mPushMessageSendDialog.show();
	}

	/**
	 * プッシュメッセージ送信画面、定型文一覧画面切り替え
	 * @param sendPushMessageViewFlg (true : プッシュメッセージ送信画面、false : 定型文一覧画面)
	 */
	private void showSendPushMessgeView(boolean sendPushMessageViewFlg) {
		if (sendPushMessageViewFlg) {
			mFixPushMessageListView.setVisibility(View.GONE);
			mPushMessageSendDialog.findViewById(R.id.back_btn).setVisibility(View.GONE);

			mPushMessageSendDialog.findViewById(R.id.push_send_layout).setVisibility(View.VISIBLE);
			mPushMessageSendDialog.findViewById(R.id.close_btn).setVisibility(View.VISIBLE);
			EditText messageEditText = (EditText) mPushMessageSendDialog.findViewById(R.id.message);
			KeyboardUtils.show(this, messageEditText);
		} else {
			mFixPushMessageListView.setVisibility(View.VISIBLE);
			mPushMessageSendDialog.findViewById(R.id.back_btn).setVisibility(View.VISIBLE);

			mPushMessageSendDialog.findViewById(R.id.push_send_layout).setVisibility(View.GONE);
			mPushMessageSendDialog.findViewById(R.id.close_btn).setVisibility(View.GONE);
		}
	}

	/**
	 * プッシュメッセージ送信依頼
	 */
	private void sendPushMessageRequest(long operationId) {
		if (!ABVEnvironment.getInstance().networkAdapter.isNetworkConnected()) {
			ABVToastUtil.showMakeText(this, R.string.request_network_connection, Toast.LENGTH_SHORT);
			return;
		}
		//文字列チェック
		EditText messageEditText = (EditText) mPushMessageSendDialog.findViewById(R.id.message);
		String message = messageEditText.getText().toString();

		String checkMessage = message.replace(" ", "");
		if (checkMessage.length() == 0) {
			// リソースパターンの適用
			ABVToastUtil.showMakeText(this, PatternStringUtil.patternToString(getApplicationContext(),
					R.string.push_message_input_null,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)), Toast.LENGTH_SHORT);
			return;
		}
		byte[] messageByte = message.getBytes(StandardCharsets.UTF_8);
		if (messageByte.length >= 207) {
			// リソースパターンの適用
			ABVToastUtil.showMakeText(this, PatternStringUtil.patternToString(getApplicationContext(),
					R.string.push_message_input_over,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)), Toast.LENGTH_SHORT);
			return;
		}

		// リソースパターンの適用
		ProgressDialogHelper.showProgressPopup(this, true, PatternStringUtil.patternToString(getApplicationContext(),
				R.string.file_initialization,
				getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)));
		try {
			boolean result = pushMessageLogic.sendPushMessageRequest(mSendType, message, operationId);
			if (result) {
				mPushMessageSendDialog.dismiss();
				// リソースパターンの適用
				ABVToastUtil.showMakeText(this, PatternStringUtil.patternToString(getApplicationContext(),
						R.string.push_message_send_success,
						getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)), Toast.LENGTH_SHORT);
			} else {
				// リソースパターンの適用
				ABVToastUtil.showMakeText(this, PatternStringUtil.patternToString(getApplicationContext(),
						R.string.push_message_send_fail,
						getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)), Toast.LENGTH_SHORT);
			}
		} catch (Exception e) {
			Logger.e(TAG, e);
			// リソースパターンの適用
			ABVToastUtil.showMakeText(this, PatternStringUtil.patternToString(getApplicationContext(),
					R.string.push_message_send_fail,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)), Toast.LENGTH_SHORT);
		} finally {
			ProgressDialogHelper.closeProgressPopup();
		}
	}

	private void showFixPushMessageView() {
		if (!ABVEnvironment.getInstance().networkAdapter.isNetworkConnected()) {
			ABVToastUtil.showMakeText(this, R.string.request_network_connection, Toast.LENGTH_SHORT);
			return;
		}
		EditText messageEditText = (EditText) mPushMessageSendDialog.findViewById(R.id.message);
		KeyboardUtils.hide(this, messageEditText);

		// リソースパターンの適用
		ProgressDialogHelper.showProgressPopup(this, true, PatternStringUtil.patternToString(getApplicationContext(),
				R.string.file_initialization,
				getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)));
		try {
			List<FixPushMessageDto> fixPushMessageList = pushMessageLogic.getFixPushMessage();
			FixPushMessageDto dto = new FixPushMessageDto();
			// リソースパターンの適用
			dto.name = PatternStringUtil.patternToString(getApplicationContext(),
					R.string.free_input,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0));
			fixPushMessageList.add(0, dto);
			mFixPushMessageListView.setAdapter(new FixPushMessageAdapter(this, fixPushMessageList, mSelectedFixPuchMessagePosition));
			mFixPushMessageListView.invalidate();
			mFixPushMessageListView.setClickable(true);
			mFixPushMessageListView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
				@Override
				public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
					mSelectedFixPuchMessagePosition = position;
					FixPushMessageDto item = (FixPushMessageDto)parent.getItemAtPosition(position);

					TextView messageTitleTextView = (TextView) mPushMessageSendDialog.findViewById(R.id.message_title);
					messageTitleTextView.setText(item.name + " > ");

					EditText messageEditText = (EditText) mPushMessageSendDialog.findViewById(R.id.message);
					messageEditText.setText(item.value);
					messageEditText.setSelection(messageEditText.getText().length());

					showSendPushMessgeView(true);
				}
			});
			showSendPushMessgeView(false);
		} catch (Exception e) {
			Logger.e(TAG, e);
			// リソースパターンの適用
			ABVToastUtil.showMakeText(this, PatternStringUtil.patternToString(getApplicationContext(),
					R.string.respons_fix_push_message_fail,
					getUserPref(AppDefType.UserPrefKey.RESOURCE_PATTERN_TYPE, 0)), Toast.LENGTH_SHORT);
		} finally {
			ProgressDialogHelper.closeProgressPopup();
		}
	}

	// コミュニケーションのアイコン設定（未既読があれば、バッチ付きアイコンでセット）
	protected void setCommunicationImageButton() {
		List<PushMessageDto> pushMessageDtoList = pushMessageLogic.getAllPushMessageList();
		boolean existUnreadFlg = checkUnReadCommunication(pushMessageDtoList);
		communicationButton.setImageResource(existUnreadFlg ? R.drawable.ic_communication_menu_with_badge : R.drawable.ic_communication_menu);
	}

	/**
	 * 未読のプッシュメッセージが存在するかチェック
	 * @param pushMessageDtoList チェックするリスト
	 * @return
	 */
	private boolean checkUnReadCommunication(List<PushMessageDto> pushMessageDtoList) {
		boolean existUnreadFlg = false;
		for (PushMessageDto pushMessageDto : pushMessageDtoList) {
			if (!pushMessageDto.readingFlg) {
				existUnreadFlg = true;
				break;
			}
		}
		return existUnreadFlg;
	}

}
