package jp.agentec.abook.abv.bl.data.dao;

import java.util.Date;
import java.util.List;

import jp.agentec.abook.abv.bl.common.db.Cursor;
import jp.agentec.abook.abv.bl.dto.TaskReportDto;
import jp.agentec.adf.util.DateTimeFormat;
import jp.agentec.adf.util.DateTimeUtil;

/**
 * Created by leej on 2018/09/10.
 */

public class TaskReportDao extends AbstractDao {

    private static final String TAG = "TaskReportDao";

    /*package*/ TaskReportDao() {
    }

    @Override
    protected TaskReportDto convert(Cursor cursor) {
        TaskReportDto dto = new TaskReportDto();

        int column = cursor.getColumnIndex("task_key");
        if (column != -1) {
            dto.taskKey = cursor.getString(column);
        }
        column = cursor.getColumnIndex("json_data");
        if (column != -1) {
            dto.jsonData = cursor.getString(column);
        }
        column = cursor.getColumnIndex("attached_file_name");
        if (column != -1) {
            dto.attachedFileName = cursor.getString(column);
        }
        column = cursor.getColumnIndex("local_attached_file_name");
        if (column != -1) {
            dto.localAttachedFileName = cursor.getString(column);
        }
        column = cursor.getColumnIndex("attached_file_send_flg");
        if (column != -1) {
            dto.attachedFileSendFlg = toBool(cursor.getInt(column));
        }
        column = cursor.getColumnIndex("data_send_flg");
        if (column != -1) {
            dto.dataSendFlg = toBool(cursor.getInt(column));
        }
        // t_taskテーブルへINNER JOINで取得したカラム取得
        column = cursor.getColumnIndex("task_hotspot_info");
        if (column != -1) {
            dto.taskHotSpotInfo = cursor.getString(column);
        }
        column = cursor.getColumnIndex("del_flg");
        if (column != -1) {
            dto.delFlg = toBool(cursor.getInt(column));
        }

        column = cursor.getColumnIndex("enable_report");
        if (column != -1) {
            dto.enableReport = cursor.getInt(column);
        }

        column = cursor.getColumnIndex("task_report_level");
        if (column != -1) {
            dto.taskReportLevel = cursor.getInt(column);
        }

        column = cursor.getColumnIndex("task_report_id");
        if (column != -1) {
            dto.taskReportId = cursor.getInt(column);
        }

        column = cursor.getColumnIndex("task_report_info_id");
        if (column != -1) {
            dto.taskReportInfoId = cursor.getInt(column);
        }

        column = cursor.getColumnIndex("report_start_date");
        if (column != -1) {
            dto.reportStartDate = DateTimeUtil.toDate(cursor.getString(column), "UTC", DateTimeFormat.yyyyMMddHHmmss_hyphen);
        }

        column = cursor.getColumnIndex("report_end_date");
        if (column != -1) {
            dto.reportEndDate = DateTimeUtil.toDate(cursor.getString(column), "UTC", DateTimeFormat.yyyyMMddHHmmss_hyphen);
        }

        column = cursor.getColumnIndex("local_saved_flg");
        if (column != -1) {
            dto.localSavedFlg = toBool(cursor.getInt(column));
        }

        return dto;
    }

    /**
     * 報告登録処理
     * @param dto
     */
    public void insert(TaskReportDto dto) {
        insert("insert into t_task_report "
                        + "(task_key, "
                        + "json_data, "
                        + "attached_file_name, "
                        + "local_attached_file_name, "
                        + "attached_file_send_flg, "
                        + "data_send_flg, "
                        + "task_report_id, "
                        + "task_report_info_id, "
                        + "report_start_date, "
                        + "report_end_date, "
                        + "enable_report, "
                        + "task_report_level, "
                        + "local_saved_flg ) "
                        + "values "
                        + "(?,?,?,?,?,?,?,?,?,?,?,?,?)",
                dto.getInsertValues());
    }

    /**
     * 報告更新処理
     * @param dto
     * @return
     */
    public boolean update(TaskReportDto dto) {
        Object[] objects;
        StringBuffer sql = new StringBuffer();
        sql.append("UPDATE t_task_report SET ");
        sql.append("json_data=?, ");
        sql.append("attached_file_name=?, ");
        sql.append("local_attached_file_name=?, ");
        sql.append("attached_file_send_flg=?,  ");
        sql.append("data_send_flg=?,  ");
        sql.append("report_end_date=?, ");
        sql.append("enable_report=?, ");
        sql.append("task_report_level=?, ");
        sql.append("local_saved_flg=?, ");
        sql.append("task_report_info_id=? ");
        sql.append("WHERE task_key=? AND task_report_level=?");
        if (dto.reportStartDate != null) {
            sql.append(" AND datetime(report_start_date)=datetime(?)");
            objects = new Object[]{dto.jsonData, dto.attachedFileName, dto.localAttachedFileName, dto.attachedFileSendFlg, dto.dataSendFlg,
                    dto.reportEndDate, dto.enableReport, dto.taskReportLevel, dto.localSavedFlg, dto.taskReportInfoId, dto.taskKey, dto.taskReportLevel, dto.reportStartDate};
        } else {
            objects = new Object[]{dto.jsonData, dto.attachedFileName, dto.localAttachedFileName, dto.attachedFileSendFlg, dto.dataSendFlg,
                    dto.reportEndDate, dto.enableReport, dto.taskReportLevel, dto.localSavedFlg, dto.taskReportInfoId, dto.taskKey, dto.taskReportLevel};
        }

        return update(sql.toString(), objects) > 0;
    }

    /**
     * 作業IDで報告取得
     * （作業報告タイプ　昇順、作業報告ID 昇順、報告開始日　昇順）
     * @param operationId
     * @return
     */
    public List<TaskReportDto> getTaskReportByOperationId(Long operationId) {
        StringBuilder sb = new StringBuilder();
        sb.append(" select ttr.* ");
        sb.append(" from t_task_report ttr ");
        sb.append(" inner join ");
        sb.append("     t_task tt ");
        sb.append("         ON tt.task_key == ttr.task_key ");
        sb.append("         WHERE tt.del_flg = 0");
        sb.append("         AND tt.operation_id = ?");
        sb.append("         ORDER BY task_report_level ASC, task_report_id ASC, report_start_date ASC");
        return rawQueryGetDtoList(sb.toString(), new String[]{"" + operationId}, TaskReportDto.class);
    }

    /**
     * 作業キーと作業報告タイプで作業報告データ取得
     * @param taskKey
     * @param taskReportLevel
     * @return
     */
    public TaskReportDto getTaskReport(String taskKey, Integer taskReportLevel) {
        return rawQueryGetDto("select * from t_task_report where task_key=? AND task_report_level=?", new String[]{ "" + taskKey, "" + taskReportLevel }, TaskReportDto.class);
    }

    /**
     * 作業キーで報告リストを取得
     * @param taskKey
     * @return
     */
    public List<TaskReportDto> getTaskReportListByTaskKey(String taskKey) {
        return rawQueryGetDtoList("select * from t_task_report where task_key=? ORDER BY task_report_id, report_start_date", new String[]{ "" + taskKey }, TaskReportDto.class);
    }

    /**
     * 送信フラグのあるデータが存在するか
     *
     * @param operationId
     * @return
     */
    public boolean isExistSendTaskData(Long operationId) {
        return getSendableTaskData(operationId).size() > 0;
    }

    public List<TaskReportDto> getSendableTaskData(Long operationId) {
        StringBuilder sb = new StringBuilder();
        sb.append(" select * ");
        sb.append(" from t_task_report ttr ");
        sb.append(" inner join ");
        sb.append("     t_task tt ");
        sb.append("         ON tt.task_key == ttr.task_key ");
        sb.append("         WHERE ttr.data_send_flg = 1");
        sb.append("         AND tt.operation_id = ?");
        return rawQueryGetDtoList(sb.toString(), new String[]{"" + operationId}, TaskReportDto.class);
    }

    /**
     * ホットスポットが更新フラグがtrueのデータ取得
     * @param operationId
     * @param taskKey
     * @return
     */
    public List<TaskReportDto> getUpdateTargetHotSpotTaskData(Long operationId, String taskKey) {
        StringBuilder sb = new StringBuilder();
        sb.append(" select * ");
        sb.append(" from t_task_report ttr ");
        sb.append(" inner join ");
        sb.append("     t_task tt ");
        sb.append("         ON tt.task_key == ttr.task_key ");
        sb.append("         WHERE tt.hotspot_changed_flg = 1");
        sb.append("         AND tt.operation_id = ?");
        sb.append("         AND ttr.task_report_level = 0");
        if (taskKey != null) {
            sb.append("         AND ttr.task_key = ?");
        }
        return rawQueryGetDtoList(sb.toString(), new String[]{"" + operationId}, TaskReportDto.class);
    }

    /**
     * ホットスポットが更新フラグがtrueのデータ存在チェック
     * @param operationId
     * @return
     */
    public boolean isExistUpdateTargetHotSpotTaskData(Long operationId) {
        return getUpdateTargetHotSpotTaskData(operationId, null).size() > 0;
    }


    /**
     * 送信フラグがtrueのデータを取得
     * @param operationId
     * @param taskKey
     * @param taskReportLevel
     * @return
     */
    public List<TaskReportDto> getSendableTaskData(Long operationId, String taskKey, int taskReportLevel) {
        StringBuilder sb = new StringBuilder();
        sb.append(" select * ");
        sb.append(" from t_task_report ttr ");
        sb.append(" inner join ");
        sb.append("     t_task tt ");
        sb.append("         ON tt.task_key == ttr.task_key ");
        sb.append("         WHERE ttr.data_send_flg = 1");
        sb.append("         AND tt.operation_id = ?");
        sb.append("         AND tt.task_key = ?");
        sb.append("         AND ttr.task_report_level = ?");
        return rawQueryGetDtoList(sb.toString(), new String[]{"" + operationId, "" + taskKey, "" + taskReportLevel}, TaskReportDto.class);
    }

    /**
     *
     * taskReportLevel によって添付ファイル名を取得する
     * @param taskKey
     * @param taskReportLevel
     * @return
     */
    public String getTaskReportAttachedFileName(String taskKey, int taskReportLevel) {
        return rawQueryGetString("select local_attached_file_name from t_task_report where task_key=? AND task_report_level=?", new String[]{"" + taskKey, "" + taskReportLevel});
    }

    /**
     * 報告キー、報告担当レベルでt_task_Reportデータ取得
     * @param taskKey
     * @param taskReportLevel
     * @return
     */
    public TaskReportDto selectByTaskKey(String taskKey, int taskReportLevel) {
        return rawQueryGetDto("select * from t_task_report where task_key=? and task_report_level=?", new String[]{ "" + taskKey, "" + taskReportLevel }, TaskReportDto.class);
    }

    public List<TaskReportDto> selectAll() {
        return rawQueryGetDtoList("select * from t_task_report", null, TaskReportDto.class);
    }

    /**
     * データの存在チェック
     * @param taskKey
     * @param taskReportLevel
     * @return
     */
    public boolean isExistTaskReport(String taskKey, Integer taskReportLevel) {
        return rawQueryGetInt("select COUNT(*) from t_task_report where task_key=? AND task_report_level=?", new String[]{"" + taskKey, "" + taskReportLevel }) > 0;
    }

    /**
     * テーブル物理削除
     * @param dto
     */
    public void delete(TaskReportDto dto) {
        delete("t_task_report", "task_key=? and task_report_level=?", new String[] { dto.taskKey, "" + dto.taskReportLevel });
    }

    /**
     * テーブル物理削除（定期点検専用）
     * @param dto
     */
    public void deleteRoutineTaskReport(TaskReportDto dto) {
        delete("t_task_report", "task_key=? AND task_report_level=? AND task_report_id=? AND datetime(report_start_date)=datetime(?, 'utc')", new String[] { dto.taskKey, "" + dto.taskReportLevel, "" + dto.taskReportId, DateTimeUtil.toString(dto.reportStartDate, DateTimeFormat.yyyyMMddHHmmss_hyphen) });
    }

    /**
     * 定期点検報告データで最後の時間データ取得
     * @param operationId
     * @return
     */
    public TaskReportDto getLastRoutineTaskReportData(long operationId) {
        StringBuilder sb = new StringBuilder();
        sb.append(" select tr.* ");
        sb.append(" from t_task_report tr ");
        sb.append(" inner join ");
        sb.append("     t_task tt ");
        sb.append("         ON tr.task_key == tt.task_key");
        sb.append("     WHERE tt.operation_id = ? ORDER BY tr.report_start_date DESC LIMIT 1");
        List<TaskReportDto> dtoList = rawQueryGetDtoList(sb.toString(), new String[]{ "" + operationId }, TaskReportDto.class);

        if (dtoList != null && dtoList.size() > 0) {
            return dtoList.get(0);
        } else {
            TaskReportDto dto = new TaskReportDto();
            dto.taskReportId = 0;
            return dto;
        }
    }

    /**
     * 定期点検の報告データ取得（時間はutcで検索）
     * @param taskKey
     * @param taskReportId
     * @param reportStartDate
     * @return
     */
    public TaskReportDto getRoutineTaskReportUtc(String taskKey, int taskReportId, String reportStartDate){
        return rawQueryGetDto("select * from t_task_report where task_key=? and task_report_id=? and datetime(report_start_date)=datetime(?, 'utc')", new String[]{ taskKey, "" + taskReportId, reportStartDate }, TaskReportDto.class);
    }

    /**
     * 定期点検のSQL専用のため、taskReportId, report_start_dateが存在するデータの条件追加
     * @param operationId
     * @return
     */
    public boolean existsToDoRoutineTaskReportData(long operationId) {
        String currDate = DateTimeUtil.toString(DateTimeUtil.getCurrentSqlDate(), DateTimeFormat.yyyyMMddHHmmss_hyphen);
        StringBuilder sb = new StringBuilder();
        sb.append(" select count(*) ");
        sb.append(" from t_task_report tr ");
        sb.append("     inner join t_task tt ON tt.task_key == tr.task_key ");
        sb.append("     WHERE tt.del_flg = 0 ");
        sb.append("         AND tt.operation_id = ? ");
        // 定期点検のみ
        sb.append("         AND tr.task_report_id != 0");
        sb.append("         AND tr.report_start_date IS NOT NULL");
        sb.append("         AND datetime(report_end_date) >= datetime(?, 'utc') ");
        return rawQueryGetInt(sb.toString(), new String[]{ "" + operationId, currDate }) > 0;
    }

    /**
     * 定期点検報告の添付zipファイル名を取得
     * @param taskKey
     * @param taskReportId
     * @param reportStartDate
     * @return
     */
    public String getRoutineTaskReportAttachedFileName(String taskKey, int taskReportId, String reportStartDate) {
        return rawQueryGetString("select local_attached_file_name from t_task_report where task_key=? and task_report_id=? and datetime(report_start_date)=datetime(?, 'utc')", new String[]{ taskKey, "" + taskReportId, reportStartDate });
    }
}
