package jp.agentec.abook.abv.bl.logic;

import java.io.IOException;
import java.util.List;

import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.common.util.ContentFileUtil;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.ContentBookmarkDao;
import jp.agentec.abook.abv.bl.data.dao.ContentDao;
import jp.agentec.abook.abv.bl.data.dao.ContentPageDao;
import jp.agentec.abook.abv.bl.dto.ContentBookmarkDto;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.adf.util.DateTimeUtil;

public class BookmarkLogic extends AbstractLogic {
	private static final String TAG = "BookmarkLogic";
	private ContentBookmarkDao contentBookmarkDao = AbstractDao.getDao(ContentBookmarkDao.class);
	private ContentDao contentDao = AbstractDao.getDao(ContentDao.class);
	private ContentPageDao contentPageDao = AbstractDao.getDao(ContentPageDao.class);

	/**
	 * しおり情報を返します。
	 * @return　しおり情報のリストを返します。
	 * @throws IOException 
	 * @throws Exception その他、例外です。
	 * @since 1.0.0
	 */
	public List<ContentBookmarkDto> getContentBookmark() throws IOException {
		List<ContentBookmarkDto> list = contentBookmarkDao.getAllBookmarkList();
		
		if (list != null && list .size() > 0) {			
			for (ContentBookmarkDto contentBookmarkDto : list) {
				contentBookmarkDto.pageThumbnailPath = ContentFileUtil.getPdfThumbnailPath(contentBookmarkDto.contentId, contentBookmarkDto.pageNum, ContentFileUtil.SIZE_S);
			}
		}
		
		return list;
	}

	/**
	 * しおり情報を返します。
	 * @return　しおり情報のリストを返します。
	 * @param contentId コンテンツIDです。
	 * @throws Exception その他、例外です。
	 * @since 1.0.0
	 */
	public List<ContentBookmarkDto> getContentBookmark(long contentId) {
		List<ContentBookmarkDto> list = contentBookmarkDao.getBookmarkList(contentId);
		if (list != null && list .size() > 0) {
			for (ContentBookmarkDto contentBookmarkDto : list) {
				contentBookmarkDto.pageThumbnailPath = ContentFileUtil.getPdfThumbnailPath(contentBookmarkDto.contentId, contentBookmarkDto.pageNum, ContentFileUtil.SIZE_S);
			}
		}
		return list;
	}
	
	/**
	 * しおり情報の存在を返します。
	 * @return　しおり情報の存在を返します。
	 * @param contentId コンテンツIDです。
	 * @param pageNum   ページナンバーです。
	 * @throws Exception その他、例外です。
	 * @since 1.0.0
	 */
	public boolean isExsitContentBookmark(long contentId,int pageNum) {
		return contentBookmarkDao.isBookmark(contentId, pageNum);
	}
	
	public boolean isBookmarkExist() {
		boolean exist = false;
		try{
			List<ContentBookmarkDto> list = AbstractDao.getDao(ContentBookmarkDao.class).getAllBookmarkList();
			if (list !=null && list.size() > 0) {
				exist = true;
			}
		} catch(Exception e) {
			Logger.e(TAG, e.toString());
			return false;
		}
		
		return exist;
	}
	
	/**
	 * しおり情報を削除します。
	 * @param contentId コンテンツIDです。
	 * @param pageNum ページナンバー。
	 * @throws Exception その他、例外です。
	 * @since 1.0.0
	 */
	public void deleteContentBookmark(long contentId,int pageNum) {
		contentBookmarkDao.deleteBookmark(contentId, pageNum);
	}
	
	/**
	 * コンテンツ間でしおりをコピーします。
	 * 
	 * @param srcContentId コピー元コンテンツID
	 * @param dstContentId コピー先コンテンツID
	 * @throws Exception
	 */
	public void copyContentBookmark(long srcContentId,long dstContentId) {
		List<ContentBookmarkDto> srcBookmarkDtoList = getContentBookmark(srcContentId);
		ContentDto dstContentDto = contentDao.getContent(dstContentId);
		for (ContentBookmarkDto srcBookmarkDto: srcBookmarkDtoList) {
			if (dstContentDto.allPageNum > srcBookmarkDto.pageNum && !isExsitContentBookmark(dstContentId, srcBookmarkDto.pageNum)) {
				srcBookmarkDto.contentId = dstContentId;
				insertContentBookmark(srcBookmarkDto);
			}
		}
	}

	/**
	 * しおり情報を追加します。
	 * @param dto しおり情報
	 * @throws Exception その他、例外です。
	 * @since 1.0.0
	 */
	public void insertContentBookmark(ContentBookmarkDto dto) {
		dto.pageText = contentPageDao.getContentPageText(dto.contentId, dto.pageNum + 1);
		dto.insertDate = DateTimeUtil.getCurrentDate();
		dto.updateDate = DateTimeUtil.getCurrentDate();
		//サムネイルコピー
		dto.pageThumbnailName = dto.pageNum + ".jpg";

		contentBookmarkDao.insertBookmark(dto);
	}


}
