package com.theta.network;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.util.Log;

import com.theta.model.ImageSize;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.TimeZone;
import java.util.Timer;
import java.util.TimerTask;

import jp.agentec.abook.abv.bl.common.log.Logger;

/**
 * HTTP connection to device
 */
public class HttpConnector {
    private static final String TAG = "HttpConnector";

    private final static long CHECK_STATUS_PERIOD_MS = 50;
    private final static int HTTP_CONNECT_TIME_OUT = 5000;
    private final static int GET_IMAGE_MAX_COUNT = 1000;

    private String mIpAddress = null;

    private String mFingerPrint = null;
    private Timer mCheckStatusTimer = null;
    private HttpEventListener mHttpEventListener = null;

    private boolean mIsOldApi = false; //true:api2.0, false:api2.1
    //API2.0利用
    private String mSessionId = null;
    private String mContinuationToken = null;

    public enum ShootResult {
        SUCCESS, FAIL_CAMERA_DISCONNECTED, FAIL_STORE_FULL, FAIL_DEVICE_BUSY
    }

    /**
     * Constructor
     * @param cameraIpAddress IP address of connection destination
     */
    public HttpConnector(String cameraIpAddress) {
        mIpAddress = cameraIpAddress;

    }

    /**
     * Constructor
     * @param cameraIpAddress IP address of connection destination
     * @param isOldApi true:api2.0, false:api2.1
     */
    public HttpConnector(String cameraIpAddress, boolean isOldApi) {
        mIpAddress = cameraIpAddress;
        mIsOldApi = isOldApi;
    }

    /**
     * Connect to device
     * @return Session ID (null is returned if the connection fails)
     */
    private String connect() {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        String sessionId = null;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.startSession");

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");
                sessionId = results.getString("sessionId");
            } else if (status.equals("error")) {
                JSONObject errors = output.getJSONObject("error");
                String errorCode = errors.getString("code");
                if (errorCode.equals("invalidSessionId")) {
                    sessionId = null;
                }
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return sessionId;
    }

    /**
     * Acquire storage information of device
     * @return Storage information
     */
    public StorageInfo getStorageInfo() {

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        StorageInfo storageInfo = new StorageInfo();
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.getOptions");
            JSONObject parameters = new JSONObject();
            JSONArray optionNames = new JSONArray();
            optionNames.put("remainingPictures");
            optionNames.put("remainingSpace");
            optionNames.put("totalSpace");
            parameters.put("optionNames", optionNames);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");
                JSONObject options = results.getJSONObject("options");
                int remainingPictures = options.getInt("remainingPictures");
                storageInfo.setFreeSpaceInImages(remainingPictures);

                long remainingSpace = options.getLong("remainingSpace");
                storageInfo.setFreeSpaceInBytes(remainingSpace);

                long totalSpace = options.getLong("totalSpace");
                storageInfo.setMaxCapacity(totalSpace);
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return storageInfo;
    }

    /**
     * Acquire device information
     * @return Device information
     */
    public ThetaDeviceInfo getDeviceInfo() {
        HttpURLConnection getConnection = createHttpConnection("GET", "/osc/info");
        String responseData;
        ThetaDeviceInfo deviceInfo = new ThetaDeviceInfo();
        InputStream is = null;

        try {
            // send HTTP GET
            // this protocol has no input.
            getConnection.connect();
            is = getConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);

            String model = output.getString("model");
            deviceInfo.setModel(model);

            String version = output.getString("firmwareVersion");
            deviceInfo.setDeviceVersion(version);

            String serialNumber = output.getString("serialNumber");
            deviceInfo.setSerialNumber(serialNumber);
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return deviceInfo;
    }

    /**
     * Acquire list of media files on device
     * @return Media file list
     */
    public ArrayList<ImageInfo> getList() {
        ArrayList<ImageInfo> imageInfos = new ArrayList<>();

        for (int continuation = 0; continuation < 10; continuation++) {
            ArrayList<ImageInfo> receivedImageInfo = getListInternal(GET_IMAGE_MAX_COUNT, imageInfos.size());
            if (receivedImageInfo == null) {
                imageInfos = null;
                break;
            } else {
                imageInfos.addAll(receivedImageInfo);
                if (mIsOldApi) {
                    //API2.0
                    if (mContinuationToken == null) {
                        break;
                    }
                } else {
                    //API2.1
                    if (receivedImageInfo.size() < GET_IMAGE_MAX_COUNT) {
                        break;
                    }
                }
            }

        }

        return imageInfos;
    }

    /**
     * Acquire media file list (limited number of items)
     * @param maxReceiveEntry Maximum number of files that can be acquired at once
     * @param startPosition Set the previously acquired token to continue. Set null if acquiring for the first time.
     * @return List of specified number of media files
     */
    private ArrayList<ImageInfo> getListInternal(int maxReceiveEntry, int startPosition) {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        ArrayList<ImageInfo> imageInfos = new ArrayList<>();
        InputStream is = null;

        try {
            JSONObject parameters = new JSONObject();
            //AIP2.0
            if (mIsOldApi) {
                input.put("name", "camera._listAll");
                if (mContinuationToken != null) {
                    parameters.put("continuationToken", mContinuationToken);
                }
            } else {
                input.put("name", "camera.listFiles");
                //THETA SC2の場合、imageのみ取得すると、最新２つの画像ファイル取得できないため、全て取得
                parameters.put("fileType", "all");
                parameters.put("maxThumbSize", 0);
                parameters.put("startPosition", startPosition);
            }
            parameters.put("entryCount", maxReceiveEntry);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");
                JSONArray entries = results.getJSONArray("entries");
                int entrySize = entries.length();
                if (mIsOldApi) {
                    try {
                        mContinuationToken = results.getString("continuationToken");
                    } catch (JSONException e) {
                        mContinuationToken = null;
                    }
                }

                for (int index = 0; index < entrySize; index++) {
                    JSONObject entry = entries.getJSONObject(index);
                    ImageInfo imageInfo = new ImageInfo();

                    String name = entry.getString("name");
                    imageInfo.setFileName(name);

                    //API2.0
                    String id = null;
                    if (mIsOldApi) {
                        id = entry.getString("uri");
                    } else {
                        id = entry.getString("fileUrl");
                    }
                    imageInfo.setFileId(id);

                    long size = Long.parseLong(entry.getString("size"));
                    imageInfo.setFileSize(size);

                    String date = entry.getString("dateTimeZone");
                    imageInfo.setCaptureDate(date);

                    int width = entry.getInt("width");
                    imageInfo.setWidth(width);

                    int height = entry.getInt("height");
                    imageInfo.setHeight(height);

                    try {
                        //API2.0
                        if (mIsOldApi) {
                            //画像ファイルの場合、「recordTime」がないのでJSONException発生
                            entry.getInt("recordTime");
                            //動画ファイルは保存しない。
                            continue;
                        } else { //API2.1では画像ファイルのみとなるので、FileFormatはJPEG固定
                            //THETA SC2では「_recordTime」値が画像ファイルでも0が設定されているので、例外発生しない。
                            int recordTime = entry.getInt("_recordTime");
                            if (recordTime > 0) {
                                //動画ファイルは保存しない。
                                continue;
                            }
                        }
                    } catch (JSONException e) {
                        //画像ファイルなので何もしない。
                    }
                    imageInfos.add(imageInfo);
                }
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
            imageInfos = null;
        } catch (JSONException e) {
            Logger.e(TAG, e);
            imageInfos = null;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                    imageInfos = null;
                }
            }
        }

        return imageInfos;
    }
    /**
     * Acquire thumbnail image(Only API2.0)
     * @param fileId File ID
     * @return Thumbnail (null is returned if acquisition fails)
     */
    private Bitmap getThumbOld(String fileId) {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        Bitmap thumbnail = null;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.getImage");
            JSONObject parameters = new JSONObject();
            parameters.put("fileUri", fileId);
            parameters.put("_type", "thumb");
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            BufferedInputStream bis = new BufferedInputStream(is);
            thumbnail = BitmapFactory.decodeStream(bis);
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return thumbnail;
    }

    /**
     * Acquire thumbnail image
     * @param fileId File ID
     * @return Thumbnail (null is returned if acquisition fails)
     */
    public Bitmap getThumb(String fileId) {
        //API2.0
        if (mIsOldApi) {
            return getThumbOld(fileId);
        }

        HttpURLConnection postConnection = null;
        try {
            postConnection = (HttpURLConnection)new URL(fileId+"?type=thumb").openConnection();
        } catch (IOException e) {
            Logger.e(TAG, e);
        }
        JSONObject input = new JSONObject();
        Bitmap thumbnail = null;
        InputStream is = null;

        try {
            // send HTTP GET
            assert postConnection != null;
            postConnection.connect();

            is = postConnection.getInputStream();
            BufferedInputStream bis = new BufferedInputStream(is);
            thumbnail = BitmapFactory.decodeStream(bis);
        } catch (IOException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return thumbnail;
    }

    /**
     * Take photo<p>
     * After shooting, the status is checked for each {@link HttpConnector#CHECK_STATUS_PERIOD_MS} and the listener notifies you of the status.
     * @param listener Post-shooting event listener
     * @return Shooting request results
     */
    public ShootResult takePicture(HttpEventListener listener) {
        ShootResult result = ShootResult.FAIL_DEVICE_BUSY;

        //API2.0 start
        if (mIsOldApi) {
            mSessionId = connect();
            if (mSessionId == null) {
                listener.onError("cannot get to start session");
                result = ShootResult.FAIL_DEVICE_BUSY;
                return result;
            }
        }

        // set capture mode to image
        String errorMessage = setImageCaptureMode();
        if (errorMessage != null) {
            listener.onError(errorMessage);
            result = ShootResult.FAIL_DEVICE_BUSY;
            return result;
        }

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        mHttpEventListener = listener;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.takePicture");

            //API2.0 start
            if (mIsOldApi) {
                JSONObject parameters = new JSONObject();
                parameters.put("sessionId", mSessionId);
                input.put("parameters", parameters);
            }
            //API2.0 end

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");
            String commandId = output.getString("id");

            if (status.equals("inProgress")) {
                mCheckStatusTimer = new Timer(true);
                CapturedTimerTask capturedTimerTask = new CapturedTimerTask();
                capturedTimerTask.setCommandId(commandId);
                mCheckStatusTimer.scheduleAtFixedRate(capturedTimerTask, CHECK_STATUS_PERIOD_MS, CHECK_STATUS_PERIOD_MS);
                result = ShootResult.SUCCESS;
            } else if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");
                String lastFileId = results.getString("fileUri");

                mHttpEventListener.onObjectChanged(lastFileId);
                mHttpEventListener.onCompleted();
                result = ShootResult.SUCCESS;
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
            result = ShootResult.FAIL_DEVICE_BUSY;
        } catch (JSONException e) {
            Logger.e(TAG, e);
            result = ShootResult.FAIL_DEVICE_BUSY;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return result;
    }

    private class CapturedTimerTask extends TimerTask {
        private String mCommandId;

        public void setCommandId(String commandId) {
            mCommandId = commandId;
        }

        @Override
        public void run() {
            String capturedFileId = checkCaptureStatus(mCommandId);
            if (capturedFileId != null) {
                if (capturedFileId.equals("fail")) {
                    mCheckStatusTimer.cancel();
                    mHttpEventListener.onError("Execption Catch");
                } else {
                    mHttpEventListener.onCheckStatus(true);
                    mCheckStatusTimer.cancel();
                    mHttpEventListener.onObjectChanged(capturedFileId);
                    mHttpEventListener.onCompleted();
                }

            } else {
                mHttpEventListener.onCheckStatus(false);
            }
        }
    }

    /**
     * Check still image shooting status
     * @param commandId Command ID for shooting still images
     * @return ID of saved file (null is returned if the file is not saved)
     */
    private String checkCaptureStatus(String commandId) {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/status");
        JSONObject input = new JSONObject();
        String responseData;
        String capturedFileId = null;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("id", commandId);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");

                //API2.0
                if (mIsOldApi) {
                    capturedFileId = results.getString("fileUri");
                } else {
                    capturedFileId = results.getString("fileUrl");
                }
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
            capturedFileId = "fail";
        } catch (JSONException e) {
            capturedFileId = "fail";
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    capturedFileId = "fail";
                    Logger.e(TAG, e);
                }
            }
        }

        return capturedFileId;
    }
    /**
     * Acquire raw data of specified image (Only API2.0)
     * @param fileId File ID
     * @param listener Listener for receiving received data count
     * @return Image data
     */
    public ImageData getImageOld(String fileId, HttpDownloadListener listener) {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        ImageData imageData = new ImageData();
        int totalSize = 0;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.getImage");
            JSONObject parameters = new JSONObject();
            parameters.put("fileUri", fileId);
            parameters.put("_type", "full");
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            totalSize = postConnection.getContentLength();
            listener.onTotalSize(totalSize);
            is = postConnection.getInputStream();
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            byte[] buffer = new byte[2048];
            int length;

            while ((length = is.read(buffer)) >= 0) {
                baos.write(buffer, 0, length);
                listener.onDataReceived(length);
            }
            byte[] rawData = baos.toByteArray();
            imageData.setRawData(rawData);

            XMP xmp = new XMP(rawData);
            imageData.setPitch(xmp.getPosePitchDegrees());
            imageData.setRoll(xmp.getPoseRollDegrees());
        } catch (IOException e) {
            Logger.e(TAG, e);
            imageData = null;
        } catch (JSONException e) {
            Logger.e(TAG, e);
            imageData = null;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                    imageData = null;
                }
            }
        }

        return imageData;
    }
    /**
     * Acquire raw data of specified image
     * @param fileId File ID
     * @param listener Listener for receiving received data count
     * @return Image data
     */
    public ImageData getImage(String fileId, HttpDownloadListener listener) {
        //API2.0
        if (mIsOldApi) {
            return getImageOld(fileId, listener);
        }
        HttpURLConnection postConnection = null;
        try {
            postConnection = (HttpURLConnection)new URL(fileId).openConnection();
            postConnection.setConnectTimeout(HTTP_CONNECT_TIME_OUT);
        } catch (IOException e) {
            Logger.e(TAG, e);
        }
        ImageData imageData = new ImageData();
        int totalSize = 0;
        InputStream is = null;

        try {
            if (postConnection == null) {
                return null;
            }
            // send HTTP GET
            postConnection.connect();

            totalSize = postConnection.getContentLength();
            listener.onTotalSize(totalSize);
            is = postConnection.getInputStream();
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            byte[] buffer = new byte[2048];
            int length;

            while ((length = is.read(buffer)) >= 0) {
                baos.write(buffer, 0, length);
                listener.onDataReceived(length);
            }
            byte[] rawData = baos.toByteArray();
            imageData.setRawData(rawData);

            XMP xmp = new XMP(rawData);
            imageData.setPitch(xmp.getPosePitchDegrees());
            imageData.setRoll(xmp.getPoseRollDegrees());
        } catch (IOException e) {
            Logger.e(TAG, e);
            imageData = null;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                    imageData = null;
                }
            }
        }

        return imageData;
    }

    /**
     * Acquire live view stream
     * @return Stream for receiving data
     * @throws IOException
     */
    public InputStream getLivePreview() throws IOException, JSONException {
        //API2.0
        if (mIsOldApi) {
            mSessionId = connect();
        }


        // set capture mode to image
        setImageCaptureMode();

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        InputStream is = null;

        try {
            JSONObject parameters = new JSONObject();
            //API2.0
            if (mIsOldApi) {
                input.put("name", "camera._getLivePreview");
                parameters.put("sessionId", mSessionId);
            } else {
                input.put("name", "camera.getLivePreview");
            }
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
        } catch (IOException e) {
            Logger.e(TAG, e);
            String errorMessage = null;
            InputStream es = postConnection.getErrorStream();
            try {
                if (es != null) {
                    String errorData = InputStreamToString(es);
                    JSONObject output = new JSONObject(errorData);
                    JSONObject errors = output.getJSONObject("error");
                    errorMessage = errors.getString("message");
                    Logger.e(TAG, errorMessage);
                }
            } catch (IOException e1) {
                e1.printStackTrace();
                Logger.e(TAG, e1);
            } catch (JSONException e1) {
                e1.printStackTrace();
                Logger.e(TAG, e1);
            } finally {
                if (es != null) {
                    try {
                        es.close();
                    } catch (IOException e1) {
                        e1.printStackTrace();
                        Logger.e(TAG, e1);
                    }
                }
            }
            throw e;
        } catch (JSONException e) {
            Logger.e(TAG, e);
            throw e;
        }

        return is;
    }

    /**
     * Delete specified file
     * @param deletedFileId File ID
     * @param listener Listener for receiving deletion results
     */
    public void deleteFile(String deletedFileId, HttpEventListener listener) {
        //API2.0
        if (mIsOldApi) {
            mSessionId = connect();
        }

        // set capture mode to image
        String errorMessage = setImageCaptureMode();
        if (errorMessage != null) {
            listener.onError(errorMessage);
            return;
        }

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");

        JSONObject input = new JSONObject();
        String responseData;
        mHttpEventListener = listener;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.delete");
            JSONObject parameters = new JSONObject();
            if (mIsOldApi) {
                parameters.put("fileUri", deletedFileId);
            } else {
                JSONArray ja = new JSONArray();
                ja.put(deletedFileId);
                parameters.put("fileUrls", ja);
            }

            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("inProgress")) {
                getState();
                mCheckStatusTimer = new Timer(true);
                DeletedTimerTask deletedTimerTask = new DeletedTimerTask();
                deletedTimerTask.setDeletedFileId(deletedFileId);
                mCheckStatusTimer.scheduleAtFixedRate(deletedTimerTask, CHECK_STATUS_PERIOD_MS, CHECK_STATUS_PERIOD_MS);
            } else if (status.equals("done")) {
                mHttpEventListener.onObjectChanged(deletedFileId);
                mHttpEventListener.onCompleted();
                mFingerPrint = null;
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }
    }

    /**
     * Status check class for file deletion
     */
    private class DeletedTimerTask extends TimerTask {
        private String mDeletedFileId = null;

        public void setDeletedFileId(String deletedFileId) {
            mDeletedFileId = deletedFileId;
        }

        @Override
        public void run() {
            boolean update = isUpdate();
            mHttpEventListener.onCheckStatus(update);
            if (update) {
                mCheckStatusTimer.cancel();
                getState();
                mHttpEventListener.onObjectChanged(mDeletedFileId);
                mHttpEventListener.onCompleted();
                mFingerPrint = null;
            }
        }
    }

    /**
     * Specify shooting size
     * @param imageSize Shooting size
     */
    public void setImageSize(ImageSize imageSize) {
        int width;
        int height;
        switch (imageSize) {
            case IMAGE_SIZE_2048x1024:
                width = 2048;
                height = 1024;
                break;
            default:
            case IMAGE_SIZE_5376x2688:
                width = 5376;
                height = 2688;
                break;
        }

        // set capture mode to image
        setImageCaptureMode();

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.setOptions");
            JSONObject parameters = new JSONObject();
            JSONObject options = new JSONObject();
            JSONObject fileFormat = new JSONObject();
            fileFormat.put("type", "jpeg");
            fileFormat.put("width", width);
            fileFormat.put("height", height);
            options.put("fileFormat", fileFormat);
            parameters.put("options", options);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }
    }

    /**
     * Acquire currently set shooting size
     * @return Shooting size (null is returned if acquisition fails)
     */
    public ImageSize getImageSize() {

        // set capture mode to image
        setImageCaptureMode();

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        ImageSize imageSize = null;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.getOptions");
            JSONObject parameters = new JSONObject();
            JSONArray optionNames = new JSONArray();
            optionNames.put("fileFormat");
            parameters.put("optionNames", optionNames);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");
                JSONObject options = results.getJSONObject("options");
                JSONObject fileFormat = options.getJSONObject("fileFormat");
                int width = fileFormat.getInt("width");

                if (width == 2048) {
                    imageSize = ImageSize.IMAGE_SIZE_2048x1024;
                } else {
                    imageSize = ImageSize.IMAGE_SIZE_5376x2688;
                }
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return imageSize;
    }

    /**
     * Set still image as shooting mode
     * @return Error message (null is returned if successful)
     */
    private String setImageCaptureMode() {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        String errorMessage = null;
        InputStream is = null;

        try {
            // send HTTP POST
            input.put("name", "camera.setOptions");
            JSONObject parameters = new JSONObject();
            JSONObject options = new JSONObject();
            options.put("captureMode", "image");
            //API2.0
            if (mSessionId != null) {
                parameters.put("sessionId", mSessionId);
                //THETA S端末の時間初期化問題で、現在時間は常に設定
                options.put("dateTimeZone", currentDateTimeZone());
            }

            parameters.put("options", options);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("error")) {
                JSONObject errors = output.getJSONObject("error");
                errorMessage = errors.getString("message");
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
            errorMessage = e.toString();
            InputStream es = postConnection.getErrorStream();
            try {
                if (es != null) {
                    String errorData = InputStreamToString(es);
                    JSONObject output = new JSONObject(errorData);
                    JSONObject errors = output.getJSONObject("error");
                    errorMessage = errors.getString("message");
                }
            } catch (IOException e1) {
                e1.printStackTrace();
            } catch (JSONException e1) {
                e1.printStackTrace();
            } finally {
                if (es != null) {
                    try {
                        es.close();
                    } catch (IOException e1) {
                        e1.printStackTrace();
                    }
                }
            }
        } catch (JSONException e) {
            Logger.e(TAG, e);
            errorMessage = e.toString();
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return errorMessage;
    }

    /**
     * Acquire device status
     * @return Last saved file
     */
    private String getState() {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/state");
        String responseData;
        String lastFile = "";
        InputStream is = null;

        try {
            // send HTTP POST
            postConnection.connect();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            mFingerPrint = output.getString("fingerprint");
            JSONObject status = output.getJSONObject("state");
            lastFile = status.getString("_latestFileUri");
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return lastFile;
    }

    /**
     * Check for updates to device status
     * @return true:Update available, false:No update available
     */
    private boolean isUpdate() {
        boolean update = false;
        InputStream is = null;

        if (mFingerPrint == null) {
            return update;
        }

        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/checkForUpdates");
        JSONObject input = new JSONObject();
        String responseData = null;

        try {
            // send HTTP POST
            input.put("stateFingerprint", mFingerPrint);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String currentFingerPrint = output.getString("stateFingerprint");
            if (!currentFingerPrint.equals(mFingerPrint)) {
                mFingerPrint = currentFingerPrint;
                update = true;
            }
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return update;
    }

    /**
     * Generate connection destination URL
     * @param path Path
     * @return URL
     */
    private String createUrl(String path) {
        StringBuilder sb = new StringBuilder();
        sb.append("http://");
        sb.append(mIpAddress);
        sb.append(path);

        return sb.toString();
    }

    /**
     * Generate HTTP connection
     * @param method Method
     * @param path Path
     * @return HTTP Connection instance
     */
    private HttpURLConnection createHttpConnection(String method, String path) {
        HttpURLConnection connection = null;
        try {
            URL url = new URL(createUrl(path));
            connection = (HttpURLConnection) url.openConnection();
            connection.setRequestProperty("Content-Type", "application/json;charset=utf-8");
            connection.setRequestProperty("Accept", "application/json");
            connection.setDoInput(true);
            connection.setConnectTimeout(HTTP_CONNECT_TIME_OUT);

            if (method.equals("POST")) {
                connection.setRequestMethod(method);
                connection.setDoOutput(true);
            }

        } catch (MalformedURLException e) {
            Logger.e(TAG, e);
        } catch (IOException e) {
            Logger.e(TAG, e);
        }

        return connection;
    }

    /**
     * Convert input stream to string
     * @param is InputStream
     * @return String
     * @throws IOException IO error
     */
    private String InputStreamToString(InputStream is) throws IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(is, "UTF-8"));
        StringBuilder sb = new StringBuilder();
        String lineData;
        while ((lineData = br.readLine()) != null) {
            sb.append(lineData);
        }
        br.close();
        return sb.toString();
    }

    /**
     * THETAカメラのWi-Fi機能をOFFに設定
     * @return 結果値（true:成功、false:失敗）
     */
    public boolean cameraFinishWlan() {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        String errorMessage = null;
        InputStream is = null;
        boolean isSuccess = false;
        try {
            // send HTTP POST
            input.put("name", "camera._finishWlan");

            //API2.0
            if (mIsOldApi) {
                mSessionId = connect();
                if (mSessionId != null) {
                    JSONObject parameters = new JSONObject();
                    parameters.put("sessionId", mSessionId);
                    input.put("parameters", parameters);
                }
            }

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            InputStreamToString(is);
            isSuccess = true;
        } catch (IOException e) {
            Logger.e(TAG, e);
            Logger.e(TAG, e.toString());
            InputStream es = postConnection.getErrorStream();
            try {
                if (es != null) {
                    String errorData = InputStreamToString(es);
                    JSONObject output = new JSONObject(errorData);
                    JSONObject errors = output.getJSONObject("error");
                    Logger.e(TAG, errors.getString("message"));
                }
            } catch (IOException e1) {
                Logger.e(TAG, e1);
            } catch (JSONException e1) {
                Logger.e(TAG, e1);
            } finally {
                if (es != null) {
                    try {
                        es.close();
                    } catch (IOException e1) {
                        Logger.e(TAG, e1);
                    }
                }
            }
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }
        return isSuccess;
    }

    /**
     * THETAカメラの露出情報を取得
     * @return 文字列型の露出値
     */
    public String getOptionExposure() {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        InputStream is = null;
        String exposureCompensation = null;
        try {
            // send HTTP POST
            input.put("name", "camera.getOptions");
            JSONObject parameters = new JSONObject();
            JSONArray optionNames = new JSONArray();
            optionNames.put("exposureCompensation");

            //API2.0
            if (mIsOldApi) {
                mSessionId = connect();
                if (mSessionId != null) {
                    parameters.put("sessionId", mSessionId);
                }
            }

            parameters.put("optionNames", optionNames);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            responseData = InputStreamToString(is);

            // parse JSON data
            JSONObject output = new JSONObject(responseData);
            String status = output.getString("state");

            if (status.equals("done")) {
                JSONObject results = output.getJSONObject("results");
                JSONObject options = results.getJSONObject("options");
                double exposure = options.getDouble("exposureCompensation");
                if (exposure == 0) {
                    exposureCompensation = "0.0";
                } else {
                    BigDecimal decimal = new BigDecimal(String.valueOf(exposure));
                    BigDecimal halfUp = decimal.setScale(1, RoundingMode.HALF_UP); //小数点1桁まで表示し、2桁で切り上げ
                    exposureCompensation = halfUp.toPlainString();
                }

            }
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }
        return exposureCompensation;
    }

    /**
     * THETAカメラの露出情報を設定
     * @param exposure 露出値
     * @return 結果値（true:成功、false:失敗）
     */
    public boolean setOptionExposure(String exposure) {
        HttpURLConnection postConnection = createHttpConnection("POST", "/osc/commands/execute");
        JSONObject input = new JSONObject();
        String responseData;
        InputStream is = null;
        String exposureCompensation = null;
        boolean isSuccess = false;
        try {
            // send HTTP POST
            input.put("name", "camera.setOptions");
            JSONObject parameters = new JSONObject();
            JSONObject options = new JSONObject();
            options.put("exposureCompensation", Double.parseDouble(exposure));
            //API2.0 start
            if (mIsOldApi) {
                mSessionId = connect();
                if (mSessionId != null) {
                    parameters.put("sessionId", mSessionId);
                }
            }
            parameters.put("options", options);
            input.put("parameters", parameters);

            OutputStream os = postConnection.getOutputStream();
            os.write(input.toString().getBytes());
            postConnection.connect();
            os.flush();
            os.close();

            is = postConnection.getInputStream();
            InputStreamToString(is);
            isSuccess = true;
        } catch (IOException e) {
            Logger.e(TAG, e);
        } catch (JSONException e) {
            Logger.e(TAG, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    Logger.e(TAG, e);
                }
            }
        }

        return isSuccess;
    }

    /**
     * yyyy:MM:dd HH:mm:ssXXXフォマットで現在の日付を返す。
     * @return 現在の日付
     */
    private String currentDateTimeZone() {
        return new SimpleDateFormat("yyyy:MM:dd HH:mm:ssXXX").format(new Date());
    }
}
