package jp.agentec.abook.abv.ui.viewer.activity;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.os.Bundle;
import android.util.Base64;
import android.view.Surface;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.DownloadListener;
import android.webkit.JavascriptInterface;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ImageButton;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;

import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.common.dialog.ABookAlertDialog;
import jp.agentec.abook.abv.ui.common.util.AlertDialogUtil;

public class PhotoEditActivity extends Dialog {
    private final String TAG = "PhotoEditActivity";
    private final String editToolPath = "file:///android_asset/Drawing/index.html"; //EditToolのパス。
    private final String editToolName = "Drawing";  //EditToolの名
    private final int dialogMinWidthWeb = 280; //ダイアローグの最小の横幅
    private WebView editPageWebView;    //ダイアローグの上に編集画面をロード
    private Bitmap bitmapOfPhoto;   //写真のBitmap情報
    private float photoWidth;
    private float photoHeight;
    private float displayHeight;
    private float displayDensity;
    private String photoFilePath;   //ロードして修正するファイルのパス
    private JsInf jsInf = new JsInf();  //Javascript Interface
    private Context context;

    /**
     * 生成される時、イメージパスを取得。
     * @param context
     * @param photoFilePath　イメージパス
     */
    public PhotoEditActivity(Context context, String photoFilePath){
        super(context);
        this.context = context;
        this.photoFilePath = photoFilePath;
    }

    /**
     * ダイアローグが生成される時の処理
     */
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        //ダイアローグの設定
        requestWindowFeature(Window.FEATURE_NO_TITLE); //ダイアローグのタイトルバーを消す
        setContentView(R.layout.photo_edit_dialog); //ダイアローグにWebViewを設定
        setCanceledOnTouchOutside(false);   //編集画面の外をタッチしてもダイアローグを閉じらないようにする。

        //写真のBitmapをロード
        bitmapOfPhoto = BitmapFactory.decodeFile(photoFilePath);
        photoWidth = bitmapOfPhoto.getWidth();
        photoHeight = bitmapOfPhoto.getHeight();

        //ダイアローグのサイズの設定
        //画面のサイズを取得
        Rect displayRectangle = new Rect();
        Window window = ((Activity) context).getWindow();
        window.getDecorView().getWindowVisibleDisplayFrame(displayRectangle);
        float photoRatio = photoWidth / photoHeight; //画面の横と縦の割合
        displayHeight = (float) displayRectangle.height(); //画面の縦幅
        displayDensity = getContext().getResources().getDisplayMetrics().density;
        float displayWidth = (float) displayRectangle.width(); //画面の横幅
        int dialogWidth = (int) (displayHeight * photoRatio); //ダイアローグの横幅を縦幅について計算

        if(photoRatio > 1){ dialogWidth = (int) (dialogWidth * 0.75f); } //横の写真はダイアローグの横幅を75パーセントに計算
        else{ dialogWidth = (int) (dialogWidth * 0.9f); } //縦の写真はダイアローグの横幅を90パーセントに計算

        if(getWindow() != null) {
            if (dialogWidth < displayWidth) {
                //計算した横幅が画面より小さい場合はそのまま設定
                if (dialogWidth < dialogMinWidthWeb * displayDensity){
                    dialogWidth = (int) (dialogMinWidthWeb * displayDensity);
                }
                getWindow().setLayout(dialogWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
            } else {
                //計算した横幅が画面より大き場合は画面の横幅に反映
                getWindow().setLayout((int) displayWidth, ViewGroup.LayoutParams.WRAP_CONTENT);
            }
        }

        //クローズボタン
        final ImageButton imageButton = findViewById(R.id.edit_page_close_btn);
        imageButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                dismiss();
            }
        });

        //WebViewの設定
        editPageWebView = findViewById(R.id.edit_webview);
        editPageWebView.setOverScrollMode(View.OVER_SCROLL_NEVER);  //オーバースクロールしない。
        editPageWebView.setVerticalScrollBarEnabled(false); //スクロールバーを消す。
        editPageWebView.addJavascriptInterface(jsInf, "android");

        editPageWebView.getSettings().setJavaScriptEnabled(true); //Javascriptを有効にする。

        editPageWebView.setVisibility(View.GONE);

        //ターゲットバージョン30以上からデフォルトがfalseに設定されている問題対応
        editPageWebView.getSettings().setAllowFileAccess(true);

        //ページをロード
        editPageWebView.loadUrl(editToolPath);

        editPageWebView.reload();
        editPageWebView.setVisibility(View.VISIBLE);

        editPageWebView.setWebViewClient(new WebViewClient() {
            @Override
            public void onPageFinished(WebView view, String url) {
                //ページロードの終了後の処理
                super.onPageFinished(view, url);
                if (url.contains(editToolName)) {
                    //最初ページロードの場合
                    //ページロードの終了後のCanvas設定
                    float photoRatio = photoHeight / photoWidth; //写真の割合を計算
                    view.loadUrl(String.format("javascript:resizeCanvasToRatio(%f, %f);", photoRatio, (displayHeight / displayDensity * 0.7f)));  //写真の割合によってCanvasのサイズを決める。
                    view.loadUrl(String.format("javascript:setBackground('%s');", photoFilePath));  //Canvasの背景をイメージにする
                }
            }
        });
        editPageWebView.setDownloadListener(new DownloadListener() {
            /**
             * ダウンロードが始まるときの処理。urlとcontentLength以外のパラメタは内容がない。
             * @param url   HTML CanvasのBase 64情報
             * @param contentLength    urlの長さ
             */
            @Override
            public void onDownloadStart(String url, String userAgent,
                                        String contentDisposition, String mimeType,
                                        long contentLength) {
                //編集画面の保存ボタンを押した時の処理
                //手書きのイメージの処理
                //手書きのBase64情報をBitmapに変換
                Bitmap bitmapOfDrawing = decodeBase64ToBitmap(url);

                //写真と手書きを合わせる
                Bitmap completeImage = mergeImages(bitmapOfDrawing, bitmapOfPhoto);

                //合わせたイメージをphotoFilePathに保存
                boolean isError = saveBitmapToPath(completeImage, photoFilePath, 100);

                if (isError) {
                    //保存中にエラーが発生した場合、既存ファイルを保存する。
                    saveBitmapToPath(bitmapOfPhoto, photoFilePath, 100);
                    //保存失敗のアラートを表示する。
                    showAlertDialog(R.string.error,R.string.msg_error_edit_page_save, false);
                } else {
                    //保存中にエラーがない場合、進行する。
                    bitmapOfPhoto.recycle();    //メモリーでBitmapを解放
                    editPageWebView.loadUrl(photoFilePath);    //編集画面のダイアローグが閉じる前、イメージファイルをロードしてWeb Cacheを更新したら、作業報告画面の画像が更新される。
                    superDismiss(); //ダウンロードの後はすぐ終了。
                }
            }
        });
    }

    @Override
    public void show() {
        super.show();
        //HTMLファイルがない場合アラートを表示して終了
        //TODO: EditToolのパスが決まるとFileUtil.exists(editToolPath)に変える
        //現在Edit ToolがAssetsフォルダにあり、FileUtil.exists(editToolPath)を使えない。
        //下記のコードはパスが変わるる場合、削除予定
        try {
            if (context.getResources().getAssets().list(editToolName).length == 0){
                showAlertDialog(R.string.error,R.string.msg_error_edit_page_open,false);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        //下記のコードは後他のパスに移動する場合、コメント解除予定
//        if (!FileUtil.exists(editToolPath)){
//            showAlertDialog(R.string.error,R.string.opening_file_error,false);
//        }
    }

    /**
     * ダイアローグを閉じる。
     */
    private void superDismiss(){
        super.dismiss();
    }

    /**
     * クローズボタンやバックボタンを押す時、閉じらなく、編集画面終了を確認するアラートを表示する。
     */
    @Override
    public void dismiss() {
        editPageWebView.loadUrl("javascript:android.confirmDismiss(isCanvasEdited());"); //修正可否をJavascriptで確認
    }

    private class JsInf {
        /**
         * 修正がない場合、ダイアローグがなく終了
         * @param isCanvasEdited イメージが修正されたか
         */
        @JavascriptInterface
        public void confirmDismiss(boolean isCanvasEdited){
            if (isCanvasEdited) {
                //修正事項がある場合、確認ダイアローグを開ける。
                showAlertDialog(R.string.end, R.string.msg_confirm_close_edit_page, true);    //ダイアローグの開ける。
            }
            else{
                //修正事項がない場合、すぐ終了。
                superDismiss();
            }
        }
    }

    /**
     * 終了確認ダイアローグを開ける。
     * @param title ダイアローグのタイトル
     * @param message   ダイアローグのメッセージ
     * @param cancellable   キャンサル可能可否
     */
    private void showAlertDialog(int title, int message, boolean cancellable){
        ABookAlertDialog aBookAlertDialog = AlertDialogUtil.createAlertDialog(context, title, message);
        aBookAlertDialog.setIcon(R.drawable.icon);
        aBookAlertDialog.setPositiveButton(R.string.ok, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                //OKボタンを押すと編集画面を閉じる
                superDismiss();
            }
        });
        if (cancellable) {
            //キャンセルボタンを押すと、何もしない。
            aBookAlertDialog.setNegativeButton(R.string.cancel, null);
        }
        aBookAlertDialog.show();
    }

    /**
     * Base64のイメージ情報をBitmapに変換する。
     * @param url   Base64のイメージ情報
     * @return  完成されたBitmap
     */
    private Bitmap decodeBase64ToBitmap(String url){
        url = url.split(",")[1];
        String base64EncodedString = url.replace(" ", "+");
        byte[] decodedBytes = Base64.decode(base64EncodedString, Base64.DEFAULT);
        return BitmapFactory.decodeByteArray(decodedBytes, 0, decodedBytes.length);
    }

    /**
     * Bitmapをpathに保存する。
     * @param bitmap　保存するBitmap
     * @param path  保存するパス
     */
    private boolean saveBitmapToPath(Bitmap bitmap, String path, int quality){
        try {
            OutputStream os = new FileOutputStream(new File(path));
            bitmap.compress(Bitmap.CompressFormat.JPEG, quality, os);
            os.flush();
            os.close();
            bitmap.recycle();
        } catch (Exception e) {
            e.printStackTrace();
            return true;
        }
        return false;
    }

    /**
     * 写真２枚を合わせる
     * @param topImage  上のイメージ
     * @param bottomImage   下のイメージ
     * @return  合わせたイメージを返す。
     */
    private Bitmap mergeImages(Bitmap topImage, Bitmap bottomImage) {
        topImage = scaleBitmap(topImage, bottomImage.getWidth(), bottomImage.getHeight());
        Bitmap result = Bitmap.createBitmap(bottomImage.getWidth(), bottomImage.getHeight(), Bitmap.Config.ARGB_8888);

        //Canvasに2枚のイメージを描く。
        Canvas canvas = new Canvas(result);
        canvas.drawBitmap(bottomImage, 0f, 0f, null);
        canvas.drawBitmap(topImage, 0f, 0f, null);
        topImage.recycle();
        return result;
    }

    /**
     * Bitmapのサイズを変更
     * @param bitmap    変更するイメージ
     * @param newWidth  新しい横幅
     * @param newHeight 新しい縦幅
     * @return  変更したイメージを返す。
     */
    private static Bitmap scaleBitmap(Bitmap bitmap, int newWidth, int newHeight) {
        Bitmap scaledBitmap = Bitmap.createBitmap(newWidth, newHeight, Bitmap.Config.ARGB_8888);

        float scaleX = newWidth / (float) bitmap.getWidth();
        float scaleY = newHeight / (float) bitmap.getHeight();
        float pivotX = 0;
        float pivotY = 0;

        Matrix scaleMatrix = new Matrix();
        scaleMatrix.setScale(scaleX, scaleY, pivotX, pivotY);

        Canvas canvas = new Canvas(scaledBitmap);
        canvas.setMatrix(scaleMatrix);
        canvas.drawBitmap(bitmap, 0, 0, null);

        return scaledBitmap;
    }
}