package jp.agentec.abook.abv.bl.data.dao;

import java.util.List;

import jp.agentec.abook.abv.bl.common.db.Cursor;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.dto.ChatRoomDto;
import jp.agentec.abook.abv.bl.dto.MemberInfoDto;
import jp.agentec.abook.abv.bl.dto.ShopMemberDto;

public class ChatRoomDao extends AbstractDao {

	/**
	 * {@link ChatRoomDao} のインスタンスを初期化します。
	 * アンドロイドの android.content.Context のインスタンス
	 * @throws ClassCastException 引数のcontextが android.content.Context 又は、その継承クラスではありません。
	 * @since 1.0.0
	 */
	/*package*/ ChatRoomDao() {
	}

	@Override
	protected ChatRoomDto convert(Cursor cursor) {
		ChatRoomDto dto = new ChatRoomDto();

		int column = cursor.getColumnIndex("chat_room_id");
		if (column != -1) {
			dto.chatRoomId = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("chat_room_name");
		if (column != -1) {
			dto.chatRoomName = cursor.getString(column);
		}
		column = cursor.getColumnIndex("type");
		if (column != -1) {
			dto.type = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("unread_count");
		if (column != -1) {
			dto.unreadCount = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("user_count");
		if (column != -1) {
			dto.userCount = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("favorite_register_date");
		if (column != -1) {
			dto.favoriteRegisterDate = cursor.getString(column);
		}
		column = cursor.getColumnIndex("message");
		if (column != -1) {
			dto.message = cursor.getString(column);
		}
		column = cursor.getColumnIndex("message_type");
		if (column != -1) {
			dto.messageType = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("insert_date");
		if (column != -1) {
			dto.insertDate = cursor.getString(column);
		}
		return dto;
	}

	public List<ChatRoomDto> getAllChatRoom() {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      cr.chat_room_id ");
		sql.append("      ,cr.chat_room_name ");
		sql.append("      ,cr.type ");
		sql.append("      ,coalesce (cr.favorite_register_date,'') favorite_register_date");
		sql.append("      ,coalesce (cr.unread_count,0) unread_count");
		sql.append("      ,coalesce (cm.message,'') message");
		sql.append("      ,coalesce (cm.message_type,0) message_type");
		sql.append("      ,coalesce (cm.insert_date,'') insert_date");
		sql.append("      ,coalesce (cr.user_count,0) user_count ");
		sql.append(" FROM ");
		sql.append("      t_chat_room AS cr ");
		sql.append(" LEFT JOIN ");
		sql.append("      ( SELECT max(insert_date) insert_date, message, message_type, chat_room_id FROM t_chat_message GROUP BY chat_room_id ) AS cm ");
		sql.append("   ON cr.chat_room_id = cm.chat_room_id ");
		sql.append(" GROUP BY cr.chat_room_id ");
		sql.append(" ORDER BY cm.insert_date DESC ");
		List<ChatRoomDto> list = rawQueryGetDtoList(sql.toString(), null, ChatRoomDto.class);
		return list;
	}

	public ChatRoomDto getFavoriteChatRoom() {
		return rawQueryGetDto("select * from t_chat_room where favorite_register_date is NOT NULL", null, ChatRoomDto.class);
	}

	public ChatRoomDto getNotFavoriteChatRoom() {
		return rawQueryGetDto("select * from t_chat_room where favorite_register_date is NULL", null, ChatRoomDto.class);
	}

	public void insertChatRoom(ChatRoomDto dto) {
		insert("insert into t_chat_room (chat_room_id, chat_room_name, type, unread_count, user_count, favorite_register_date) values (?,?,?,?,?,?)", dto.getInsertValues());
	}

	public void insertChatRoom(List<ChatRoomDto> chatRoomDtoList) {
		try {
			beginTransaction();
			for (ChatRoomDto chatRoomDto : chatRoomDtoList) {
				insertChatRoom(chatRoomDto);
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertChatRoomList failed.", e);
			throw new RuntimeException(e);
		}
	}

	public boolean updateChatRoom(ChatRoomDto dto) {
		long count = update("update t_chat_room set chat_room_name=?, type=?, unread_count=?, user_count=?, favorite_register_date=? where chat_room_id=?", dto.getUpdateValues());
		return count > 0;
	}

	public void deleteChatRoom() {
		try {
			beginTransaction();
			delete("t_chat_room", null, null);
			delete("r_chat_room_shop_member", null, null);
			delete("t_chat_message", null, null);
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("deleteChatRoom failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}

}
