package jp.agentec.adf.core;

import java.lang.reflect.Method;

import jp.agentec.adf.core.formatter.PropertyNameFormatter;
import jp.agentec.adf.util.StringUtil;

/**
 * ADF4Jの基本オブジェクトクラスです。{@link Object} を継承し艇ます。<br>
 * {@link ADFObject#toString()} を呼び出すことで、このクラス及びこのクラスを継承したクラスが持っているpublicのgetterメソッドの戻り値を纏めた文字列を取得することができます。
 * @author Taejin Hong
 * @version 1.0.0
 *
 */
public abstract class ADFObject {
	private static final String ToStringDelimiter = ":";
	private static final String Brace_L = "[";
	private static final String Brace_R = "]";
	private static final String Comma = ", ";
	
	/**
	 * このクラス及びこのクラスを継承したクラスが持っているpublicのgetterメソッドの戻り値を纏めた文字列を取得することができます。
	 * @see java.lang.Object#toString()
	 * @since 1.0.0
	 */
	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer();
		Class<?> c = getClass();
		Method[] methods = c.getMethods();
		
		StringUtil.appendLine(sb, c.getName());
		sb.append(Brace_L);
		
		for (Method method : methods) {
			if (method.getName().startsWith(PropertyNameFormatter.DefaultGetterPrefix)) {
				String temp = PropertyNameFormatter.generatePropertyNameFromGetter(method.getName());
				if (!StringUtil.isNullOrWhiteSpace(temp) && !temp.equals(PropertyNameFormatter.IgnorePropertyName)) {
					sb.append(temp);
					sb.append(ToStringDelimiter);
					
					try {
						sb.append(method.invoke(this, new Object[]{}));
					} catch (Exception e) {
						sb.append(e.getMessage());
					}
					
					sb.append(Comma);
				}
			}
		}
		
		if (sb.length() >= Comma.length()) {
			sb.delete(sb.length() - Comma.length(), sb.length());
		}
		
		sb.append(Brace_R);
		
		return sb.toString();
	}
}
