package jp.agentec.abook.abv.cl.util;

import android.app.Activity;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothGattService;
import android.bluetooth.BluetoothManager;
import android.bluetooth.BluetoothProfile;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanResult;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;

import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.UUID;

import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.adf.util.StringUtil;

import static android.bluetooth.BluetoothDevice.TRANSPORT_LE;

public class BleManagerUtil {
    private final String TAG = "BleManagerUtil";

    private BluetoothManager bluetoothManager;
    private Context mContext;
    private BleManagerUtilListener mListener;
    private Handler mHandler;

    // for Notification
    private UUID UUID_NOTIFY;

    // メンバー変数
    public BluetoothAdapter mBluetoothAdapter;      // BluetoothAdapter : Bluetooth処理で必要
    public BluetoothGatt mBluetoothGatt = null;  // Gattサービスの検索、キャラスタリスティックの読み書き
    private int mBleConnectDeviceType;
    private String mDeviceAddress = "";    // デバイスアドレス
    private BluetoothDevice mDevice;
    private int retryCount = 0;
    private static final int RETRY_MAX = 10;

    public BleManagerUtil(Context context, BleManagerUtilListener listener) {
        mContext = context;
        mListener = listener;
        mHandler = new Handler();
    }

    // BluetoothGattコールバック
    private final BluetoothGattCallback mGattcallback = new BluetoothGattCallback() {
        // 接続状態変更（connectGatt()の結果として呼ばれる。）
        @Override
        public void onConnectionStateChange(BluetoothGatt gatt, final int status, final int newState ) {
            Logger.i("onConnectionStateChange status = " + status + "  newState = " + newState);
            if (newState == BluetoothProfile.STATE_CONNECTED) {
                gatt.discoverServices();
                return;
            }
            // デバイスと接続されていない場合のメッセージコード：133, 62
            // デバイスと離れて応答がなく、タイムアウトになる場合：8
            // デバイスと接続が切れた場合のメッセージコード：19
            if (status == 133 || status == 62 || status == 8 || status == 19) {  // 接続失敗
//                if (status == 133) {
//                    if (retryCount < RETRY_MAX) {
//                        retryCount++;
//                        new Handler(Looper.getMainLooper()).postDelayed(new Runnable() {
//                            @Override
//                            public void run() {
//                                mBluetoothGatt = mDevice.connectGatt(mContext, false, mGattcallback, TRANSPORT_LE);
//                            }
//                        }, 1500);
//                    }
//                }
                ((Activity)mContext).runOnUiThread( new Runnable() {
                    @Override
                    public void run() {
                        if (mListener != null) {
                            mListener.onConnectionError(status);
                        }
                        Logger.e(TAG,"onConnectionStateChange status = " + status);
                    }
                });
                gatt.disconnect();
                gatt = null;
                return;
            }

            if( status == BluetoothGatt.GATT_SUCCESS && BluetoothProfile.STATE_CONNECTED == newState ) {
                new Handler(Looper.getMainLooper()).postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        // 接続完了
                        boolean isGattaNull = false;
                        boolean isNotDiscoverService = false;
                        if (mBluetoothGatt != null) {
                            if (!mBluetoothGatt.discoverServices()) {
                                isNotDiscoverService = true;
                            }
                        }
                        if (mBluetoothGatt == null) {
                            isGattaNull = true;
                        }
                        if (isGattaNull || isNotDiscoverService) {
                            ((Activity)mContext).runOnUiThread( new Runnable() {  // 接続失敗
                                @Override
                                public void run() {
                                    if (mListener != null) {
                                        mListener.onGetDeviceInfoFailed(status);
                                    }
                                    Logger.e("onConnectionStateChange status = " + status);
                                }
                            });
                        }
                    }
                }, 600);
                return;
            }

            if( BluetoothProfile.STATE_DISCONNECTED == newState ) {    // 切断完了（接続可能範囲から外れて切断された）
                Logger.w(TAG,"BluetoothProfile.STATE_DISCONNECTED");
                //Logger.e(TAG,"out of range");
                // 切断が発生する場合、Bluetoothと接続を切断する。
                if (mListener != null) {
                    mListener.onConnectionError(status);
                }
            }
        }

        // サービス検索が完了したときの処理（mBluetoothGatt.discoverServices()の結果として呼ばれる。）
        @Override
        public void onServicesDiscovered(BluetoothGatt gatt, int status) {
            try {
                Logger.i(TAG, "onServicesDiscovered status = %s", status);

                if (BluetoothGatt.GATT_SUCCESS != status) {
                    return;
                }

                for (BluetoothGattService svc : gatt.getServices()){
                    Logger.d(TAG, svc.getUuid().toString());
                }

                //mBluetoothGatt = gatt;
                //mBluetoothGatt.requestConnectionPriority(BluetoothGatt.CONNECTION_PRIORITY_HIGH);
                if (mListener != null) {
                    mListener.onServicesDiscovered(gatt, status);
                }
            } catch (Exception e) {
                Logger.d(TAG,e);
            }
        }

        // キャラクタリスティックが読み込まれたときの処理
        @Override
        public void onCharacteristicRead(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic, int status ) {
            try {
                Logger.d(TAG, "onCharacteristicRead");
                if (BluetoothGatt.GATT_SUCCESS != status)
                {
                    return;
                }
            } catch (Exception e) {
                Logger.d(TAG,"null");
            }
        }

        // キャラクタリスティック変更が通知されたときの処理
        @Override
        public void onCharacteristicChanged(final BluetoothGatt gatt, final BluetoothGattCharacteristic characteristic ) {
            Logger.d(TAG,"onCharacteristicChanged");
            if (mListener != null) {
                mListener.onCharacteristicChanged(gatt, characteristic);
            }
        }

        @Override
        public void onDescriptorWrite(BluetoothGatt gatt, BluetoothGattDescriptor descriptor, int status){
            Logger.d(TAG,"onDescriptorWrite");
            if (mListener != null) {
                mListener.onDescriptorWrite(gatt, descriptor, status);
            }
        }
    };

    /**
     * 接続してるデバイスタイプを取得
     * @return
     */
    public int getBluetoothDeviceType() {
        return mBleConnectDeviceType;
    }

    // スキャンコールバック
    private final ScanCallback mScanCallback = new ScanCallback() {
        @Override
        public void onScanResult(int callbackType, final ScanResult result) {
            super.onScanResult(callbackType, result);
            ((Activity)mContext).runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    BluetoothDevice device = result.getDevice();
                    Logger.d(TAG, "--scaning : " + device.getName() );
                    if (device.getAddress().equals(mDeviceAddress)) {
                        // 異常と見做し、stopScanでエラーを返す。
                        stopScan(device.getAddress());
                    }
                }
            });
        }

        // スキャンに失敗
        @Override
        public void onScanFailed(final int errorCode) {
            super.onScanFailed(errorCode);
            Logger.e(TAG, "scan failed errorCode : " + errorCode);
            ((Activity)mContext).runOnUiThread( new Runnable() {
                @Override
                public void run() {
                    if (mListener != null) {
                        mListener.onConnectionError(errorCode);
                    }
                }
            });
        }
    };

    /**
     * スキャンを中止
     */
    private void stopScan(String deviceAddress) {
        Logger.d(TAG, "stop ble scan");

        // 一定期間後にスキャン停止するためのHandlerのRunnableの削除
        mHandler.removeCallbacksAndMessages(null);

        if (StringUtil.isNullOrEmpty(deviceAddress)) {
            // deviceAddressが見つからなかったと見做し、エラーで返す。
            mListener.onConnectionError(-1);
            return;
        }

        // スキャン停止
        mBluetoothAdapter.getBluetoothLeScanner().stopScan(mScanCallback);

        // ble接続
        BluetoothDevice device = mBluetoothAdapter.getRemoteDevice(deviceAddress);
        bleGattConnect(device);
    }

    // 接続
    public void connect(int connectTargetDeviceType, String deviceAddress) {
        mDeviceAddress = deviceAddress;
        // デバイスタイプセット
        mBleConnectDeviceType = connectTargetDeviceType;
        if(StringUtil.isNullOrEmpty(mDeviceAddress)) {  // DeviceAddressが空の場合は処理しない
            return;
        }

        if(mBluetoothGatt != null) {  // mBluetoothGattがnullでないなら接続済みか、接続中。
            return;
        }

        // mBluetoothGattのサービスと接続
        mDevice = mBluetoothAdapter.getRemoteDevice(deviceAddress);
        if (mDevice.getName() == null) {
            // deviceの名称がない場合、接続エラーになるため、再スキャンすることで検知したらgetRemoteDeviceメソッドに名称がセットされる
            mBluetoothAdapter.getBluetoothLeScanner().startScan(mScanCallback);
            // スキャン開始（一定時間後にスキャン停止する）
            mHandler.postDelayed( new Runnable() {
                @Override
                public void run() {
                    // 20秒後に呼ばれた時はスキャンで端末を取得できなかったと見做す。
                    stopScan(null);
                    Logger.d(TAG, "scan in 20 sec");
                }
            }, 20000 );
        } else {
            bleGattConnect(mDevice);
        }
    }

    /**
     * GATT BLE接続処理
     * @param device
     */
    private void bleGattConnect(BluetoothDevice device) {
        // GATT BLEを利用する時Androidのバージョン「23」をチェックしてGATTと接続する。
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            mBluetoothGatt = device.connectGatt(mContext, false, mGattcallback, TRANSPORT_LE);
        } else {
            mBluetoothGatt = device.connectGatt(mContext, false, mGattcallback );
        }
        // 該当デバイスのキャッシュをリフレッシュ
        refreshDeviceCache(mBluetoothGatt);
    }

    // 切断
    public void disconnect() {
        Logger.d(TAG,"disconnect");

        if(mBluetoothGatt == null) {
            Logger.d(TAG,"already gatt is null");
            return;
        }

        // 切断
        //   mBluetoothGatt.disconnect()ではなく、mBluetoothGatt.close()しオブジェクトを解放する。
        //   理由：「ユーザーの意思による切断」と「接続範囲から外れた切断」を区別するため。
        //   ①「ユーザーの意思による切断」は、mBluetoothGattオブジェクトを解放する。再接続は、オブジェクト構築から。
        //   ②「接続可能範囲から外れた切断」は、内部処理でmBluetoothGatt.disconnect()処理が実施される。
        //     切断時のコールバックでmBluetoothGatt.connect()を呼んでおくと、接続可能範囲に入ったら自動接続する。
        mBluetoothGatt.disconnect();
        //mBluetoothGatt.close();
        mBluetoothGatt = null;

        mListener = null;
    }

    // キャラクタリスティックの読み込み
    private void readCharacteristic( UUID uuid_service, UUID uuid_characteristic ) {
        if( null == mBluetoothGatt ) {
            return;
        }
        BluetoothGattCharacteristic blechar = mBluetoothGatt.getService( uuid_service ).getCharacteristic( uuid_characteristic );
        mBluetoothGatt.readCharacteristic( blechar );
    }

    // キャラクタリスティック通知の設定
    private void setCharacteristicNotification( UUID uuid_service, UUID uuid_characteristic, boolean enable ) {
        if( null == mBluetoothGatt ) {
            return;
        }

        BluetoothGattService service = mBluetoothGatt.getService( uuid_service );

        if(service == null) {
            return;
        }

        BluetoothGattCharacteristic blechar = service.getCharacteristic( uuid_characteristic );
        mBluetoothGatt.setCharacteristicNotification( blechar, enable );
        BluetoothGattDescriptor descriptor = blechar.getDescriptor(UUID.fromString("00002902-0000-1000-8000-00805f9b34fb"));
        descriptor.setValue( BluetoothGattDescriptor.ENABLE_INDICATION_VALUE );
        mBluetoothGatt.writeDescriptor(descriptor);
    }

    // Signed short (16-bit) Two's complement to short, String
    public String byteToString(byte[] bytes) {
        short sht = ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).getShort();
        float flt = sht / (float)100;

        return String.valueOf(flt);
    }

    /**
     * Bluetoothアダプタの取得処理
     * @return Android端末がBluetooth端末をサポートしていない場合 false
     */
    public boolean startDeviceInfo() {
        // Bluetoothアダプタの取得
        BluetoothManager bleMgr = (BluetoothManager) mContext.getSystemService( Context.BLUETOOTH_SERVICE );
        if (bleMgr != null) {
            mBluetoothAdapter = bleMgr.getAdapter();
            if( mBluetoothAdapter == null ) {    // Android端末がBluetoothをサポートしていない
                //Toast.makeText( mContext, R.string.bluetooth_is_not_supported, Toast.LENGTH_SHORT ).show();
                return false;
            }
        } else {
            Logger.e(TAG, "bluetoothManager is null");
            return false;
        }
        return true;
    }

    /**
     * bleGattのcache更新
     * @param gatt
     * @return
     */
    private boolean refreshDeviceCache(BluetoothGatt gatt){
        try {
            BluetoothGatt localBluetoothGatt = gatt;
            Method localMethod = localBluetoothGatt.getClass().getMethod("refresh", new Class[0]);
            if (localMethod != null) {
                boolean bool = ((Boolean) localMethod.invoke(localBluetoothGatt, new Object[0])).booleanValue();
                return bool;
            }
        }
        catch (Exception localException) {
            Logger.e(TAG, "An exception occured while refreshing device");
        }
        return false;
    }

    /**
     * 再接続処理
     */
    private void bleGattReconnect() {
        disconnect();
        connect(mBleConnectDeviceType, mDeviceAddress);
    }

    // @@@@@
    public void DescriptorDevice(BluetoothGattCharacteristic uRxChar, UUID ucccd) {
        if (mBluetoothGatt != null){
            mBluetoothGatt.setCharacteristicNotification(uRxChar,true);
            BluetoothGattDescriptor descriptor = uRxChar.getDescriptor(ucccd);
            descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
            mBluetoothGatt.writeDescriptor(descriptor);
        }
    }

    public void NotificationDisable(BluetoothGattCharacteristic uRxChar, UUID ucccd) {
        if (mBluetoothGatt != null){
            mBluetoothGatt.setCharacteristicNotification(uRxChar,true);
            BluetoothGattDescriptor descriptor = uRxChar.getDescriptor(ucccd);
            descriptor.setValue(BluetoothGattDescriptor.DISABLE_NOTIFICATION_VALUE);
            mBluetoothGatt.writeDescriptor(descriptor);
        }
    }
    public boolean readPhyConnection(){
        if (android.os.Build.VERSION.SDK_INT < Build.VERSION_CODES.O){
            return false;
        }
        if (mBluetoothGatt == null){
            return false;
        }
        mBluetoothGatt.readPhy();
        return true;
    }

    public BluetoothGatt getBluetoothGatt() {
        return mBluetoothGatt;
    }
    public void setBluetoothGatt(BluetoothGatt gatt) {
        mBluetoothGatt = gatt;
    }

    /**
     * リスナー設定
     * @param listener リスナー
     */
    public void setListener(BleManagerUtilListener listener) {
        mListener = listener;
    }
}
