package jp.agentec.abook.abv.bl.data.dao;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.agentec.abook.abv.bl.common.db.Cursor;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.dto.ShopMemberDto;
import jp.agentec.adf.util.CollectionUtil;
import jp.agentec.adf.util.DateTimeUtil;
import jp.agentec.adf.util.StringUtil;

public class ShopMemberDao extends AbstractCommunicationDao {

	/**
	 * {@link ShopMemberDao} のインスタンスを初期化します。
	 * アンドロイドの android.content.Context のインスタンス
	 * @throws ClassCastException 引数のcontextが android.content.Context 又は、その継承クラスではありません。
	 * @since 1.0.0
	 */
	/*package*/ ShopMemberDao() {
	}

	@Override
	protected ShopMemberDto convert(Cursor cursor) {
		ShopMemberDto dto = new ShopMemberDto();

		int column = cursor.getColumnIndex("shop_member_id");
		if (column != -1) {
			dto.shopMemberId = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("shop_member_name");
		if (column != -1) {
			dto.shopMemberName = cursor.getString(column);
		}
		column = cursor.getColumnIndex("login_id");
		if (column != -1) {
			dto.loginId = cursor.getString(column);
		}
		column = cursor.getColumnIndex("profile_url");
		if (column != -1) {
			dto.profileUrl = cursor.getString(column);
		}
		column = cursor.getColumnIndex("favorite_register_date");
		if (column != -1) {
			dto.favoriteRegisterDate = cursor.getString(column);
		}
		column = cursor.getColumnIndex("self_flg");
		if (column != -1) {
			dto.selfFlg = cursor.getInt(column);
		}
		return dto;
	}

	public List<ShopMemberDto> getAllShopMember() {
		List<ShopMemberDto> list = rawQueryGetDtoList("SELECT * FROM m_shop_member", null, ShopMemberDto.class);
		return list;
	}

	public ShopMemberDto getShopMember(int shopMemberId) {
		return rawQueryGetDto("SELECT * FROM m_shop_member WHERE shop_member_id=?", new String[]{""+ shopMemberId}, ShopMemberDto.class);
	}

	public void insertShopMember(ShopMemberDto dto) {
		try {
			beginTransaction();
			insert("INSERT OR REPLACE INTO m_shop_member (shop_member_id, shop_member_name, login_id, profile_url, self_flg, favorite_register_date) VALUES (?,?,?,?,?,(SELECT favorite_register_date FROM m_shop_member WHERE shop_member_id = ?))", dto.getInsertValues());
			if (dto.groupIdList != null && dto.groupIdList.size() > 0) {
				update(" DELETE FROM r_shop_member_group WHERE shop_member_id = ? ", new Integer[]{dto.shopMemberId});
				for (Integer groupId : dto.groupIdList) {
					insert("INSERT OR REPLACE INTO r_shop_member_group (shop_member_id, group_id) VALUES ("+dto.shopMemberId+",?)", new Integer[]{groupId});
				}
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertShopMember failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}

	public void insertShopMember(List<ShopMemberDto> dtoList) {
		try {
			beginTransaction();
			for (ShopMemberDto dto : dtoList) {
				insert("INSERT OR REPLACE INTO m_shop_member (shop_member_id, shop_member_name, login_id, profile_url, self_flg, favorite_register_date) VALUES (?,?,?,?,?, (SELECT favorite_register_date FROM m_shop_member WHERE shop_member_id = ?))", dto.getInsertValues());
				if (dto.groupIdList == null) { continue; }
				for (Integer groupId : dto.groupIdList) {
					insert("INSERT OR REPLACE INTO r_shop_member_group (shop_member_id, group_id) VALUES ("+dto.shopMemberId+",?)", new Integer[]{groupId});
				}
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertShopMember failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}


	public void insertShopMemberGroupRelation(Integer groupId, List<ShopMemberDto> dtoList) {
		try {
			update("DELETE FROM r_shop_member_group WHERE group_id = ? ", new Object[] { groupId });
			for (ShopMemberDto dto : dtoList) {
				insert("INSERT OR REPLACE INTO r_shop_member_group (shop_member_id, group_id) VALUES ("+dto.shopMemberId+",?)", new Integer[]{groupId});
			}
		} catch (Exception e) {
			rollback();
			Logger.e("insertShopMember failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}

	public ShopMemberDto getMyInfo() {
		return rawQueryGetDto("SELECT * FROM m_shop_member WHERE self_flg = 1", null, ShopMemberDto.class);
	}

	public void resetSelfFlg() {
		update("UPDATE m_shop_member SET self_flg = 0", new String[]{});
	}

	public List<ShopMemberDto> getUserInRoom(Integer chatRoomId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT sm.* ");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_chat_room_shop_member AS rrm ");
		sql.append("   ON sm.shop_member_id = rrm.shop_member_id ");
		sql.append(" WHERE rrm.chat_room_id =?");
		sql.append(" AND sm.self_flg != 1");
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{""+ chatRoomId}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getSelectedUsers(String userIdList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" WHERE sm.shop_member_id IN ( "+userIdList+" )");
		sql.append(" AND sm.self_flg != 1");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserList(String userIdList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" WHERE sm.shop_member_id IN ( "+userIdList+" )");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserListByLoginId(String[] loginIdList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.login_id ");
		sql.append("      ,sm.profile_url");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" WHERE ");
		sql.append("      sm.login_id IN (");

		ArrayList<String> whereSqlList = new ArrayList<String>();
		for (String loginId : loginIdList) {
			if (StringUtil.isNullOrEmpty(loginId)){
				continue;
			}
			String whereSql = "'"+loginId+"' ";
			whereSqlList.add(whereSql);
		}
		if (CollectionUtil.isNotEmpty(whereSqlList)) {
			sql.append(StringUtil.join(" , ", whereSqlList) + ")");
		}
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getfavoriteUserList() {
		return rawQueryGetDtoList("SELECT * FROM m_shop_member WHERE favorite_register_date NOT NULL", null, ShopMemberDto.class);
	}

	public List<ShopMemberDto> getfavoriteUserListNotInRoom(Integer roomId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT * ");
		sql.append(" FROM m_shop_member sm ");
		sql.append(" WHERE favorite_register_date NOT NULL ");
		sql.append(" AND sm.shop_member_id NOT IN");
		sql.append("     (SELECT shop_member_id FROM r_chat_room_shop_member WHERE chat_room_id = ?)");
		return rawQueryGetDtoList(sql.toString(), new String[] {"" + roomId}, ShopMemberDto.class);
	}

	public List<ShopMemberDto> getUserListByGroupId(Integer groupId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append("      ,sm.favorite_register_date");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append(" WHERE rsmg.group_id =?");
		sql.append(" AND sm.self_flg != 1");
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{""+ groupId}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserListNotInRoomByGroupId(Integer groupId, Integer roomId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append("      ,sm.favorite_register_date");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append(" WHERE rsmg.group_id =?");
		sql.append(" AND sm.self_flg != 1");
		sql.append(" AND sm.shop_member_id NOT IN (SELECT shop_member_id FROM r_chat_room_shop_member WHERE chat_room_id = ?)");
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{"" + groupId, "" + roomId}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserInMyGroupByName(String[] keywords) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT DISTINCT");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append("      ,sm.favorite_register_date");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append("WHERE rsmg.group_id IN (SELECT group_id FROM m_shop_member sm INNER JOIN r_shop_member_group rmg ON sm.shop_member_id = rmg.shop_member_id WHERE sm.self_flg = 1) ");
		sql.append(" AND sm.self_flg != 1 ");
		ArrayList<String> whereSqlList = new ArrayList<String>();
		for (String keyword : keywords) {
			if (StringUtil.isNullOrEmpty(keyword)){
				continue;
			}
			String whereSql = " sm.shop_member_name LIKE '%"+keyword+"%' ";
			whereSqlList.add(whereSql);
		}
		if (CollectionUtil.isNotEmpty(whereSqlList)) {
			sql.append(" AND" + StringUtil.join(" AND ", whereSqlList));
		}
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserInMyGroupNotInRoomByName(String[] keywords, Integer roomId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT DISTINCT");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append("      ,sm.favorite_register_date");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append(" WHERE rsmg.group_id IN (SELECT group_id FROM m_shop_member sm INNER JOIN r_shop_member_group rmg on sm.shop_member_id = rmg.shop_member_id WHERE sm.self_flg = 1)");
		sql.append(" AND sm.self_flg != 1");
		sql.append(" AND sm.shop_member_id NOT IN (SELECT shop_member_id FROM r_chat_room_shop_member WHERE chat_room_id = ?)");
		ArrayList<String> whereSqlList = new ArrayList<String>();
		for (String keyword : keywords) {
			if (StringUtil.isNullOrEmpty(keyword)){
				continue;
			}
			String whereSql = " sm.shop_member_name LIKE '%"+keyword+"%' ";
			whereSqlList.add(whereSql);
		}
		if (CollectionUtil.isNotEmpty(whereSqlList)) {
			sql.append(" AND" + StringUtil.join(" AND ", whereSqlList));
		}
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{"" + roomId}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserInAllGroupByName(String[] keywords) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT DISTINCT");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append("      ,sm.favorite_register_date");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append(" AND sm.self_flg != 1");
		sql.append(" AND ");
		ArrayList<String> whereSqlList = new ArrayList<String>();
		for (String keyword : keywords) {
			String whereSql = "sm.shop_member_name LIKE '%"+keyword+"%'";
			whereSqlList.add(whereSql);
		}
		sql.append(StringUtil.join(" AND ", whereSqlList));
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getUserInAllGroupNotInRoomByName(String[] keywords, Integer roomId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT DISTINCT");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append("      ,sm.favorite_register_date");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append(" AND sm.self_flg != 1");
		ArrayList<String> whereSqlList = new ArrayList<String>();
		for (String keyword : keywords) {
			if (StringUtil.isNullOrEmpty(keyword)){
				continue;
			}
			String whereSql = " sm.shop_member_name LIKE '%"+keyword+"%' ";
			whereSqlList.add(whereSql);
		}
		if (CollectionUtil.isNotEmpty(whereSqlList)) {
			sql.append(" AND" + StringUtil.join(" AND ", whereSqlList));
		}
		sql.append(" AND sm.shop_member_id NOT IN (SELECT shop_member_id FROM r_chat_room_shop_member WHERE chat_room_id = ?) ");
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[] {"" + roomId}, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getMyGroupUsers() {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("      sm.shop_member_id ");
		sql.append("      ,sm.shop_member_name ");
		sql.append("      ,sm.profile_url");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_shop_member_group AS rsmg ");
		sql.append("   ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append(" WHERE rsmg.group_id IN (SELECT group_id FROM m_shop_member INNER JOIN r_shop_member_group ON m_shop_member.shop_member_id = r_shop_member_group.shop_member_id WHERE self_flg = 1 GROUP BY group_id)");
		sql.append(" AND sm.self_flg != 1 ");
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), null, ShopMemberDto.class);
		return list;
	}

	public List<ShopMemberDto> getCollaborationMemberList(Integer collaborationDetailId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT sm.* ");
		sql.append(" FROM ");
		sql.append("      m_shop_member AS sm ");
		sql.append(" INNER JOIN r_chat_collaboration_member AS rccm ");
		sql.append("   ON sm.shop_member_id = rccm.shop_member_id ");
		sql.append(" WHERE rccm.chat_collaboration_detail_id =?");
		sql.append(" ORDER BY sm.shop_member_name DESC ");
		List<ShopMemberDto> list = rawQueryGetDtoList(sql.toString(), new String[]{""+ collaborationDetailId}, ShopMemberDto.class);
		return list;
	}

	public boolean updateShopMember(ShopMemberDto dto) {
		long count = update("UPDATE m_shop_member SET shop_member_name=?, profile_url=? WHERE shop_member_id=?", dto.getUpdateValues());
		return count > 0;
	}

	public boolean deleteShopMember(ShopMemberDto dto) {
		long count = update("DELETE FROM m_shop_member WHERE shop_member_id=?", new String[] {"" + dto.shopMemberId});
		return count > 0;
	}

	public void deleteShopMemberByList(List<ShopMemberDto> dtoList) {
		try {
			beginTransaction();
			for (ShopMemberDto dto : dtoList) {
				update("DELETE FROM r_shop_member_group WHERE shop_member_id=?", new String[] {"" + dto.shopMemberId});
				update("DELETE FROM r_chat_room_shop_member WHERE shop_member_id=?", new String[] {"" + dto.shopMemberId});
				update("DELETE FROM m_shop_member WHERE shop_member_id=?", new String[] {"" + dto.shopMemberId});
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertShopMember failed.", e);
			throw new RuntimeException(e);
		} finally {

		}
	}

	public boolean updateFavoriteUser(ShopMemberDto dto) {
		long count = update("UPDATE m_shop_member SET favorite_register_date=? WHERE shop_member_id=?", new Object[]{dto.favoriteRegisterDate, dto.shopMemberId});
		return count > 0;
	}

	public void updateFavoriteUserList(List<Integer> shopMemberIdList) {
		beginTransaction();
		for (Integer shopMemberId : shopMemberIdList ) {
			ShopMemberDto favoriteUserDto= new ShopMemberDto();
			favoriteUserDto.shopMemberId = shopMemberId;
			favoriteUserDto.favoriteRegisterDate = DateTimeUtil.toString_yyyyMMddHHmmss_none(new Date(System.currentTimeMillis()));
			update("UPDATE m_shop_member SET favorite_register_date=? WHERE shop_member_id=?", new Object[]{favoriteUserDto.favoriteRegisterDate, favoriteUserDto.shopMemberId});
		}
		commit();
	}

	public void deleteShopMember() {
		try {
			beginTransaction();
			delete("r_collaboration_member", null, null);
			delete("m_shop_member", null, null);
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("deleteShopMember failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}

}
