package jp.agentec.abook.abv.bl.logic;

import static jp.agentec.abook.abv.bl.acms.type.OperationType.DRAWING;
import static jp.agentec.abook.abv.bl.acms.type.OperationType.PANO;
import static jp.agentec.abook.abv.bl.acms.type.OperationType.PDF;

import net.lingala.zip4j.exception.ZipException;

import org.json.adf.JSONArray;
import org.json.adf.JSONException;
import org.json.adf.JSONObject;

import java.io.File;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import jp.agentec.abook.abv.bl.acms.client.AcmsClient;
import jp.agentec.abook.abv.bl.acms.client.json.ChatPushDataJSON;
import jp.agentec.abook.abv.bl.acms.client.json.OperationDataJSON;
import jp.agentec.abook.abv.bl.acms.client.json.OperationListJSON;
import jp.agentec.abook.abv.bl.acms.client.json.ProcessDataJSON;
import jp.agentec.abook.abv.bl.acms.client.json.SceneEntryJSON;
import jp.agentec.abook.abv.bl.acms.client.json.WorkerGroupJSON;
import jp.agentec.abook.abv.bl.acms.client.json.content.PhaseInfoJSON;
import jp.agentec.abook.abv.bl.acms.client.parameters.AcmsParameters;
import jp.agentec.abook.abv.bl.acms.client.parameters.DeleteProcessParameters;
import jp.agentec.abook.abv.bl.acms.client.parameters.GetOperationDataParameters;
import jp.agentec.abook.abv.bl.acms.client.parameters.GetTaskFileParameters;
import jp.agentec.abook.abv.bl.acms.type.OperationSortingType;
import jp.agentec.abook.abv.bl.common.ABVEnvironment;
import jp.agentec.abook.abv.bl.common.CommonExecutor;
import jp.agentec.abook.abv.bl.common.Constant;
import jp.agentec.abook.abv.bl.common.Callback;
import jp.agentec.abook.abv.bl.common.constant.ABookKeys;
import jp.agentec.abook.abv.bl.common.exception.ABVException;
import jp.agentec.abook.abv.bl.common.exception.ABVExceptionCode;
import jp.agentec.abook.abv.bl.common.exception.AcmsException;
import jp.agentec.abook.abv.bl.common.exception.NetworkDisconnectedException;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.common.util.SecurityUtil;
import jp.agentec.abook.abv.bl.data.ABVDataCache;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.ContentDao;
import jp.agentec.abook.abv.bl.data.dao.OperationContentDao;
import jp.agentec.abook.abv.bl.data.dao.OperationDao;
import jp.agentec.abook.abv.bl.data.dao.OperationGroupMasterOperationDao;
import jp.agentec.abook.abv.bl.data.dao.PushMessageDao;
import jp.agentec.abook.abv.bl.data.dao.TaskDao;
import jp.agentec.abook.abv.bl.data.dao.TaskReportApprovalDao;
import jp.agentec.abook.abv.bl.data.dao.TaskReportDao;
import jp.agentec.abook.abv.bl.data.dao.TaskReportItemsDao;
import jp.agentec.abook.abv.bl.data.dao.TaskReportSendDao;
import jp.agentec.abook.abv.bl.data.dao.TaskWorkerGroupDao;
import jp.agentec.abook.abv.bl.data.dao.WorkerGroupDao;
import jp.agentec.abook.abv.bl.dto.CategoryContentDto;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.abook.abv.bl.dto.OperationContentDto;
import jp.agentec.abook.abv.bl.dto.OperationDto;
import jp.agentec.abook.abv.bl.dto.OperationGroupMasterDto;
import jp.agentec.abook.abv.bl.dto.OperationGroupMasterRelationDto;
import jp.agentec.abook.abv.bl.dto.PushMessageDto;
import jp.agentec.abook.abv.bl.dto.TaskDto;
import jp.agentec.abook.abv.bl.dto.TaskReportApprovalDto;
import jp.agentec.abook.abv.bl.dto.TaskReportDto;
import jp.agentec.abook.abv.bl.dto.TaskReportItemsDto;
import jp.agentec.abook.abv.bl.dto.TaskReportSendDto;
import jp.agentec.abook.abv.bl.dto.TaskWorkerGroupDto;
import jp.agentec.abook.abv.bl.dto.WorkerGroupDto;
import jp.agentec.adf.util.DateTimeFormat;
import jp.agentec.adf.util.DateTimeUtil;
import jp.agentec.adf.util.FileUtil;
import jp.agentec.adf.util.StringUtil;

/**
 * Created by leej on 2018/08/17.
 */

public class OperationLogic extends AbstractLogic {
    private static final String TAG = "OperationLogic";
    private OperationDao mOperationDao = AbstractDao.getDao(OperationDao.class);
    private OperationContentDao mOperationContentDao = AbstractDao.getDao(OperationContentDao.class);
    private ContentDao mContentDao = AbstractDao.getDao(ContentDao.class);

    private WorkerGroupDao mWorkerGroupDao = AbstractDao.getDao(WorkerGroupDao.class);
    private TaskDao mTaskDao = AbstractDao.getDao(TaskDao.class);
    private TaskReportDao mTaskReportDao = AbstractDao.getDao(TaskReportDao.class);
    private TaskReportSendDao mTaskReportSendDao = AbstractDao.getDao(TaskReportSendDao.class);
    private TaskReportItemsDao mTaskReportItemsDao = AbstractDao.getDao(TaskReportItemsDao.class);
    private OperationGroupMasterOperationDao mOperationGroupMasterOperationDao = AbstractDao.getDao(OperationGroupMasterOperationDao.class);

    private ContentLogic mContentLogic = AbstractLogic.getLogic(ContentLogic.class);

    private OperationGroupMasterLogic mOperationGroupMasterLogic = AbstractLogic.getLogic(OperationGroupMasterLogic.class);

    private PushMessageDao mPushMessageDao = AbstractDao.getDao(PushMessageDao.class);

    private TaskWorkerGroupDao mTaskWorkerGroupDao = AbstractDao.getDao(TaskWorkerGroupDao.class);
    private TaskReportApprovalDao mTaskReportApprovalDao = AbstractDao.getDao(TaskReportApprovalDao.class);

    private GroupLogic mGroupLogic = AbstractLogic.getLogic(GroupLogic.class);

    /**
     * 作業に関連する情報取得（API通信で取得して、DB保存処理）
     *
     * @throws AcmsException
     * @throws NetworkDisconnectedException
     */
    public void initializeOperations() throws AcmsException, NetworkDisconnectedException, IOException {
        // 作業グループリスト取得
        setWorkingGroupList();
        // 作業一覧取得し、登録・更新・削除する
        retrieveServerOperation();
    }

    public void initializeOperations(Callback mCallBack) throws AcmsException, NetworkDisconnectedException {
        // 作業グループリスト取得
        setWorkingGroupList();
        mCallBack.callback(2);
        // 作業一覧取得し、登録・更新・削除する
        retrieveServerOperation(mCallBack);
    }

    /**
     * 作業一覧取得し、登録・更新・削除する
     *
     * @throws AcmsException
     * @throws NetworkDisconnectedException
     */
    private void retrieveServerOperation() throws AcmsException, NetworkDisconnectedException, IOException {
        List<OperationDto> localOperations = mOperationDao.getAllOperation();
        AcmsParameters param = new AcmsParameters(cache.getMemberInfo().sid);
        OperationListJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getOperationList(param);
        List<OperationDto> serverOperations = json.operationList;
        //サーバーからチャットプシュデータを取得
        if (ABVDataCache.getInstance().serviceOption.isChat()) {
            ChatPushDataJSON chatPushJson = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getChatPushList(param);
            for (PushMessageDto dto : chatPushJson.pushMessageList) {
                PushMessageDto dtoTemp = mPushMessageDao.selectChat(dto.pushMessageId, dto.operationId, dto.pushSendDate);
                if (dtoTemp == null) {
                    mPushMessageDao.insert(dto);
                }
            }
        }

        for (OperationDto serverOperationDto : serverOperations) {
            // 登録フラグ
            boolean insertFlg = true;
            for (OperationDto localOperationDto : localOperations) {
                if (serverOperationDto.operationId.equals(localOperationDto.operationId)) {
                    //update
                    serverOperationDto.needSyncFlg = serverOperationDto.equalsLastEdit(localOperationDto) ? localOperationDto.needSyncFlg : true;
                    List<Long> localOperationContentIds = mOperationContentDao.getContentIds(serverOperationDto.operationId);
                    for (OperationContentDto operationContentDto : serverOperationDto.operationContentDtoList) {
                        if (localOperationContentIds == null || localOperationContentIds.size() == 0) {
                            // insert
                            mOperationContentDao.insertOperationContent(operationContentDto);
                        } else {
                            int localOperationContentIndex = localOperationContentIds.indexOf(operationContentDto.contentId);
                            if (localOperationContentIndex >= 0) {
                                // 存在する
                                localOperationContentIds.remove(localOperationContentIndex);
                            } else {
                                // insert
                                mOperationContentDao.insertOperationContent(operationContentDto);
                            }
                        }
                    }
                    if (localOperationContentIds != null) {
                        // 関連資料の削除処理
                        for (Long deleteContentId : localOperationContentIds) {
                            mOperationContentDao.deleteOperationContent(localOperationDto.operationId, deleteContentId);
                        }
                    }

                    if (mOperationContentDao.isExistMainOperationContent(serverOperationDto.operationId)) {
                        serverOperationDto.contentCreatingFlg = false;
                    }

                    // ローカルにある作業情報に紐づく全作業IDを取得
                    List<Integer> localOperationGroupMasterIds = mOperationGroupMasterOperationDao.getOperationGroupMasterIds(serverOperationDto.operationId);
                    for (OperationGroupMasterRelationDto operationGroupMasterRelationDto : serverOperationDto.operationGroupMasterRelationDtoList) {
                        if (localOperationGroupMasterIds == null || localOperationGroupMasterIds.size() == 0) {
                            // 作業IDが存在しなければ、登録する
                            mOperationGroupMasterOperationDao.insertOperationGroupMasterOperation(operationGroupMasterRelationDto);
                        } else {
                            // ローカル（DB）に作業IDが存在するかチェックして更新・登録を判定
                            int localOperationContentIndex = localOperationGroupMasterIds.indexOf(operationGroupMasterRelationDto.operationGroupMasterId);
                            if (localOperationContentIndex >= 0) {
                                // 作業IDが存在するので更新
                                localOperationGroupMasterIds.remove(localOperationContentIndex);
                            } else {
                                // 作業IDが存在しないので登録
                                mOperationGroupMasterOperationDao.insertOperationGroupMasterOperation(operationGroupMasterRelationDto);
                            }
                        }
                    }
                    if (localOperationGroupMasterIds != null) {
                        // ローカルとサーバーの差分は削除と見做し、削除処理を行う。
                        for (Integer deleteOperationGroupMasterId : localOperationGroupMasterIds) {
                            mOperationGroupMasterOperationDao.deleteOperationGroupMasterOperation(deleteOperationGroupMasterId, serverOperationDto.operationId);
                        }
                    }

                    mOperationDao.update(serverOperationDto);
                    // 更新することで登録フラグをfalseにセット
                    insertFlg = false;
                    break;
                }
            }

            // 更新されたら以下の処理は行わない
            if (insertFlg) {
                boolean isNotExistContent = false;
                for (OperationContentDto operationContentDto : serverOperationDto.operationContentDtoList) {
                    if (operationContentDto.operationContentFlg) {
                        if (mContentDao.getContent(operationContentDto.contentId) == null) {
                            isNotExistContent = true;
                        }
                    }
                }
                // コンテンツテーブルに存在しない場合、プロジェクトを登録しない
                if (isNotExistContent) {
                    continue;
                }

                serverOperationDto.needSyncFlg = true;
                serverOperationDto.contentCreatingFlg = false;
                mOperationDao.insert(serverOperationDto);
                // 関連資料の登録
                for (OperationContentDto operationContentDto : serverOperationDto.operationContentDtoList) {
                    mOperationContentDao.insertOperationContent(operationContentDto);
                }

                // 作業種別・作業のリレーションテーブルにデータ登録
                for (OperationGroupMasterRelationDto operationGroupMasterRelationDto : serverOperationDto.operationGroupMasterRelationDtoList) {
                    mOperationGroupMasterOperationDao.insertOperationGroupMasterOperation(operationGroupMasterRelationDto);
                }
            }

            // 作業担当グループ登録
            mTaskWorkerGroupDao.deleteByOperationId(serverOperationDto.operationId);
            for (TaskWorkerGroupDto taskWorkerGroupDto : serverOperationDto.taskWorkerGroupDtoList) {
                taskWorkerGroupDto.operationId = serverOperationDto.operationId;
                for (Integer groupId : taskWorkerGroupDto.groupList) {
                    taskWorkerGroupDto.groupId = groupId;
                    mTaskWorkerGroupDao.insert(taskWorkerGroupDto);
                }
            }

            for (PushMessageDto dto : serverOperationDto.pushMessageList) {
                mPushMessageDao.insert(dto);
            }

            //save processList, warningReport json file
            String processListJsonPath = ABVEnvironment.getInstance().getPickupProcessListOperationFilePath(serverOperationDto.operationId);
            FileUtil.delete(processListJsonPath);
            if (serverOperationDto.processJsonArray != null && serverOperationDto.processJsonArray.length() > 0) {
                FileUtil.createFile(processListJsonPath, serverOperationDto.processJsonArray.toString());
            }
            String warningReportJsonPath = ABVEnvironment.getInstance().getPickupWarningReportListOperationFilePath(serverOperationDto.operationId);
            FileUtil.delete(warningReportJsonPath);
            if (serverOperationDto.warningReportJsonArray != null && serverOperationDto.warningReportJsonArray.length() > 0) {
                FileUtil.createFile(warningReportJsonPath, serverOperationDto.warningReportJsonArray.toString());
            }
        }

        List<OperationDto> deleteListDtoArray = new ArrayList<OperationDto>();

        for (OperationDto localProjcetDto : localOperations) {
            boolean deleteFlg = true;
            for (OperationDto serverOperationDto : serverOperations) {
                if (localProjcetDto.operationId.equals(serverOperationDto.operationId)) {
                    deleteFlg = false;
                    break;
                }
            }
            if (deleteFlg) {
                deleteListDtoArray.add(localProjcetDto);
            }
        }
        // サーバーから取得した作業情報がローカルに存在しないので削除する
        for (OperationDto deleteOperationDto : deleteListDtoArray) {
            List<TaskDto> taskDtoList = mTaskDao.selectAllTaskByOperationId(deleteOperationDto.operationId);
            List<Long> contentIds = mOperationContentDao.getContentIds(deleteOperationDto.operationId);
            for (TaskDto taskDto : taskDtoList) {
                mTaskDao.delete(taskDto);
            }
            mPushMessageDao.delete(deleteOperationDto.operationId);
            mOperationDao.delete(deleteOperationDto);

            // 作業削除時、紐づいた資料を削除
            for (Long contentId : contentIds) {
                if (mOperationContentDao.isDeleteContent(contentId)) {
                    ContentDto contentDto = mContentDao.getContent(contentId);
                    mContentLogic.deleteContent(contentDto, true);
                }
            }

            String operationTaskDirPath = ABVEnvironment.getInstance().getOperationDirFile(deleteOperationDto.operationId);
            FileUtil.delete(operationTaskDirPath);
        }
    }

    private void retrieveServerOperation(Callback mCallBack) throws AcmsException, NetworkDisconnectedException {
        List<OperationDto> localOperations = mOperationDao.getAllOperation();
        AcmsParameters param = new AcmsParameters(cache.getMemberInfo().sid);
        OperationListJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getOperationList(param);
        List<OperationDto> serverOperations = json.operationList;
        //サーバーからチャットプシュデータを取得
        if (ABVDataCache.getInstance().serviceOption.isChat()) {
            ChatPushDataJSON chatPushJson = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getChatPushList(param);
            for (PushMessageDto dto : chatPushJson.pushMessageList) {
                PushMessageDto dtoTemp = mPushMessageDao.selectChat(dto.pushMessageId, dto.operationId, dto.pushSendDate);
                if (dtoTemp == null) {
                    mPushMessageDao.insert(dto);
                }
            }
        }
        for (OperationDto serverOperationDto : serverOperations) {
            // 登録フラグ
            boolean insertFlg = true;
            for (OperationDto localOperationDto : localOperations) {
                if (serverOperationDto.operationId.equals(localOperationDto.operationId)) {
                    //update
                    serverOperationDto.needSyncFlg = serverOperationDto.equalsLastEdit(localOperationDto) ? localOperationDto.needSyncFlg : true;
                    List<Long> localOperationContentIds = mOperationContentDao.getContentIds(serverOperationDto.operationId);
                    for (OperationContentDto operationContentDto : serverOperationDto.operationContentDtoList) {
                        if (localOperationContentIds == null || localOperationContentIds.size() == 0) {
                            // insert
                            mOperationContentDao.insertOperationContent(operationContentDto);
                        } else {
                            int localOperationContentIndex = localOperationContentIds.indexOf(operationContentDto.contentId);
                            if (localOperationContentIndex >= 0) {
                                // 存在する
                                localOperationContentIds.remove(localOperationContentIndex);
                            } else {
                                // insert
                                mOperationContentDao.insertOperationContent(operationContentDto);
                            }
                        }
                    }
                    if (localOperationContentIds != null) {
                        // 関連資料の削除処理
                        for (Long deleteContentId : localOperationContentIds) {
                            mOperationContentDao.deleteOperationContent(localOperationDto.operationId, deleteContentId);
                        }
                    }

                    if (mOperationContentDao.isExistMainOperationContent(serverOperationDto.operationId)) {
                        serverOperationDto.contentCreatingFlg = false;
                    }

                    // ローカルにある作業情報に紐づく全作業IDを取得
                    List<Integer> localOperationGroupMasterIds = mOperationGroupMasterOperationDao.getOperationGroupMasterIds(serverOperationDto.operationId);
                    for (OperationGroupMasterRelationDto operationGroupMasterRelationDto : serverOperationDto.operationGroupMasterRelationDtoList) {
                        if (localOperationGroupMasterIds == null || localOperationGroupMasterIds.size() == 0) {
                            // 作業IDが存在しなければ、登録する
                            mOperationGroupMasterOperationDao.insertOperationGroupMasterOperation(operationGroupMasterRelationDto);
                        } else {
                            // ローカル（DB）に作業IDが存在するかチェックして更新・登録を判定
                            int localOperationContentIndex = localOperationGroupMasterIds.indexOf(operationGroupMasterRelationDto.operationGroupMasterId);
                            if (localOperationContentIndex >= 0) {
                                // 作業IDが存在するので更新
                                localOperationGroupMasterIds.remove(localOperationContentIndex);
                            } else {
                                // 作業IDが存在しないので登録
                                mOperationGroupMasterOperationDao.insertOperationGroupMasterOperation(operationGroupMasterRelationDto);
                            }
                        }
                    }
                    if (localOperationGroupMasterIds != null) {
                        // ローカルとサーバーの差分は削除と見做し、削除処理を行う。
                        for (Integer deleteOperationGroupMasterId : localOperationGroupMasterIds) {
                            mOperationGroupMasterOperationDao.deleteOperationGroupMasterOperation(deleteOperationGroupMasterId, serverOperationDto.operationId);
                        }
                    }

                    mOperationDao.update(serverOperationDto);
                    // 更新することで登録フラグをfalseにセット
                    insertFlg = false;
                    break;
                }
            }

            // 更新されたら以下の処理は行わない
            if (insertFlg) {
                boolean isNotExistContent = false;
                for (OperationContentDto operationContentDto : serverOperationDto.operationContentDtoList) {
                    if (operationContentDto.operationContentFlg) {
                        if (mContentDao.getContent(operationContentDto.contentId) == null) {
                            isNotExistContent = true;
                        }
                    }
                }
                // コンテンツテーブルに存在しない場合、プロジェクトを登録しない
                if (isNotExistContent) {
                    continue;
                }

                serverOperationDto.needSyncFlg = true;
                serverOperationDto.contentCreatingFlg = false;
                mOperationDao.insert(serverOperationDto);
                // 関連資料の登録
                for (OperationContentDto operationContentDto : serverOperationDto.operationContentDtoList) {
                    mOperationContentDao.insertOperationContent(operationContentDto);
                }

                // 作業種別・作業のリレーションテーブルにデータ登録
                for (OperationGroupMasterRelationDto operationGroupMasterRelationDto : serverOperationDto.operationGroupMasterRelationDtoList) {
                    mOperationGroupMasterOperationDao.insertOperationGroupMasterOperation(operationGroupMasterRelationDto);
                }
            }

            // 作業担当グループ登録
            mTaskWorkerGroupDao.deleteByOperationId(serverOperationDto.operationId);
            for (TaskWorkerGroupDto taskWorkerGroupDto : serverOperationDto.taskWorkerGroupDtoList) {
                taskWorkerGroupDto.operationId = serverOperationDto.operationId;
                for (Integer groupId : taskWorkerGroupDto.groupList) {
                    taskWorkerGroupDto.groupId = groupId;
                    mTaskWorkerGroupDao.insert(taskWorkerGroupDto);
                }
            }

            for (PushMessageDto dto : serverOperationDto.pushMessageList) {
                mPushMessageDao.insert(dto);
            }
        }

        List<OperationDto> deleteListDtoArray = new ArrayList<OperationDto>();

        for (OperationDto localProjcetDto : localOperations) {
            boolean deleteFlg = true;
            for (OperationDto serverOperationDto : serverOperations) {
                if (localProjcetDto.operationId.equals(serverOperationDto.operationId)) {
                    deleteFlg = false;
                    break;
                }
            }
            if (deleteFlg) {
                deleteListDtoArray.add(localProjcetDto);
            }
        }
        // サーバーから取得した作業情報がローカルに存在しないので削除する
        for (OperationDto deleteOperationDto : deleteListDtoArray) {
            List<TaskDto> taskDtoList = mTaskDao.selectAllTaskByOperationId(deleteOperationDto.operationId);
            List<Long> contentIds = mOperationContentDao.getContentIds(deleteOperationDto.operationId);
            for (TaskDto taskDto : taskDtoList) {
                mTaskDao.delete(taskDto);
            }
            mPushMessageDao.delete(deleteOperationDto.operationId);
            mOperationDao.delete(deleteOperationDto);

            // 作業削除時、紐づいた資料を削除
            for (Long contentId : contentIds) {
                if (mOperationContentDao.isDeleteContent(contentId)) {
                    ContentDto contentDto = mContentDao.getContent(contentId);
                    mContentLogic.deleteContent(contentDto, true);
                }
            }

            String operationTaskDirPath = ABVEnvironment.getInstance().getOperationDirFile(deleteOperationDto.operationId);
            FileUtil.delete(operationTaskDirPath);
        }
    }

    /**
     * サーバーから作業者グループ取得し、m_worker_groupテーブルへ登録
     *
     * @throws NetworkDisconnectedException
     * @throws AcmsException
     */
    private void setWorkingGroupList() throws NetworkDisconnectedException, AcmsException {
        AcmsParameters param = new AcmsParameters(cache.getMemberInfo().sid);
        WorkerGroupJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getWorkingGroupList(param);
        List<Integer> localWorkerGroupIdList = mWorkerGroupDao.getAllWorkerGroupId();
        for (WorkerGroupDto dto : json.workerGroupList) {
            if (mWorkerGroupDao.isExist(dto.groupId)) {
                localWorkerGroupIdList.remove(dto.groupId);
                mWorkerGroupDao.update(dto);
            } else {
                mWorkerGroupDao.insert(dto);
            }
        }

        // 削除
        for (Integer groupId : localWorkerGroupIdList) {
            mWorkerGroupDao.deleteById(groupId);
        }
    }

    /**
     * 報告データ登録
     *
     * @param taskKey
     * @param operationId
     * @param contentId
     * @param reportLevel
     * @param enableReport
     * @param taskReportJson
     * @param localAttachedFileName
     * @param attachedChangeFlag
     * @param dataSendFlg
     * @param localSavedFlg
     * @param processKey            工程キー
     * @param phaseNo               工程NO
     * @throws IOException
     */
    public void insertTaskReport(String taskKey, long operationId, long contentId,
                                 int reportLevel, int enableReport, JSONObject taskReportJson,
                                 String localAttachedFileName, boolean attachedChangeFlag, boolean dataSendFlg,
                                 boolean localSavedFlg, String processKey, Integer phaseNo) throws IOException {
        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskKey);
        if (taskDto == null) {
            //TODO error
            return;
        }

        TaskReportDto taskReportDto = new TaskReportDto();
        taskReportDto.taskKey = taskKey;

        if (taskReportJson != null) {
            // suggest登録・更新処理
            registTaskReportItem(taskKey, reportLevel, taskReportJson.has("suggest") ? taskReportJson.getJSONObject("suggest") : null);
            taskReportDto.jsonData = taskReportJson.toString();
        } else {
            // taskJsonがnullの場合、jsonDataに空でセット
            taskReportDto.jsonData = "";
        }
        // 一時保存のフラグ
        taskReportDto.localSavedFlg = localSavedFlg;
        // 送信フラグ
        taskReportDto.dataSendFlg = dataSendFlg;
        taskReportDto.attachedFileSendFlg = attachedChangeFlag;
        // 作業報告
        taskReportDto.taskReportLevel = reportLevel;
        // 報告可能区分
        taskReportDto.enableReport = enableReport;
        if (localAttachedFileName != null) {
            taskReportDto.localAttachedFileName = localAttachedFileName;
        }

        mTaskReportDao.insert(taskReportDto);

        if (taskReportJson != null) {
            String tempDirPath = ABVEnvironment.getInstance().getTempTaskDirPath(contentId, taskKey);

            //添付ファイル変更の場合、以下の処理を行う
            JSONObject attachedListJson = taskReportJson.getJSONObject("attached");
            List<String> attachedFileNames = JSONObject.getValues(attachedListJson);
            // コピー元のファイルで、添付ファイルとして使用しないファイル削除
            deleteDifferentialFile(tempDirPath, attachedFileNames);

            String operationDrectionOrReportDirPath = ABVEnvironment.getInstance().getOperationTaskReportLevelDirPath(operationId, taskKey, reportLevel);
            if (processKey != null && phaseNo != 0) { //連続作業の添付ファイル保存場所
                operationDrectionOrReportDirPath = ABVEnvironment.getInstance().getOperationDirectionOrContinuousReportDirPath(operationId, taskKey, reportLevel, processKey, phaseNo);
            }
            // 添付ディレクトリの移動
            boolean result = FileUtil.copy(tempDirPath, operationDrectionOrReportDirPath, true);
            if (result) {
                FileUtil.delete(tempDirPath);
            }
        }
    }

    /**
     * 作業報告を更新
     *
     * @param taskKey               タスクキー
     * @param operationId           作業ID
     * @param contentId             コンテンツID
     * @param taskReportLevel       報告レベル（報告：0, 回答：1）
     * @param enableReport          報告可能区分
     * @param taskReportJson        フォームデータ
     * @param localAttachedFileName 添付ファイル名
     * @param attachedChangeFlag    添付ファイル更新フラグ
     * @param dataSendFlg           データ送信フラグ
     * @param localSavedFlg         ローカル保存フラグ
     * @param processKey            固定キー
     * @param phaseNo               工程NO
     * @throws IOException
     */
    public void updateTaskReport(String taskKey, long operationId, long contentId, int taskReportLevel,
                                 int enableReport, JSONObject taskReportJson, String localAttachedFileName,
                                 boolean attachedChangeFlag, boolean dataSendFlg, boolean localSavedFlg,
                                 String processKey, Integer phaseNo) throws IOException {
        TaskReportDto taskReportDto = mTaskReportDao.getTaskReport(taskKey, taskReportLevel);
        if (taskReportDto == null) {
            //TODO error?
            return;
        }
        if (taskReportJson != null) {
            // suggest登録・更新処理
            registTaskReportItem(taskKey, taskReportLevel, taskReportJson.has("suggest") ? taskReportJson.getJSONObject("suggest") : null);

            taskReportDto.jsonData = taskReportJson.toString();
        } else {
            taskReportDto.jsonData = "";
        }
        // 一時保存のフラグ
        taskReportDto.localSavedFlg = localSavedFlg;
        // 送信フラグ
        taskReportDto.dataSendFlg = dataSendFlg;
        // 作業報告階層
        taskReportDto.taskReportLevel = taskReportLevel;
        // 報告可能区分
        taskReportDto.enableReport = enableReport;

        // attachedFileSendFlgがtrueの場合は、更新しない
        if (!taskReportDto.attachedFileSendFlg) {
            taskReportDto.attachedFileSendFlg = attachedChangeFlag;
        }
        if (localAttachedFileName != null) {
            taskReportDto.localAttachedFileName = localAttachedFileName;
        }

        mTaskReportDao.update(taskReportDto);

        String tempDirPath = ABVEnvironment.getInstance().getTempTaskDirPath(contentId, taskKey);

        if (taskReportDto.attachedFileSendFlg) {
            String operationDrectionOrReportDirPath = ABVEnvironment.getInstance().getOperationTaskReportLevelDirPath(operationId, taskKey, taskReportLevel);
            if (processKey != null && phaseNo != 0) { //連続作業の添付ファイル保存場所
                operationDrectionOrReportDirPath = ABVEnvironment.getInstance().getOperationDirectionOrContinuousReportDirPath(operationId, taskKey, taskReportLevel, processKey, phaseNo);
            }

            //添付ファイル変更の場合、以下の処理を行う
            JSONObject attachedListJson = taskReportJson.getJSONObject("attached");
            List<String> attachedFileNames = JSONObject.getValues(attachedListJson);

            // コピー元のファイルで、添付ファイルとして使用しないファイル削除
            deleteDifferentialFile(tempDirPath, attachedFileNames);
            // 同じファイル名が存在すれば、コピー先ファイルを削除
            deleteDifferentialFile(operationDrectionOrReportDirPath, attachedFileNames);

            // 添付ディレクトリの移動
            boolean result = FileUtil.copy(tempDirPath, operationDrectionOrReportDirPath, true);
            if (result) {
                FileUtil.delete(tempDirPath);
            }
        }
    }

    /**
     * 作業報告/作業報告（回答）の削除
     *
     * @param operationId
     * @param contentId
     * @param taskKey
     */
    public void deleteTaskReport(long operationId, long contentId, String taskKey, int taskReportLevel) {
        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskKey);
        if (taskDto == null) {
            return;
        }
        TaskReportDto taskReportDto = mTaskReportDao.getTaskReport(taskKey, taskReportLevel);
        if (taskReportDto != null) {
            taskReportDto.dataSendFlg = true;
            taskReportDto.attachedFileSendFlg = false;
            if (taskReportDto.taskReportLevel == Constant.TaskReportLevel.ReportType) {
                // 報告
                taskDto.delFlg = true;
            } else {
                // 報告（回答）
                taskReportDto.jsonData = "";
                taskDto.taskStatus = 0;
            }
            mTaskReportDao.update(taskReportDto);
            mTaskDao.update(taskDto);
            // プロジェクトの作業データディレクトリ削除
            deleteTaskFileData(operationId, contentId, taskKey, taskReportLevel);
        }
    }

    /**
     * 「全削除」ボタンで、すべての工程情報を削除する。
     *
     * @param operationId 作業ID
     * @param contentId   コンテンツID
     * @param contentPath コンテンツファイルパス
     * @param processKey  工程ID
     * @throws IOException
     */
    public void deleteProcessAllReport(long operationId, long contentId, String processKey, String contentPath) throws IOException {
        //作業送信フラグ更新
        mOperationDao.updateNeedSyncFlg(operationId, true);
        //作業タスク削除フラグ更新
        List<TaskDto> taskDtoList = mTaskDao.getTaskByProcessKey(processKey);
        for (TaskDto taskDto : taskDtoList) {
            taskDto.delFlg = true;
            taskDto.taskStatus = 0;
            mTaskDao.update(taskDto);
            //作業報告更新
            TaskReportDto taskReportDto = mTaskReportDao.getTaskReport(taskDto.taskKey, Constant.TaskReportLevel.ReportType);
            taskReportDto.dataSendFlg = false;
            taskReportDto.attachedFileSendFlg = false;
            taskReportDto.jsonData = "";
            mTaskReportDao.update(taskReportDto);

            //承認情報削除
            mTaskReportApprovalDao.delete(processKey);

            //報告(Suggest)情報削除
            mTaskReportItemsDao.delete(taskDto.taskKey);

            //作業ファイル削除
            deleteTaskFileData(operationId, contentId, taskDto.taskKey, Constant.TaskReportLevel.ReportType);
        }
        createJsonForOperationContent(operationId, contentPath, false);
    }

    /**
     * 定期点検データ新規
     *
     * @param operationId
     * @param taskReportDto
     * @param attachedChangeFlag
     * @param dataSendFlg
     * @throws IOException
     */
    public void insertRoutineTaskReport(long operationId, long contentId, TaskReportDto taskReportDto, boolean attachedChangeFlag, boolean dataSendFlg) throws IOException {
        taskReportDto.attachedFileSendFlg = attachedChangeFlag;
        taskReportDto.dataSendFlg = dataSendFlg;
        mTaskReportDao.insert(taskReportDto);

        if (!StringUtil.isNullOrEmpty(taskReportDto.jsonData)) {
            JSONObject taskReportJson = new JSONObject(taskReportDto.jsonData);
            JSONObject taskJson = taskReportJson.getJSONObject("task");
            Iterator taskKeys = taskJson.keys();
            while (taskKeys.hasNext()) {
                TaskReportItemsDto taskReportItemsDto = new TaskReportItemsDto();
                String itemKey = (String) taskKeys.next();
                taskReportItemsDto.taskKey = taskReportDto.taskKey;
                taskReportItemsDto.itemKey = itemKey;
                try {
                    taskReportItemsDto.inputValue = taskJson.getString(itemKey);
                } catch (JSONException e) {
                    // 値がStringではない場合、無視する
                    continue;
                }
                mTaskReportItemsDao.insertTaskReportItems(taskReportItemsDto);
            }
        }

        //添付ファイル変更の場合、以下の処理を行う
        String strReportStartDate = DateTimeUtil.toString_yyyyMMddHHmmss_none(taskReportDto.reportStartDate);

        String tempDirPath = ABVEnvironment.getInstance().getTempTaskDirPath(contentId, taskReportDto.taskKey);
        String routineTaskReportDirPath = ABVEnvironment.getInstance().getRoutineTaskReportDirFilePath(operationId, taskReportDto.taskKey, taskReportDto.taskReportId, strReportStartDate);

        if (!StringUtil.isNullOrEmpty(taskReportDto.jsonData)) {
            JSONObject attachedListJson = (new JSONObject(taskReportDto.jsonData)).getJSONObject("attached");
            List<String> attachedFileNames = JSONObject.getValues(attachedListJson);
            // コピー元のファイルで、添付ファイルとして使用しないファイル削除
            deleteDifferentialFile(tempDirPath, attachedFileNames);
        }

        // 添付ディレクトリの移動
        boolean result = FileUtil.copy(tempDirPath, routineTaskReportDirPath, true);
        if (result) {
            FileUtil.delete(tempDirPath);
        }
    }

    /**
     * 定期点検データ更新
     *
     * @param operationId
     * @param contentId
     * @param taskReportDto
     * @param attachedChangeFlag
     * @param dataSendFlg
     * @param localSavedFlg
     * @throws IOException
     */
    public void updateRoutineTaskReport(long operationId, long contentId, TaskReportDto taskReportDto, boolean attachedChangeFlag, boolean dataSendFlg, boolean localSavedFlg) throws IOException {
        taskReportDto.attachedFileSendFlg = attachedChangeFlag;
        taskReportDto.dataSendFlg = dataSendFlg;
        taskReportDto.localSavedFlg = localSavedFlg;
        mTaskReportDao.update(taskReportDto);

        if (!StringUtil.isNullOrEmpty(taskReportDto.jsonData)) {
            JSONObject taskReportJson = new JSONObject(taskReportDto.jsonData);
            JSONObject taskJson = taskReportJson.getJSONObject("task");

            List<TaskReportItemsDto> taskReportItemsDaoList = mTaskReportItemsDao.getTaskReportItemByTaskKey(taskReportDto.taskKey, Constant.TaskReportLevel.ReportType);
            if (taskReportItemsDaoList.size() > 0) {
                // 更新
                for (TaskReportItemsDto taskReportItemsDto : taskReportItemsDaoList) {
                    try {
                        String newValue = taskJson.getString(taskReportItemsDto.itemKey);
                        if (!newValue.equals(taskReportItemsDto.inputValue)) {
                            // 値が異なる場合のみ、更新する
                            taskReportItemsDto.inputValue = newValue;
                            mTaskReportItemsDao.updateTaskReportItems(taskReportItemsDto);
                        }
                    } catch (JSONException e) {
                        // 値がStringではない場合、無視する
                        continue;
                    }
                }
            } else {
                // 登録
                Iterator taskKeys = taskJson.keys();
                while (taskKeys.hasNext()) {
                    TaskReportItemsDto taskReportItemsDto = new TaskReportItemsDto();
                    String itemKey = (String) taskKeys.next();
                    taskReportItemsDto.taskKey = taskReportDto.taskKey;
                    taskReportItemsDto.itemKey = itemKey;
                    taskReportItemsDto.taskReportLevel = Constant.TaskReportLevel.ReportType;
                    try {
                        taskReportItemsDto.inputValue = taskJson.getString(itemKey);
                    } catch (JSONException e) {
                        // 値がStringではない場合、無視する
                        continue;
                    }
                    mTaskReportItemsDao.insertTaskReportItems(taskReportItemsDto);
                }
            }

            String strReportStartDate = DateTimeUtil.toString_yyyyMMddHHmmss_none(taskReportDto.reportStartDate);
            String tempDirPath = ABVEnvironment.getInstance().getTempTaskDirPath(contentId, taskReportDto.taskKey);

            String routineTaskReportDirPath = ABVEnvironment.getInstance().getRoutineTaskReportDirFilePath(operationId, taskReportDto.taskKey, taskReportDto.taskReportId, strReportStartDate);

            if (taskReportDto.attachedFileSendFlg) {
                if (!StringUtil.isNullOrEmpty(taskReportDto.jsonData)) {
                    //添付ファイル変更の場合、以下の処理を行う
                    JSONObject attachedListJson = (new JSONObject(taskReportDto.jsonData)).getJSONObject("attached");
                    List<String> attachedFileNames = JSONObject.getValues(attachedListJson);
                    // コピー元のファイルで、添付ファイルとして使用しないファイル削除
                    deleteDifferentialFile(tempDirPath, attachedFileNames);
                    // 同じファイル名が存在すれば、コピー先ファイルを削除
                    deleteDifferentialFile(routineTaskReportDirPath, attachedFileNames);
                }
            }
            // 添付ディレクトリの移動
            boolean result = FileUtil.copy(tempDirPath, routineTaskReportDirPath, true);
            if (result) {
                FileUtil.delete(tempDirPath);
            }
        }
    }

    /**
     * 作業報告送信データ、新規
     *
     * @param taskKey
     * @param taskReportJson
     * @param deleteFlg
     * @return
     * @throws IOException
     */
    public int insertTaskReportSend(String taskKey, int taskReportId, String reportStartDate, JSONObject taskReportJson, boolean deleteFlg, boolean hotspotChangeFlg) throws IOException {
        // 登録前にホットスポット更新フラグが存在すれば、そのデータは削除する
        mTaskReportSendDao.deleteHotspotChangedData(taskKey);

        TaskReportSendDto taskReportSendDto = new TaskReportSendDto();

        taskReportSendDto.taskKey = taskKey;

        if (taskReportId > 0 && !StringUtil.isNullOrEmpty(reportStartDate)) {
            taskReportSendDto.taskReportId = taskReportId;
            taskReportSendDto.reportStartDate = reportStartDate;
        }

        if (taskReportJson == null) {
            taskReportSendDto.jsonData = "";
        } else {
            taskReportSendDto.jsonData = taskReportJson.toString();
        }
        taskReportSendDto.saveDate = DateTimeUtil.getCurrentSqlDate();
        taskReportSendDto.deleteFlg = deleteFlg;
        taskReportSendDto.hotspotChangeFlg = hotspotChangeFlg;

        return mTaskReportSendDao.insert(taskReportSendDto);
    }

    /**
     * 作業報告送信データ、更新
     *
     * @param taskKey
     * @param taskReportId
     * @param reportStartDate
     * @param taskReportJson
     * @param deleteFlg
     * @return
     * @throws IOException
     */
    public int updateTaskReportSend(String taskKey, int taskReportId, String reportStartDate, JSONObject taskReportJson, boolean deleteFlg) throws IOException {
        // 登録前にホットスポット更新フラグが存在すれば、そのデータは削除する
        mTaskReportSendDao.deleteHotspotChangedData(taskKey);

        // 最新のtaskReportSendデータを更新（更新時、データは１つしかないのが前提）
        TaskReportSendDto taskReportSendDto = mTaskReportSendDao.getLastTaskReportSend(taskKey);

        if (taskReportSendDto == null) {
            // データがない場合、insertを行うため新しいインスタントでセット
            taskReportSendDto = new TaskReportSendDto();
        }

        taskReportSendDto.taskKey = taskKey;

        if (taskReportId > 0 && !StringUtil.isNullOrEmpty(reportStartDate)) {
            taskReportSendDto.taskReportId = taskReportId;
            taskReportSendDto.reportStartDate = reportStartDate;
        }

        if (taskReportJson == null) {
            taskReportSendDto.jsonData = "";
        } else {
            taskReportSendDto.jsonData = taskReportJson.toString();
        }
        taskReportSendDto.saveDate = DateTimeUtil.getCurrentSqlDate();
        taskReportSendDto.deleteFlg = deleteFlg;

        int taskReportSendId = mTaskReportSendDao.update(taskReportSendDto);

        if (taskReportSendId == 0) {
            taskReportSendId = insertTaskReportSend(taskKey, taskReportId, reportStartDate, taskReportJson, false, false);
        }

        return taskReportSendId;
    }

    /**
     * 作業報告送信データID取得
     *
     * @param taskKey
     * @return
     */
    public int getLastTaskReportSendId(String taskKey) {
        return mTaskReportSendDao.getLastTaskReportSendId(taskKey);
    }

    /**
     * 定期点検用、作業削除
     *
     * @param operationId
     * @param contentId
     * @param dto
     */
    public void deleteRoutineTaskReport(long operationId, long contentId, TaskReportDto dto) {
        mTaskReportDao.deleteRoutineTaskReport(dto);

        String reportStartDate = DateTimeUtil.toString(dto.reportStartDate, DateTimeFormat.yyyyMMddHHmmss_hyphen);
        mTaskReportSendDao.deleteByTaskKeyIdDate(dto.taskKey, dto.taskReportId, reportStartDate);

        reportStartDate = DateTimeUtil.toString_yyyyMMddHHmmss_none(dto.reportStartDate);

        // 作業報告のディレクトリ削除
        FileUtil.delete(ABVEnvironment.getInstance().getTempTaskDirPath(contentId, dto.taskKey));
        FileUtil.delete(ABVEnvironment.getInstance().getRoutineTaskReportDirFilePath(operationId, dto.taskKey, dto.taskReportId, reportStartDate));
    }

    /**
     * 定期点検用、作業内容削除
     *
     * @param operationId
     * @param contentId
     * @param taskKey
     * @param taskReportId
     * @param reportStartDate
     */
    public void deleteRoutineTaskReport(long operationId, long contentId, String taskKey, int taskReportId, String reportStartDate) {
        TaskReportDto taskReportDto = mTaskReportDao.getRoutineTaskReportUtc(taskKey, taskReportId, reportStartDate);
        if (taskReportDto != null) {
            taskReportDto.dataSendFlg = true;
            taskReportDto.attachedFileSendFlg = false;
            taskReportDto.jsonData = "";
            mTaskReportDao.update(taskReportDto);
        }

        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskKey);
        if (taskDto != null) {
            taskDto.taskStatus = 0;
            mTaskDao.update(taskDto);
        }

        // 作業報告のディレクトリ削除

        // reportStartDateをファイルパス用の文字列に変換
        String reportFileDateStr = DateTimeUtil.toString_yyyyMMddHHmmss_none(DateTimeUtil.toDate(reportStartDate, DateTimeFormat.yyyyMMddHHmmss_hyphen));
        FileUtil.delete(ABVEnvironment.getInstance().getTempTaskDirPath(contentId, taskKey));
        FileUtil.delete(ABVEnvironment.getInstance().getRoutineTaskReportDirFilePath(operationId, taskKey, taskReportId, reportFileDateStr));
    }


    public TaskReportDto getRoutineTaskReportUtc(String taskKey, int taskReportId, String reportStartDate) {
        return mTaskReportDao.getRoutineTaskReportUtc(taskKey, taskReportId, reportStartDate);
    }

    /**
     * ホットスポットの更新
     *
     * @param taskKey
     * @param param
     */
    public void updateTaskHotspot(String taskKey, Map<String, String> param) throws IOException {
        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskKey);
        TaskReportDto taskReportDto = mTaskReportDao.getTaskReport(taskKey, Constant.TaskReportLevel.ReportType);
        if (mTaskReportSendDao.isNotExistSendableData(taskReportDto.taskKey) && !taskReportDto.dataSendFlg && !taskReportDto.localSavedFlg) {
            // 送信データが存在しない場合のみ、ホットスポットをtrueにセット
            insertTaskReportSend(taskReportDto.taskKey, taskReportDto.taskReportId, null, new JSONObject(taskReportDto.jsonData), false, true);
        }
        taskDto.taskHotSpotInfo = param.get(ABookKeys.HOT_SPOT);
        taskReportDto.dataSendFlg = true;
        mTaskReportDao.update(taskReportDto);
        mTaskDao.update(taskDto);
    }


    /**
     * ホットスポットの位置更新
     *
     * @param taskKey
     * @param newX
     * @param newY
     */

    public void updateTaskHotspot(String taskKey, float newX, float newY, int taskReportLevel) throws IOException {
        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskKey);
        // #32782 指示者テーブル関連削除 start
        TaskReportDto taskReportDto = mTaskReportDao.getTaskReport(taskKey, taskReportLevel);
        JSONObject hotSpot = new JSONObject(taskDto.taskHotSpotInfo);
        hotSpot.put("x", newX);
        hotSpot.put("y", newY);
        taskDto.taskHotSpotInfo = hotSpot.toString();
        if (mTaskReportSendDao.isNotExistSendableData(taskReportDto.taskKey) && !taskReportDto.dataSendFlg && !taskReportDto.localSavedFlg) {
            // 送信データが存在しない場合のみ、ホットスポットをtrueにセット
            insertTaskReportSend(taskReportDto.taskKey, taskReportDto.taskReportId, null, new JSONObject(taskReportDto.jsonData), false, true);
        }
        taskReportDto.dataSendFlg = true;
        mTaskReportDao.update(taskReportDto);
        mTaskDao.update(taskDto);
    }

    /**
     * 作業指示、作業報告用のjsonファイル
     * taskReport.jsonとtaskReportSuggest.jsonとhotspot.jsonを作成
     *
     * @param operationId
     * @param contentPath
     * @param routineTaskReportFlg
     * @throws IOException
     */
    public void createJsonForOperationContent(Long operationId, String contentPath, boolean routineTaskReportFlg) throws IOException {
        try {
            OperationDto operationDto = mOperationDao.getOperation(operationId);
            if (operationDto.reportType == Constant.ReportType.ReportContinuous) {
                contentPath = contentPath.replaceAll("panoImage", "processList");
                contentPath = contentPath.replaceAll("taskPdf", "processList");
                createContinuousTaskReportJson(operationId);
                createContinuousTaskReportSuggestJson(operationId);
                createProcessInfoJson(operationId, contentPath);
                createPhaseStatusJson(operationId, contentPath);
            } else {
                if (routineTaskReportFlg) {
                    // 定期点検
                    createRoutineTaskReportJson(operationId, contentPath);
                } else {
                    createTaskReportJson(operationId, contentPath);
                }
                createTaskReportSuggestJson(operationId, contentPath);
            }

            createHopSpotJson(operationId, contentPath);

        } catch (IOException e) {
            Logger.e(TAG, "createJsonForOperationContent error : ", e);
            throw e;
        }
    }

    public void createJsonForRoutineTaskReport(int reportCycle, int enableReportUpdate, int taskReportId, String contentPath) throws IOException {
        JSONObject routineTaskJson = new JSONObject();

        routineTaskJson.put(ABookKeys.REPORT_CYCLE, reportCycle);
        routineTaskJson.put(ABookKeys.ENABLE_REPORT_UPDATE, enableReportUpdate);
        routineTaskJson.put(ABookKeys.TASK_REPORT_ID, taskReportId);

        Logger.d(TAG, "routineTask.json : " + routineTaskJson.toString());
        FileUtil.createFile(contentPath + "/" + ABookKeys.ROUTINE_TASK + ".json", routineTaskJson.toString());
    }

    /**
     * 作業指示、作業報告用のjsonファイル
     * content.jsonとworkerGroup.jsonを作成
     *
     * @throws IOException
     */
    public void createJsonForOpenABookCheckPano(Long operationId, Long contentId, String contentPath) throws IOException {
        try {
            createContentJson(operationId, contentId, contentPath);
            // #32926 作業報告画面改善 start
            // 作業タイプ確認用のparameterを追加
            createWorkerGroupJson(operationId, contentPath);
            // #32926 作業報告画面改善 end
        } catch (IOException e) {
            Logger.e(TAG, "createJsonForOpenABookCheckPano error : ", e);
            throw e;
        }
    }


    /**
     * content.jsonファイル作成
     *
     * @param contentId
     * @param contentPath
     * @throws IOException
     */
    private void createContentJson(Long operationId, Long contentId, String contentPath) throws IOException {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("workerCode", ABVDataCache.getInstance().getMemberInfo().workerCode);

        //　データディレクトリへの相対パス
        OperationDto operationDto = getOperation(operationId);
        if (operationDto.reportType == Constant.ReportType.ReportContinuous) {
            jsonObject.put("attachedPath", "../../../../../../files/ABook/operation/" + operationId);
        } else {
            jsonObject.put("attachedPath", "../../../../files/ABook/operation/" + operationId);
        }


        jsonObject.put("attachedMoviePath", ABVEnvironment.getInstance().getAttachedMoviesFilePath(contentId));

        // 絞り検索マスタデータのパス
        jsonObject.put("masterPath", ABVEnvironment.getInstance().getMasterFilePath() + File.separator + ABVEnvironment.getInstance().ApertureMasterDataFileName);

        FileUtil.createFile(contentPath + "/content.json", jsonObject.toString());
    }

    /**
     * workerGroup.jsonファイル作成
     *
     * @param operationId
     * @param contentPath
     * @throws IOException
     */
    // 現在作業タイプを確認用のparameter追加
    private void createWorkerGroupJson(Long operationId, String contentPath) throws IOException {
        int level = 0;
        int listCount = 0;
        List<JSONObject> workerGroupJsonList = new ArrayList<JSONObject>();
        JSONObject taskGroupJson = new JSONObject();
        List<TaskWorkerGroupDto> taskWorkerGroupDtoList = mTaskWorkerGroupDao.getTaskWorkerGroupByOperationId(operationId);
        // 報告と報告（回答）のデータの区分が必要なので、「taskGroup_0」と「taskGroup_1」で形式で作成

        for (TaskWorkerGroupDto dto : taskWorkerGroupDtoList) {
            listCount++;        // 最後のインデックスの判定のため
            if (level != dto.taskReportLevel) {
                taskGroupJson.put(String.format("taskReport_%s", String.valueOf(level)), workerGroupJsonList);
                workerGroupJsonList = new ArrayList<JSONObject>();
                level++;
            }

            JSONObject workerGroupJson = new JSONObject();
            workerGroupJson.put(ABookKeys.GROUP_ID, dto.groupId);
            workerGroupJson.put(ABookKeys.GROUP_NAME, dto.groupName);
            workerGroupJsonList.add(workerGroupJson);

            // 最後の時、taskReport_{level}でセット
            if (listCount == taskWorkerGroupDtoList.size()) {
                taskGroupJson.put(String.format("taskGroup_%s", String.valueOf(level)), workerGroupJsonList);
            }
        }

        Logger.d(TAG, "createWorkerGroupJson : %s  contentPath : %s ", taskGroupJson.toString(), contentPath + "/workingGroup.json");
        FileUtil.createFile(contentPath + "/workingGroup.json", taskGroupJson.toString());
    }

    /**
     * taskHotspot.jsonファイル作成
     *
     * @param operationId
     * @param contentPath
     * @throws IOException
     */
    public void createHopSpotJson(Long operationId, String contentPath) throws IOException {
        List<JSONObject> taskJsonList = new ArrayList<JSONObject>();
        JSONObject hotspotJson = new JSONObject();
        List<TaskDto> taskDtoList = mTaskDao.selectTaskByOperationId(operationId);
        OperationDto operation = mOperationDao.getOperation(operationId);
        if (operation.reportType == Constant.ReportType.ReportContinuous) {
            contentPath = contentPath.replaceAll("panoImage", "processList");
            contentPath = contentPath.replaceAll("taskPdf", "processList");
        }
        for (TaskDto dto : taskDtoList) {
            if (!StringUtil.isNullOrEmpty(dto.taskHotSpotInfo)) {
                JSONObject taskHotSpot = new JSONObject(dto.taskHotSpotInfo);
                taskHotSpot.put(OperationDataJSON.TaskCode, dto.taskCode);
                taskJsonList.add(taskHotSpot);
            }
        }
        hotspotJson.put("hotspot", taskJsonList);
        Logger.d(TAG, "createHopSpotJson : " + hotspotJson.toString());
        FileUtil.createFile(contentPath + "/taskHotspot.json", hotspotJson.toString());
    }

    /**
     * phaseStatus.jsonファイル作成(工程承認情報表示用)
     *
     * @param operationId 作業ID
     * @param contentPath 保存パス
     * @throws IOException
     */
    public void createPhaseStatusJson(Long operationId, String contentPath) throws IOException {
        List<TaskDto> taskDtoProcessKeyList = mTaskDao.selectTaskGroupByProcessKeyByOperationId(operationId);
        List<JSONObject> processInfoList = new ArrayList<JSONObject>();
        for (TaskDto taskDtoProcessKey : taskDtoProcessKeyList) {
            JSONObject processJson = new JSONObject();
            String processKey = taskDtoProcessKey.processKey;
            Integer processStatus = taskDtoProcessKey.processStatus;
            processJson.put(ABookKeys.PROCESS_KEY, processKey);
            processJson.put(ABookKeys.PROCESS_STATUS, processStatus);
            List<TaskDto> taskDtoList = mTaskDao.getTaskByProcessKey(processKey);
            List<JSONObject> phaseInfoList = new ArrayList<JSONObject>();
            for (TaskDto taskDto : taskDtoList) {
                JSONObject phaseInfoJson = new JSONObject();
                phaseInfoJson.put(ABookKeys.PHASE_NO, taskDto.phaseNo);
                phaseInfoJson.put(ABookKeys.PHASE_STAUTS, taskDto.phaseStatus);

                List<JSONObject> approvalList = new ArrayList<JSONObject>();
                List<TaskReportApprovalDto> taskReportApprovalDtoList = mTaskReportApprovalDao.selectTaskReportApprovalByProcessKey(taskDto.processKey, taskDto.phaseNo);
                for (TaskReportApprovalDto taskReportApprovalDto : taskReportApprovalDtoList) {
                    JSONObject approvalJson = new JSONObject();
                    approvalJson.put(ABookKeys.APPROVAL_GROUP_ID, taskReportApprovalDto.approvalGroupId);
                    approvalJson.put(ABookKeys.APPROVAL_GROUP_NAME, taskReportApprovalDto.approvalGroupName);
                    approvalJson.put(ABookKeys.APPROVAL_NANME, taskReportApprovalDto.approvalName);
                    approvalJson.put(ABookKeys.APPROVAL_NO, taskReportApprovalDto.approvalNo);
                    approvalJson.put(ABookKeys.COMMENT, taskReportApprovalDto.comment);
                    approvalJson.put(ABookKeys.SEND_BACK_PHASE_NO, taskReportApprovalDto.sendbackPhaseNo);
                    approvalJson.put(ABookKeys.SIGNATURE, taskReportApprovalDto.signature);
                    approvalJson.put(ABookKeys.STATUS, taskReportApprovalDto.status);
                    approvalList.add(approvalJson);
                }
                phaseInfoJson.put(ABookKeys.APPROVAL_LIST, approvalList);
                phaseInfoList.add(phaseInfoJson);
            }
            processJson.put(ABookKeys.PHASE_LIST, phaseInfoList);
            processInfoList.add(processJson);
        }

        JSONObject phaseStatusJson = new JSONObject();
        phaseStatusJson.put(ABookKeys.PROCESS_LIST, processInfoList);
        OperationDto operation = mOperationDao.getOperation(operationId);
        if (operation.operationType == PANO || operation.operationType == DRAWING || operation.operationType == PDF) {
            contentPath = contentPath.replaceAll("panoImage", "processList");
            contentPath = contentPath.replaceAll("taskPdf", "processList");
        }
        FileUtil.createFile(contentPath + "/phaseStatus.json", phaseStatusJson.toString());
    }

    /**
     * processInfo.jsonファイル作成(報告一覧表示用)
     *
     * @param operationId
     * @param contentPath
     * @throws IOException
     */
    private void createProcessInfoJson(Long operationId, String contentPath) throws IOException {
        List<JSONObject> taskJsonList = new ArrayList<JSONObject>();
        JSONObject processInfoJson = new JSONObject();
        List<TaskDto> taskDtoList = mTaskDao.selectTaskGroupByProcessKeyByOperationId(operationId);
        for (TaskDto dto : taskDtoList) {
            JSONObject taskDtoJson = new JSONObject();
            taskDtoJson.put(ABookKeys.TASK_NAME, dto.taskName);
            taskDtoJson.put(ABookKeys.TASK_CODE, dto.taskCode);
            taskDtoJson.put(ABookKeys.PROCESS_KEY, dto.processKey);
            taskDtoJson.put(ABookKeys.PROCESS_STATUS, dto.processStatus);
            taskJsonList.add(taskDtoJson);
        }
        processInfoJson.put("processList", taskJsonList);
        OperationDto operation = mOperationDao.getOperation(operationId);
        if (operation.operationType == PANO || operation.operationType == DRAWING || operation.operationType == PDF) {
            contentPath = contentPath.replaceAll("panoImage", "processList");
            contentPath = contentPath.replaceAll("taskPdf", "processList");
        }
        Logger.d(TAG, "createProcessInfoJson : " + processInfoJson.toString());
        FileUtil.createFile(contentPath + "/processInfo.json", processInfoJson.toString());
    }

    private String getProcessInfoJson(Long operationId, String contentPath) throws IOException {
        List<JSONObject> taskJsonList = new ArrayList<JSONObject>();
        JSONObject processInfoJson = new JSONObject();
        List<TaskDto> taskDtoList = mTaskDao.selectTaskGroupByProcessKeyByOperationId(operationId);
        for (TaskDto dto : taskDtoList) {
            JSONObject taskDtoJson = new JSONObject();
            taskDtoJson.put(ABookKeys.TASK_NAME, dto.taskName);
            taskDtoJson.put(ABookKeys.TASK_CODE, dto.taskCode);
            taskDtoJson.put(ABookKeys.PROCESS_KEY, dto.processKey);
            taskDtoJson.put(ABookKeys.PROCESS_STATUS, dto.processStatus);
            taskJsonList.add(taskDtoJson);
        }
        processInfoJson.put("processList", taskJsonList);
        OperationDto operation = mOperationDao.getOperation(operationId);
        if (operation.operationType == PANO || operation.operationType == DRAWING || operation.operationType == PDF) {
            contentPath = contentPath.replaceAll("panoImage", "processList");
            contentPath = contentPath.replaceAll("taskPdf", "processList");
        }
        Logger.d(TAG, "createProcessInfoJson : " + processInfoJson.toString());
        FileUtil.createFile(contentPath + "/processInfo.json", processInfoJson.toString());
        return processInfoJson.toString();
    }

    /**
     * 作業報告用のjsonファイル
     * 報告と報告（回答）のデータの区分が必要なので、「taskReport_0」と「taskReport_1」で形式で作成
     * 報告（回答）の場合はtaskReport_1を作成
     *
     * @param operationId
     * @param contentPath
     * @throws IOException
     */
    public void createTaskReportJson(Long operationId, String contentPath) throws IOException {
        List<JSONObject> taskReportJsonList = new ArrayList<JSONObject>();
        JSONObject taskReportJson = new JSONObject();
        OperationDto operationDto = mOperationDao.getOperation(operationId);
        List<TaskReportDto> taskReportDtoList = mTaskReportDao.getTaskReportByOperationId(operationId);

        int level = 0;
        int listCount = 0;
        for (TaskReportDto dto : taskReportDtoList) {
            listCount++;        // 最後のインデックスの判定のため
            if (level != dto.taskReportLevel) {
                taskReportJson.put(String.format("taskReport_%s", String.valueOf(level)), taskReportJsonList);
                taskReportJsonList = new ArrayList<JSONObject>();
                level++;
            }

            if (dto.jsonData != null && dto.jsonData.length() > 0) {
                if (operationDto.reportType == Constant.ReportType.ReportReply) {
                    JSONObject editJson = new JSONObject(dto.jsonData);
                    editJson.put(ABookKeys.HAS_AUTHORITY, dto.enableReport);
                    editJson.put(ABookKeys.LOCAL_SAVE_FLG, dto.localSavedFlg);
                    taskReportJsonList.add(editJson);
                } else {
                    taskReportJsonList.add(new JSONObject(dto.jsonData));
                }
            } else {
                JSONObject emptyTaskReportJson = new JSONObject();
                emptyTaskReportJson.put(ABookKeys.TASK_KEY, dto.taskKey);
                if (operationDto.reportType == Constant.ReportType.ReportReply) {
                    emptyTaskReportJson.put(ABookKeys.HAS_AUTHORITY, dto.enableReport);
                    emptyTaskReportJson.put(ABookKeys.LOCAL_SAVE_FLG, dto.localSavedFlg);
                }
                taskReportJsonList.add(emptyTaskReportJson);
            }

            // 最後の時、taskReport_{level}でセット
            if (listCount == taskReportDtoList.size()) {
                taskReportJson.put(String.format("taskReport_%s", String.valueOf(level)), taskReportJsonList);
            }
        }

        // データが存在しない場合、
        if (taskReportDtoList.size() <= 0) {
            taskReportJson.put("taskReport_0", taskReportJsonList);
        }

        Logger.d(TAG, "createTaskReportJson : " + taskReportJson.toString());
        Logger.d(TAG, contentPath + "/" + ABookKeys.TASK_REPORT + ".json");

        FileUtil.createFile(contentPath + "/" + ABookKeys.TASK_REPORT + ".json", taskReportJson.toString());
    }

    /**
     * 定期点検用のJSON作成
     *
     * @param operationId
     * @param contentPath
     * @throws IOException
     */
    public void createRoutineTaskReportJson(Long operationId, String contentPath) throws IOException {
        List<JSONObject> taskReportJsonList = new ArrayList<JSONObject>();
        JSONObject taskReportJsonRoot = new JSONObject();
        JSONObject taskReportJsonRow;
        List<TaskReportDto> routineTaskReportDtoList = mTaskReportDao.getTaskReportByOperationId(operationId);
        // 現在の作業タイプは定期点検の場合
        for (TaskReportDto dto : routineTaskReportDtoList) {
            taskReportJsonRow = new JSONObject();
            taskReportJsonRow.put(ABookKeys.TASK_REPORT_INFO_ID, dto.taskReportInfoId);
            taskReportJsonRow.put(ABookKeys.REPORT_START_DATE, DateTimeUtil.toStringInTimeZone(dto.reportStartDate, DateTimeFormat.yyyyMMddHHmmss_hyphen, DateTimeUtil.getLocalTimeZone()));
            taskReportJsonRow.put(ABookKeys.REPORT_END_DATE, DateTimeUtil.toStringInTimeZone(dto.reportEndDate, DateTimeFormat.yyyyMMddHHmmss_hyphen, DateTimeUtil.getLocalTimeZone()));
            // 一時保存フラグ設定
            taskReportJsonRow.put(ABookKeys.LOCAL_SAVE_FLG, dto.localSavedFlg);
            List<JSONObject> taskReportInfoList = new ArrayList<JSONObject>();
            if (!StringUtil.isNullOrEmpty(dto.jsonData)) {
                taskReportInfoList.add(new JSONObject(dto.jsonData));
            }
            taskReportJsonRow.put(ABookKeys.TASK_REPORT_INFO, taskReportInfoList);
            taskReportJsonList.add(taskReportJsonRow);
        }
        taskReportJsonRoot.put(ABookKeys.ENABLE_REPORT_UPDATE, mOperationDao.getOperation(operationId).enableReportUpdate);
        // 最後のtaskReportId
        TaskReportDto lastRoutineTaskReportDto = mTaskReportDao.getLastRoutineTaskReportData(operationId);
        taskReportJsonRoot.put(ABookKeys.TASK_REPORT_ID, lastRoutineTaskReportDto.taskReportId);
        taskReportJsonRoot.put("taskReport_0", taskReportJsonList);
        Logger.d(TAG, "createRoutineTaskReportJson : " + taskReportJsonRoot.toString());
        Logger.d(TAG, contentPath + "/" + ABookKeys.TASK_REPORT + ".json");
        FileUtil.createFile(contentPath + "/" + ABookKeys.TASK_REPORT + ".json", taskReportJsonRoot.toString());
    }

    /**
     * 　連続作業用の報告Suggestデータ作成（taskReportSuggest.json）
     *
     * @param operationId 作業ID
     * @throws IOException I/O例外
     */
    private void createContinuousTaskReportSuggestJson(Long operationId) throws IOException {
        OperationDto operationDto = getOperation(operationId);
        List<TaskReportItemsDto> taskReportItemsDtoList = mTaskReportItemsDao.getTaskReportItemByOperationId(operationId);

        Map<Integer, JSONObject> taskReportItemMap = new HashMap<Integer, JSONObject>();
        for (TaskReportItemsDto dto : taskReportItemsDtoList) {
            JSONObject itemJson = taskReportItemMap.get(dto.phaseNo);
            if (itemJson == null) {
                itemJson = new JSONObject();
            }
            JSONArray jsonArray = new JSONArray();
            if (itemJson.has(dto.itemKey)) {
                boolean result = false;
                jsonArray = itemJson.getJSONArray(dto.itemKey);
                for (int i = 0; i < jsonArray.length(); i++) {
                    if (jsonArray.get(i).equals(dto.inputValue)) {
                        result = true;
                        break;
                    }
                }
                if (!result) {
                    jsonArray.put(dto.inputValue);
                    itemJson.put(dto.itemKey, jsonArray);
                }
            } else {
                jsonArray.put(dto.inputValue);
                itemJson.put(dto.itemKey, jsonArray);
            }
            taskReportItemMap.put(dto.phaseNo, itemJson);
        }
        for (Integer key : taskReportItemMap.keySet()) {
            JSONObject savedTaskReportItemJson = taskReportItemMap.get(key);
            JSONObject taskReportSuggestJson = new JSONObject();
            taskReportSuggestJson.put(ABookKeys.TASK_REPORT_SUGGEST + "_0", savedTaskReportItemJson);
            Logger.d(TAG, "createTaskReportSuggestJson : " + taskReportSuggestJson.toString());
            String saveJsonPath = ABVEnvironment.getInstance().getContinuousReportDirectoryPath(operationDto.contentId, key);
            FileUtil.createFile(saveJsonPath + "/" + ABookKeys.TASK_REPORT_SUGGEST + ".json", taskReportSuggestJson.toString());
        }
    }

    /**
     * 連続作業用の報告用データ作成（TaskReport.json作成）
     *
     * @param operationId 作業ID
     * @throws IOException I/O例外
     */
    private void createContinuousTaskReportJson(Long operationId) throws IOException {
        List<JSONObject> taskReportJsonList = new ArrayList<JSONObject>();
        OperationDto operationDto = getOperation(operationId);
        List<TaskDto> taskDtoList = mTaskDao.selectTaskByOperationId(operationId);

        Map<Integer, List<JSONObject>> reportJsonMap = new HashMap<Integer, List<JSONObject>>();

        for (TaskDto taskDto : taskDtoList) {
            if (taskDto.delFlg) {
                continue;
            }
            List<TaskReportDto> taskReportDtoList = mTaskReportDao.getTaskReportListByTaskKey(taskDto.taskKey);
            Integer phaseNo = taskDto.phaseNo;
            for (TaskReportDto taskReportDto : taskReportDtoList) {
                JSONObject taskReportJson = new JSONObject();
                if (taskReportDto.jsonData != null && taskReportDto.jsonData.length() > 0) {
                    taskReportJson = new JSONObject(taskReportDto.jsonData);
                }
                taskReportJson.put(ABookKeys.PROCESS_KEY, taskDto.processKey);
                taskReportJson.put(ABookKeys.HAS_AUTHORITY, taskReportDto.enableReport);

                List<JSONObject> reportJsonList = reportJsonMap.get(phaseNo);
                if (reportJsonList == null) {
                    reportJsonList = new ArrayList<JSONObject>();
                }
                reportJsonList.add(taskReportJson);
                reportJsonMap.put(phaseNo, reportJsonList);
            }
        }
        for (Integer key : reportJsonMap.keySet()) {
            List<JSONObject> savedReportJsonList = reportJsonMap.get(key);
            JSONObject taskReportJson = new JSONObject();
            taskReportJson.put("taskReport_0", savedReportJsonList);

            String saveJsonPath = ABVEnvironment.getInstance().getContinuousReportDirectoryPath(operationDto.contentId, key);
            FileUtil.createFile(saveJsonPath + "/" + ABookKeys.TASK_REPORT + ".json", taskReportJson.toString());
        }
    }

    private void createTaskReportSuggestJson(Long operationId, String contentPath) throws IOException {
        JSONObject itemJson = new JSONObject();
        JSONObject taskReportSuggestJson = new JSONObject();
        List<TaskReportItemsDto> taskReportItemsDtoList = mTaskReportItemsDao.getTaskReportItemByOperationId(operationId);
        for (TaskReportItemsDto dto : taskReportItemsDtoList) {
            JSONArray jsonArray = new JSONArray();
            if (itemJson.has(dto.itemKey)) {
                boolean result = false;
                jsonArray = itemJson.getJSONArray(dto.itemKey);
                for (int i = 0; i < jsonArray.length(); i++) {
                    if (jsonArray.get(i).equals(dto.inputValue)) {
                        result = true;
                        break;
                    }
                }
                if (!result) {
                    jsonArray.put(dto.inputValue);
                    itemJson.put(dto.itemKey, jsonArray);
                }
            } else {
                jsonArray.put(dto.inputValue);
                itemJson.put(dto.itemKey, jsonArray);
            }
        }
        // #32926 作業報告画面改善 start
        List<TaskReportDto> taskReportList = mTaskReportDao.getTaskReportByOperationId(operationId);
        for (TaskReportDto dto : taskReportList) {
            taskReportSuggestJson.put(String.format("%s_%s", ABookKeys.TASK_REPORT_SUGGEST, String.valueOf(dto.taskReportLevel)), itemJson);
        }
        Logger.d(TAG, "createTaskReportSuggestJson : " + taskReportSuggestJson.toString());
        Logger.d(TAG, contentPath + "/" + ABookKeys.TASK_REPORT_SUGGEST + ".json");
        // #32926 作業報告画面改善 end
        FileUtil.createFile(contentPath + "/" + ABookKeys.TASK_REPORT_SUGGEST + ".json", taskReportSuggestJson.toString());
    }

    /**
     * プロジェクト情報取得
     *
     * @param operationId
     * @return
     */
    public OperationDto getOperation(Long operationId) {
        return mOperationDao.getOperationJoinContent(operationId);
    }

    /**
     * プロジェクトの作業情報取得
     *
     * @param operationId
     * @return
     */
    public List<TaskDto> getOperationTask(Long operationId) {
        return mTaskDao.selectTaskByOperationId(operationId);
    }

    /**
     * 作業キーで報告リスト取得
     *
     * @param taskKey
     * @return
     */
    public List<TaskReportDto> getTaskReportByTaskKey(String taskKey) {
        return mTaskReportDao.getTaskReportListByTaskKey(taskKey);
    }

    /**
     * 作業一覧取得
     *
     * @param searchWord           検索条件：作業名
     * @param searchStartDateStr   　検索条件：作業日範囲（開始）
     * @param searchEndDateStr     　　検索条件：作業日範囲（終了）
     * @param operationSortingType 　ソート順
     * @return 作業リスト
     */
    public List<OperationDto> getRefreshOperation(String searchWord, String searchStartDateStr, String searchEndDateStr, OperationSortingType operationSortingType, int operationGroupMasterId) {
        List<OperationDto> operationDtoList;
        operationDtoList = mOperationDao.getOperations(searchWord, searchStartDateStr, searchEndDateStr, operationSortingType, operationGroupMasterId);
        for (OperationDto operationDto : operationDtoList) {
            // 作業送信フラグが存在する場合またはホットスポット更新フラグが存在する場合、needSyncFlgをtrueにセット
            if (mTaskReportDao.isExistSendTaskData(operationDto.operationId) || mTaskReportDao.isExistUpdateTargetHotSpotTaskData(operationDto.operationId)) {
                mOperationDao.updateNeedSyncFlg(operationDto.operationId, true);
            }
        }
        return operationDtoList;
    }

    /**
     * 作業関連資料または共通資料一覧取得
     *
     * @return ジャンル毎の共通資料
     */
    public List<CategoryContentDto> getOperationRelatedContent() {
        List<ContentDto> contentDtoList;
        contentDtoList = mContentDao.getCommonContent();


        List<CategoryContentDto> categoryContentDtoArray = new ArrayList<CategoryContentDto>();
        List<ContentDto> saveContentDtoArray = new ArrayList<ContentDto>();
        CategoryContentDto categoryContentDto = new CategoryContentDto();

        for (ContentDto contentDto : contentDtoList) {
            if (contentDto.categoryId == categoryContentDto.categoryId) {
                saveContentDtoArray.add(contentDto);
            } else {
                if (saveContentDtoArray.size() != 0) {
                    categoryContentDto.contentDtoArray = saveContentDtoArray;
                    categoryContentDtoArray.add(categoryContentDto);
                }
                categoryContentDto = new CategoryContentDto();
                saveContentDtoArray = new ArrayList<ContentDto>();
                categoryContentDto.categoryId = contentDto.categoryId;
                categoryContentDto.categoryName = contentDto.categoryNames;
                saveContentDtoArray.add(contentDto);
            }
            if (contentDtoList.get(contentDtoList.size() - 1).equals(contentDto)) {
                categoryContentDto.contentDtoArray = saveContentDtoArray;
                categoryContentDtoArray.add(categoryContentDto);
            }
        }
        return categoryContentDtoArray;
    }

    /**
     * プロジェクト関連資料コンテンツ存在チェック
     *
     * @param operationId
     * @return
     */
    public boolean isExistOperationRelatedContent(Long operationId) {
        boolean rtnVal = false;
        List<ContentDto> contentDtoList = mContentDao.getOperationRelatedContent(operationId);
        if (contentDtoList != null && contentDtoList.size() > 0) {
            rtnVal = true;
        }
        return rtnVal;
    }

    /**
     * 共通資料コンテンツ存在チェック
     *
     * @return
     */
    public boolean isExistCommonContent() {
        boolean rtnVal = false;
        List<ContentDto> contentDtoList = mContentDao.getCommonContent();
        if (contentDtoList != null && contentDtoList.size() > 0) {
            rtnVal = true;
        }
        return rtnVal;
    }

    /**
     * プロジェクト関連資料かチェック
     *
     * @param contentId
     * @param operationId
     * @return
     */
    public boolean isOperationRelatedContent(long contentId, long operationId) {
        return mContentDao.isOperationRelatedContent(contentId, operationId);
    }

    /**
     * 共通資料コンテンツかチェック
     *
     * @param contentId
     * @return
     */
    public boolean isCommonContent(long contentId) {
        return mContentDao.isCommonContent(contentId);
    }

    /**
     * 作業報告履歴データ送信(全体)
     *
     * @param operationId      作業ID
     * @param progressCallback コールバック用
     * @return
     * @throws Exception
     */
    public void sendTaskReportSendData(long operationId, Callback progressCallback) throws Exception {
        sendTaskReportSendData(operationId, null, null, progressCallback);
    }

    /**
     * 作業報告履歴データ送信
     *
     * @param operationId      作業ID
     * @param taskKey          報告・回答のキー
     * @param progressCallback コールバック用
     * @throws Exception
     */
    public void sendTaskReportSendData(long operationId, String taskKey, Integer taskReportLevel, Callback progressCallback) throws Exception {
        OperationDto operationDto = mOperationDao.getOperation(operationId);
        List<TaskReportDto> taskReportDtoList = null;
        int maxProgress = 0;

        if (taskKey == null) {
            taskReportDtoList = mTaskReportDao.getSendableTaskData(operationId);
            maxProgress = 40;
        } else {
            taskReportDtoList = mTaskReportDao.getSendableTaskData(operationId, taskKey, taskReportLevel);
            maxProgress = 100;
        }

        if (taskReportDtoList != null && taskReportDtoList.size() > 0) {

            int allSendCount = mTaskReportSendDao.getSendableTaskReportSendDataCount(operationId, taskKey);
            allSendCount = allSendCount == 0 ? 1 : allSendCount;
            int progress = maxProgress / allSendCount;

            // 送信済みsendIdをまとめて削除するため、リストに追加して最後に削除
            List<Integer> removeTaskReportSendIds = new ArrayList<Integer>();

            for (TaskReportDto taskReportDto : taskReportDtoList) {

                List<TaskReportSendDto> taskReportSendDtoList;
                int lastTaskReportSendId;
                if (operationDto.reportType == Constant.ReportType.RoutineTask) {
                    // 定期点検
                    String reportStartDateStr = DateTimeUtil.toString(taskReportDto.reportStartDate, DateTimeFormat.yyyyMMddHHmmss_hyphen);
                    taskReportSendDtoList = mTaskReportSendDao.getSendableTaskReportSendData(taskReportDto.taskKey, taskReportDto.taskReportId, reportStartDateStr);
                    lastTaskReportSendId = mTaskReportSendDao.getLastTaskReportSendId(taskReportDto.taskKey, taskReportDto.taskReportId, reportStartDateStr);
                } else {
                    taskReportSendDtoList = mTaskReportSendDao.getSendableTaskReportSendData(taskReportDto.taskKey);
                    lastTaskReportSendId = mTaskReportSendDao.getLastTaskReportSendId(taskReportDto.taskKey);
                }

                for (TaskReportSendDto taskReportSendDto : taskReportSendDtoList) {
                    // 履歴なし：元のattachedFileSendFlg
                    // 履歴あり：true、ファイルに変更が無くてもzipファイル作成
                    // 履歴ありで、ローカルの添付ファイルが存在する場合、attachedFileSendFlgをtrueにセットする
                    if (taskReportSendDto.taskReportSendId == lastTaskReportSendId) {
                        if (!StringUtil.isNullOrEmpty(taskReportDto.localAttachedFileName) && operationDto.enableReportHistory == Constant.EnableReportHistory.Enable) {
                            taskReportDto.attachedFileSendFlg = true;
                        }
                    } else {
                        taskReportDto.attachedFileSendFlg = true;
                    }

                    File zipFile = null;
                    String strReportStartDate = DateTimeUtil.toString_yyyyMMddHHmmss_none(taskReportDto.reportStartDate);
                    //連続作業のprocessKeyとphaseNo情報取得
                    TaskDto taskDto = null;
                    if (operationDto.reportType == Constant.ReportType.ReportContinuous) {
                        taskDto = mTaskDao.getTaskByTaskKey(taskReportDto.taskKey);
                    }
                    if (taskReportDto.attachedFileSendFlg) {
                        String fileName = null;
                        zipFile = createAttachedFileForSend(operationId, taskReportDto.taskKey, taskReportSendDto.taskReportSendId, operationDto.reportType == Constant.ReportType.RoutineTask, taskReportDto.taskReportId, strReportStartDate, taskReportDto.taskReportLevel, taskDto);
                        if (zipFile != null) {
                            fileName = FileUtil.getFilenameWithoutExt(zipFile.getName());
                        }
                        taskReportDto.localAttachedFileName = fileName != null ? SecurityUtil.getMd5Hash(fileName) : taskReportDto.localAttachedFileName;
                    }

                    try {
                        AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).sendTaskData(
                                cache.getMemberInfo().sid,
                                operationId,
                                taskReportDto,
                                zipFile,
                                taskReportSendDto.saveDate,
                                operationDto.reportType,
                                taskReportSendDto.hotspotChangeFlg,
                                taskDto
                        );
                        progressCallback.callback(new Integer(progress));
                        removeTaskReportSendIds.add(taskReportSendDto.taskReportSendId);

                        FileUtil.delete(ABVEnvironment.getInstance().getOperationTaskReportSendDirFilePath(operationId, taskReportSendDto.taskKey, taskReportSendDto.taskReportSendId));
                    } catch (AcmsException ex) {
                        if (ex.getCode() == ABVExceptionCode.P_E_ACMS_P005 || ex.getCode() == ABVExceptionCode.P_E_ACMS_P006) {
                            mTaskReportSendDao.deleteBySendId(taskReportSendDto.taskReportSendId);
                            FileUtil.delete(ABVEnvironment.getInstance().getOperationTaskReportSendDirFilePath(operationId, taskReportSendDto.taskKey, taskReportSendDto.taskReportSendId));
                            taskReportDto.dataSendFlg = false;
                            mTaskReportDao.update(taskReportDto);
                        }
                        throw ex;
                    } catch (Exception ex) {
                        throw ex;
                    } finally {
                        if (zipFile != null) {
                            zipFile.delete();
                        }
                    }
                }

                if (taskReportDto.delFlg) {
                    // 物理削除
                    if (operationDto.reportType == Constant.ReportType.ReportContinuous) { //連続作業
                        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskReportDto.taskKey);
                        mTaskDao.deleteTaskData(taskReportDto.taskKey, taskDto.processKey);
                    } else {
                        mTaskDao.deleteTaskData(taskReportDto.taskKey);
                    }

                } else {
                    taskReportDto.dataSendFlg = false;
                    taskReportDto.attachedFileSendFlg = false;
                    mTaskReportDao.update(taskReportDto);
                    // 削除
                    for (Integer removeTaskReportSendId : removeTaskReportSendIds) {
                        mTaskReportSendDao.deleteBySendId(removeTaskReportSendId);
                    }
                }
            }
        }
    }

    /**
     * 作業関連ディレクトリ削除
     *
     * @param operationId     作業ID
     * @param contentId       資料ID
     * @param taskKey         報告・回答のキー
     * @param taskReportLevel 報告・回答のレベル
     */
    public void deleteTaskFileData(long operationId, long contentId, String taskKey, int taskReportLevel) {
        FileUtil.delete(ABVEnvironment.getInstance().getTempTaskDirPath(contentId, taskKey));
        if (taskReportLevel == Constant.TaskReportLevel.ReportType) {
            // 報告
            FileUtil.delete(ABVEnvironment.getInstance().getOperationTaskDirFilePath(operationId, taskKey));
        } else {
            // 報告（回答）
            FileUtil.delete(ABVEnvironment.getInstance().getOperationTaskReportLevelDirPath(operationId, taskKey, taskReportLevel));
        }
    }

    /**
     * プロジェクトの最終更新日と同期フラグの更新
     *
     * @param operationId
     * @param lastEditDate
     */
    public void finishedSyncOperation(Long operationId, Date lastEditDate) {
        OperationDto operationDto = mOperationDao.getOperationJoinContent(operationId);
        operationDto.lastEditDate = lastEditDate;
        operationDto.needSyncFlg = false;
        mOperationDao.update(operationDto);
    }

    /**
     * プロジェクトの同期フラグの更新
     *
     * @param operationId
     */
    public void updateSyncOperation(Long operationId, boolean syncFlg) {
        mOperationDao.updateNeedSyncFlg(operationId, syncFlg);
    }

    /**
     * タスク関連zipファイル取得
     *
     * @param operationId
     * @param taskKey
     * @param taskId
     * @param attachedFileName
     * @param type
     * @return
     * @throws ABVException
     * @throws InterruptedException
     */
    public String getTaskFile(long operationId, String taskKey, long taskId, String attachedFileName, Integer type) throws ABVException, InterruptedException {
        GetTaskFileParameters parameters = new GetTaskFileParameters(cache.getMemberInfo().sid, taskId, type);
        String outputFilePath = ABVEnvironment.getInstance().getOperationTaskDirFilePath(operationId, taskKey) + "/" + attachedFileName + ".zip";
        AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getTaskFile(parameters, operationId, outputFilePath);
        return outputFilePath;
    }

    /**
     * 定期点検関連zipファイル取得
     *
     * @param operationId
     * @param taskKey
     * @param taskId
     * @param taskReportInfoId
     * @param reportStartDate
     * @param attachedFileName
     * @return
     * @throws ABVException
     * @throws InterruptedException
     */
    public String getRoutineTaskReportFile(long operationId, String taskKey, long taskId, int taskReportId, int taskReportInfoId, String reportStartDate, String attachedFileName) throws ABVException, InterruptedException {
        GetTaskFileParameters parameters = new GetTaskFileParameters(cache.getMemberInfo().sid, taskId, Constant.ReportType.Report, taskReportInfoId);
        String outputFilePath = ABVEnvironment.getInstance().getRoutineTaskReportDirFilePath(operationId, taskKey, taskReportId, reportStartDate) + "/" + attachedFileName + ".zip";
        AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getTaskFile(parameters, operationId, outputFilePath);
        return outputFilePath;
    }

    /**
     * シーンの登録
     *
     * @param file
     * @param contentId
     * @throws IOException
     * @throws AcmsException
     */
    public Integer sendScene(File file, Long contentId) throws IOException, AcmsException, NetworkDisconnectedException {
        SceneEntryJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).sceneEntry(cache.getMemberInfo().sid, contentId, file);
        return json.resourceId;
    }

    /**
     * 360コンテンツの登録
     *
     * @param operationId
     * @param contentName
     * @param file
     * @throws IOException
     * @throws AcmsException
     */
    public void sendPanoContent(long operationId, String contentName, File file) throws IOException, ABVException {
        AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).taskContentEntry(cache.getMemberInfo().sid, operationId, contentName, file);
    }

    public void deleteAllOperation() {
        List<OperationDto> operationDtoList = mOperationDao.getAllOperation();
        for (OperationDto operationDto : operationDtoList) {
            FileUtil.delete(ABVEnvironment.getInstance().getOperationDirFile(operationDto.operationId));
        }
        mOperationDao.deleteAll();
    }

    public void setContentCreatingFlg(long operationId) {
        mOperationDao.updateContentCreatingFlg(operationId, true);
    }

    /**
     * 作業の添付ファイルのzip生成
     * @param operationId
     * @param taskKey
     * @param directionFlg
     * @return
     * @throws ZipException
     * @throws NoSuchAlgorithmException
     * @throws IOException
     */
//    private File createAttachedFile(Long operationId, String taskKey, boolean directionFlg, int taskReportLevel) throws ZipException, NoSuchAlgorithmException, IOException {
//        File zipFile;
//        // #32926 作業報告画面改善 start
//        String filePath = ABVEnvironment.getInstance().getOperationTaskReportLevelDirPath(operationId, taskKey, taskReportLevel);
//        // #32926 作業報告画面改善 end
//
//        File fileDir = new File(filePath);
//        if (!fileDir.exists()) {
//            return null;
//        }
//        String[] files = fileDir.list();
//        if (files == null || files.length == 0) {
//            return null;
//        }
//        ArrayList<String> fileList = new ArrayList<String>();
//        for (String dataFile : files) {
//            fileList.add(fileDir.getPath() + StringUtil.Slash + dataFile);
//        }
//        String fileName = UUID.randomUUID().toString();
//        zipFile = new File(filePath + StringUtil.Slash + fileName + ".zip");
//        if (zipFile.exists()) {
//            // zipファイルが存在する時は削除して、新たに作成
//            boolean result = zipFile.delete();
//            if (result) {
//                FileUtil.saveZip(zipFile.getPath(), null, fileList.toArray(new String[fileList.size()]));
//            } else {
//                // 削除失敗
//                Logger.e(TAG, "delete zip file failed");
//                return null;
//            }
//        } else {
//            FileUtil.saveZip(zipFile.getPath(), null, fileList.toArray(new String[fileList.size()]));
//        }
//        return zipFile;
//    }

    /**
     * 作業報告送信の添付ファイルのzip生成
     *
     * @param operationId
     * @param taskKey
     * @param taskReportSendId
     * @param routineFlag
     * @param reportId
     * @param reportStartDate
     * @return
     * @throws ZipException
     * @throws NoSuchAlgorithmException
     * @throws IOException
     */
    private File createAttachedFileForSend(Long operationId, String taskKey, int taskReportSendId,
                                           boolean routineFlag, int reportId, String reportStartDate,
                                           int taskReportLevel, TaskDto taskDto) throws ZipException, NoSuchAlgorithmException, IOException {
        File zipFile;
        String filePath = ABVEnvironment.getInstance().getOperationTaskReportSendDirFilePath(operationId, taskKey, taskReportSendId);
        if (taskDto != null) {
            filePath = ABVEnvironment.getInstance().getOperationContinuousTaskReportSendDirFilePath(operationId, taskKey, taskReportSendId, taskDto.processKey, taskDto.phaseNo);
        }

        File fileDir = new File(filePath);
        if (!fileDir.exists()) {
            return null;
        }
        String[] files = fileDir.list();
        if (files == null || files.length == 0) {
            return null;
        }
        ArrayList<String> fileList = new ArrayList<String>();
        for (String dataFile : files) {
            fileList.add(fileDir.getPath() + StringUtil.Slash + dataFile);
        }
        String fileName = UUID.randomUUID().toString();
        zipFile = new File(filePath + StringUtil.Slash + fileName + ".zip");
        if (zipFile.exists()) {
            // zipファイルが存在する時は削除して、新たに作成
            boolean result = zipFile.delete();
            if (result) {
                FileUtil.saveZip(zipFile.getPath(), null, fileList.toArray(new String[fileList.size()]));
            } else {
                // 削除失敗
                Logger.e(TAG, "delete zip file failed");
                return null;
            }
        } else {
            FileUtil.saveZip(zipFile.getPath(), null, fileList.toArray(new String[fileList.size()]));
        }
        return zipFile;
    }

    /**
     * ディレクトリのファイルがattachedFileNamesに含まれないファイル削除
     *
     * @param dirPath
     * @param attachedFileNames
     */
    private void deleteDifferentialFile(String dirPath, List<String> attachedFileNames) {
        File file = new File(dirPath);
        if (file.exists()) {
            String[] fileList = file.list();
            if (fileList != null) {
                for (String localFileName : fileList) {
                    if (!attachedFileNames.contains(localFileName)) {
                        FileUtil.delete(dirPath + "/" + localFileName);
                    }
                }
            }
        }
    }

    /**
     * 作業の開始日のチェック
     *
     * @param operationId
     * @return
     */
    public String getRoutineTaskOperationAvailableDateStr(Long operationId) {
        OperationDto operationDto = getOperation(operationId);
        Date currDate = DateTimeUtil.getCurrentSqlDate();
        if (DateTimeUtil.isDateBefore(currDate, operationDto.operationStartDate)) {
            return DateTimeUtil.toString(operationDto.operationStartDate, DateTimeFormat.yyyyMMdd_slash);
        }
        return null;
    }

    /**
     * 作業報告の取得
     *
     * @param taskKey         報告・回答キー
     * @param taskReportlevel 作業レベル
     * @return TaskReportDto 作業報告
     */
    public TaskReportDto getTaskReport(String taskKey, int taskReportlevel) {
        return mTaskReportDao.getTaskReport(taskKey, taskReportlevel);
    }

    /**
     * suggestJsonデータをtaskReportItemテーブルに登録
     *
     * @param taskKey
     * @param reportLevel
     * @param suggestJson
     */
    public void registTaskReportItem(String taskKey, int reportLevel, JSONObject suggestJson) {
        if (suggestJson == null) {
            // suggestJsonがnullの場合、return
            return;
        }
        Iterator itemKeys = suggestJson.keys();
        while (itemKeys.hasNext()) {
            String itemKey = (String) itemKeys.next();
            TaskReportItemsDto taskReportItemsDto = mTaskReportItemsDao.getTaskReportItem(taskKey, reportLevel, itemKey);
            if (taskReportItemsDto != null) {
                // 更新
                String newValue = suggestJson.getString(taskReportItemsDto.itemKey);
                if (!newValue.equals(taskReportItemsDto.inputValue)) {
                    // 値が異なる場合のみ、更新する
                    taskReportItemsDto.inputValue = newValue;
                    mTaskReportItemsDao.updateTaskReportItems(taskReportItemsDto);
                }
            } else {
                // 登録
                taskReportItemsDto = new TaskReportItemsDto();
                taskReportItemsDto.taskKey = taskKey;
                taskReportItemsDto.itemKey = itemKey;
                taskReportItemsDto.taskReportLevel = reportLevel;
                try {
                    taskReportItemsDto.inputValue = suggestJson.getString(itemKey);
                } catch (JSONException e) {
                    // 値がStringではない場合、無視する
                    continue;
                }
                mTaskReportItemsDao.insertTaskReportItems(taskReportItemsDto);
            }
        }
    }

    /**
     * t_taskテーブルに登録（報告データ）
     * データが存在しないと登録する
     *
     * @param taskKey
     * @param operationId
     * @param hotSpotInfo
     */
    public void registTaskData(String taskKey, Long operationId, String hotSpotInfo, JSONObject taskJson, String processKey, Integer phaseNo) {
        TaskDto taskDto = mTaskDao.getTaskByTaskKey(taskKey);
        String taskCode = null;
        Integer taskStatus = null;
        String taskName = null;
        if (taskJson != null) {

            Iterator taskKeys = taskJson.keys();
            while (taskKeys.hasNext()) {
                String itemKey = (String) taskKeys.next();
                // 作業コード取得
                if (itemKey.startsWith("q_1_")) {
                    taskCode = taskJson.getString(itemKey);
                }

                // 報告名
                if (itemKey.startsWith("q_2_")) {
                    taskName = taskJson.getString(itemKey);
                }

                // 報告状況取得
                if (itemKey.startsWith("q_3_")) {
                    taskStatus = taskJson.getInt(itemKey);
                }
            }
        }

        if (taskDto != null) {
            // 更新
            if (taskDto.taskCode != null) {
                taskDto.taskCode = taskCode;
            }

            if (hotSpotInfo != null) {
                taskDto.taskHotSpotInfo = hotSpotInfo;
            }
            if (taskStatus != null) {
                taskDto.taskStatus = taskStatus;
            }
            if (taskName != null) {
                taskDto.taskName = taskName;
            }
            taskDto.processKey = processKey;
            taskDto.phaseNo = phaseNo;
            mTaskDao.update(taskDto);
        } else {
            // 登録
            taskDto = new TaskDto();
            taskDto.taskCode = taskCode;
            taskDto.taskKey = taskKey;
            taskDto.operationId = operationId;
            if (hotSpotInfo != null) {
                taskDto.taskHotSpotInfo = hotSpotInfo;
            }
            if (taskStatus != null) {
                taskDto.taskStatus = taskStatus;
            }
            if (taskName != null) {
                taskDto.taskName = taskName;
            }
            taskDto.processKey = processKey;
            taskDto.phaseNo = phaseNo;
            mTaskDao.insert(taskDto);
        }
    }

    /**
     * 連続作業の承認情報を取得
     *
     * @param operationId 作業ID
     * @throws NetworkDisconnectedException
     * @throws ABVException
     * @throws IOException
     */
    public void getProcessData(Long operationId) throws NetworkDisconnectedException, ABVException {
        GetOperationDataParameters param = new GetOperationDataParameters(ABVDataCache.getInstance().getMemberInfo().sid, operationId);
        ProcessDataJSON json = AcmsClient.getInstance(ABVDataCache.getInstance().getUrlPath(), ABVEnvironment.getInstance().networkAdapter).getProcessData(param);
        for (TaskDto taskDto : json.taskDtoList) {
            for (TaskDto phaseTaskDto : taskDto.phaseList) {
                //Taskデータ更新（phaseStatus, processStatus）
                mTaskDao.updateStatus(taskDto.processKey, phaseTaskDto.phaseNo, taskDto.processStatus, phaseTaskDto.phaseStatus);
                for (TaskReportApprovalDto taskReportApprovalDto : phaseTaskDto.taskReportApprovalDtoList) {
                    taskReportApprovalDto.processKey = taskDto.processKey;
                    taskReportApprovalDto.phaseNo = phaseTaskDto.phaseNo;
                    //TaskReportApprovalデータ追加・更新
                    if (mTaskReportApprovalDao.selectTaskReportApprovalByProcessKey(taskDto.processKey, phaseTaskDto.phaseNo, taskReportApprovalDto.approvalNo) != null) {
                        mTaskReportApprovalDao.update(taskReportApprovalDto);
                    } else {
                        mTaskReportApprovalDao.insert(taskReportApprovalDto);
                    }
                }

            }
        }
    }

    public String getProcessDataAndCreateJson(Long operationId, String contentPath, boolean routineTaskReportFlg) throws NetworkDisconnectedException, ABVException, IOException {
        GetOperationDataParameters param = new GetOperationDataParameters(ABVDataCache.getInstance().getMemberInfo().sid, operationId);
        ProcessDataJSON json = AcmsClient.getInstance(ABVDataCache.getInstance().getUrlPath(), ABVEnvironment.getInstance().networkAdapter).getProcessData(param);
        for (TaskDto taskDto : json.taskDtoList) {
            for (TaskDto phaseTaskDto : taskDto.phaseList) {
                //Taskデータ更新（phaseStatus, processStatus）
                mTaskDao.updateStatus(taskDto.processKey, phaseTaskDto.phaseNo, taskDto.processStatus, phaseTaskDto.phaseStatus);
                for (TaskReportApprovalDto taskReportApprovalDto : phaseTaskDto.taskReportApprovalDtoList) {
                    taskReportApprovalDto.processKey = taskDto.processKey;
                    taskReportApprovalDto.phaseNo = phaseTaskDto.phaseNo;
                    //TaskReportApprovalデータ追加・更新
                    if (mTaskReportApprovalDao.selectTaskReportApprovalByProcessKey(taskDto.processKey, phaseTaskDto.phaseNo, taskReportApprovalDto.approvalNo) != null) {
                        mTaskReportApprovalDao.update(taskReportApprovalDto);
                    } else {
                        mTaskReportApprovalDao.insert(taskReportApprovalDto);
                    }
                }

            }
        }
        return getProcessInfoJson(operationId, contentPath);
    }

    /**
     * 工程情報を全部削除する。（全削除ボタンタップ時）
     *
     * @param operationId 作業ID
     * @param processKey  工程キー
     * @throws Exception    例外
     * @throws ABVException 例外
     */
    public void sendDeleteProcess(final long operationId, final String processKey) throws Exception, ABVException {
        DeleteProcessParameters parameters = new DeleteProcessParameters(ABVDataCache.getInstance().getMemberInfo().sid, operationId, processKey);
        AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).sendDeleteProcess(parameters);
    }

    /**
     * 同期処理で、削除された工程情報を
     *
     * @param operationId
     * @throws Exception
     * @throws AcmsException
     */
    public boolean deleteProcess(long operationId) throws Exception {
        List<TaskDto> taskDtoList = mTaskDao.selectTaskByOperationId(operationId, true);
        List<String> sendedProcessKeyList = new ArrayList<String>();
        boolean isSuccess = true;
        try {
            for (TaskDto taskDto : taskDtoList) {
                if (sendedProcessKeyList.contains(taskDto.processKey)) {
                    continue;
                }
                sendDeleteProcess(taskDto.operationId, taskDto.processKey);
                sendedProcessKeyList.add(taskDto.processKey);
            }
        } catch (AcmsException ex) {
            isSuccess = false;
            Logger.e(TAG, "deleteProcess ex = ", ex);
        } catch (Exception e) {
            isSuccess = false;
            Logger.e(TAG, "deleteProcess e = ", e);
        }
        return isSuccess;
    }

    public String createOperationListJson(List<OperationDto> operationList, String filePath) {
        JSONObject operationListJsonObject = new JSONObject();
        try {
            JSONArray operationJsonArray = new JSONArray();
            for (int i = 0; i < operationList.size(); i++) {
                JSONObject operationJson = new JSONObject();
                operationJson.put(ABookKeys.OPERATION_ID, operationList.get(i).operationId);
                operationJson.put(ABookKeys.OPERATION_NAME, operationList.get(i).operationName);
                operationJson.put(ABookKeys.OPERATION_DESCRIPTIONS, operationList.get(i).operationDescriptions);
                operationJson.put(ABookKeys.OPERATION_TYPE, operationList.get(i).operationType);
                operationJson.put(ABookKeys.OPERATION_START_DATE, DateTimeUtil.toString(operationList.get(i).operationStartDate, DateTimeFormat.yyyyMMdd_slash));
                operationJson.put(ABookKeys.OPERATION_END_DATE, DateTimeUtil.toString(operationList.get(i).operationEndDate, DateTimeFormat.yyyyMMdd_slash));
                operationJson.put(ABookKeys.LAST_EDIT_DATE, DateTimeUtil.toString(operationList.get(i).lastEditDate, DateTimeFormat.yyyyMMdd_slash));
                operationJson.put(ABookKeys.NEED_SYNC_FLG, operationList.get(i).needSyncFlg);
                operationJson.put(ABookKeys.REPORT_TYPE, operationList.get(i).reportType);
                operationJson.put(ABookKeys.REPORT_CYCLE, operationList.get(i).reportCycle);
                operationJson.put(ABookKeys.ENABLE_REPORT_UPDATE, operationList.get(i).enableReportUpdate);
                operationJson.put(ABookKeys.ENABLE_REPORT_HISTORY, operationList.get(i).enableReportHistory);
                operationJson.put(ABookKeys.ENABLE_ADD_REPORT, operationList.get(i).enableAddReport);
                operationJson.put(ABookKeys.QUICK_REPORT, operationList.get(i).quickReport);
                operationJson.put(ABookKeys.PERMIT_CODE, operationList.get(i).permitCode);
                operationJson.put(ABookKeys.PERMIT_CODE_REQUIRED_FLG, operationList.get(i).permitCodeRequiredFlg);
                operationJson.put(ABookKeys.OPERATION_OPEN_DATE, DateTimeUtil.toString(operationList.get(i).operationOpenDate, DateTimeFormat.yyyyMMddHHmmss_hyphen));
                operationJson.put(ABookKeys.DISPLAY_PERMISSION_FLG, operationList.get(i).displayPermissionFlg);
                operationJson.put(ABookKeys.CONTENT_ID, operationList.get(i).contentId);
                //pickup newReportFlg, load processList and warning json file
                operationJson.put(ABookKeys.NEW_REPORT_FLG, operationList.get(i).newReportFlg);
                String processListJsonPath = ABVEnvironment.getInstance().getPickupProcessListOperationFilePath(operationList.get(i).operationId);
                String processListString = FileUtil.readTextFile(processListJsonPath);
                if (!StringUtil.isNullOrEmpty(processListString)) {
                    operationJson.put(ABookKeys.OPERATION_PROCESS_LIST, new JSONArray(processListString));
                }
                String warningReportJsonPath = ABVEnvironment.getInstance().getPickupWarningReportListOperationFilePath(operationList.get(i).operationId);
                String warningReportString = FileUtil.readTextFile(warningReportJsonPath);
                if (!StringUtil.isNullOrEmpty(warningReportString)) {
                    operationJson.put(ABookKeys.OPERATION_WARNING_REPORT_LIST, new JSONArray(warningReportString));
                }

                operationJsonArray.put(operationJson);
            }
            operationListJsonObject.put(ABookKeys.OPERATION_LIST, operationJsonArray);
            if (!StringUtil.isNullOrEmpty(filePath)) {
                FileUtil.createFile(filePath + "/operationList.json", operationListJsonObject.toString());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return operationListJsonObject.toString();
    }
    /**
     * サーバから取得したタスクデータとローカルDBのタスクデータを比較
     * @param localTaskDto ローカルDBのタスクデータ
     * @param serverTaskDto サーバ側から取得したタスクデータ
     * @return true:変更なし、false:変更あり
     */
    public boolean taskDataSameCheck(TaskDto localTaskDto, TaskDto serverTaskDto) {
        if (!Objects.equals(localTaskDto.taskId, serverTaskDto.taskId)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.taskCode, serverTaskDto.taskCode)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.taskStatus, serverTaskDto.taskStatus)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.taskHotSpotInfo, serverTaskDto.taskHotSpotInfo)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.delFlg, serverTaskDto.delFlg)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.taskName, serverTaskDto.taskName)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.phaseNo, serverTaskDto.phaseNo)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.phaseStatus, serverTaskDto.phaseStatus)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.processStatus, serverTaskDto.processStatus)) {
            return false;
        }
        if (!Objects.equals(localTaskDto.processKey, serverTaskDto.processKey)) {
            return false;
        }

        return true;
    }
    /**
     * サーバから取得したタスク報告データとローカルDBのタスク報告データを比較
     * @param localTaskReportDto ローカルDBの報告データ
     * @param serverTaskReportDto サーバ側から取得した報告データ
     * @return true:変更なし、false:変更あり
     */
    public boolean taskReportDataSameCheck(TaskReportDto localTaskReportDto, TaskReportDto serverTaskReportDto) {
        if (!Objects.equals(localTaskReportDto.attachedFileName, serverTaskReportDto.attachedFileName)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.attachedFileSendFlg, serverTaskReportDto.attachedFileSendFlg)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.dataSendFlg, serverTaskReportDto.dataSendFlg)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.delFlg, serverTaskReportDto.delFlg)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.enableReport, serverTaskReportDto.enableReport)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.jsonData, serverTaskReportDto.jsonData)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.localAttachedFileName, serverTaskReportDto.localAttachedFileName)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.localSavedFlg, serverTaskReportDto.localSavedFlg)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.reportStartDate, serverTaskReportDto.reportStartDate)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.reportEndDate, serverTaskReportDto.reportEndDate)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.taskReportInfoId, serverTaskReportDto.taskReportInfoId)) {
            return false;
        }
        if (!Objects.equals(localTaskReportDto.taskReportLevel, serverTaskReportDto.taskReportLevel)) {
            return false;
        }
        return true;
    }

    /**
     * DBから取得した定期点検タスク報告配列から該当するタスク報告を取得する
     *
     * @param localTaskReportDtoList DBから取得したタスク報告配列
     * @param taskKey タスクキー
     * @param taskReportId タスク報告ID
     * @param reportStartDate 定期点検開始日
     * @return TaskReportDto タスク報告
     */
    public TaskReportDto getLocalRoutineTaskReportDtoInList(List<TaskReportDto> localTaskReportDtoList, String taskKey, int taskReportId, Date reportStartDate) {
        TaskReportDto taskReportDto = null;
        for (TaskReportDto localTaskReportDto : localTaskReportDtoList) {
            if (localTaskReportDto.taskKey.equals(taskKey) && localTaskReportDto.taskReportId == taskReportId &&  localTaskReportDto.reportStartDate.equals(reportStartDate)) {
                taskReportDto =  localTaskReportDto;
                continue;
            }
        }
        return taskReportDto;
    }

    /**
     * ローカルDBから取得したタスク配列の中に該当するタスクDTO取得
     * @param listDto　リストDto
     * @param taskKey　タスクキー
     * @return
     */
    public TaskDto getLocalTaskInList(List<TaskDto> listDto, String taskKey) {
        for (TaskDto lDto : listDto) {
            if (lDto.taskKey.equals(taskKey)) {
                return lDto;
            }
        }
        return null;
    }

    /**
     * check user has permission to create process
     * @param phaseNo
     * @param contentPath
     * @return
     */
    public boolean checkProcessPermission(String phaseNo, String contentPath) {
        boolean hasPermission = false;
        try {
            Integer[] userGroupIds = mGroupLogic.getServerUserGroupIds(cache.getMemberInfo().sid);
            if (userGroupIds == null) {
                return hasPermission;
            }
            // check userGroupId in phaseGroupId
            String jsonPath = contentPath.replaceAll(ABookKeys.PANO_IMAGE,ABookKeys.PROCESS_LIST).replaceAll(ABookKeys.TASK_PDF, ABookKeys.PROCESS_LIST) + "/" + ABookKeys.PHASE_INFO_JSON;
            PhaseInfoJSON phaseInfoJSON = new PhaseInfoJSON(FileUtil.readTextFile(jsonPath));
            List<Integer> phaseGroupIds = phaseInfoJSON.getPhaseGroupIds(phaseNo);
            for (int i = 0; i < userGroupIds.length; i++) {
                if (phaseGroupIds.contains(userGroupIds[i])) {
                    hasPermission = true;
                    break;
                }
            }
        } catch (Exception e) {}
        return hasPermission;
    }
}