package jp.agentec.abook.abv.bl.data.dao;

import java.util.ArrayList;
import java.util.List;

import jp.agentec.abook.abv.bl.common.db.Cursor;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.dto.ChatGroupDto;
import jp.agentec.abook.abv.bl.dto.GroupDto;
import jp.agentec.adf.util.CollectionUtil;
import jp.agentec.adf.util.StringUtil;

public class ChatGroupDao extends AbstractCommunicationDao {
	private static final String TAG = "ChatGroupDao";

	/*package*/ ChatGroupDao() {
	}

	@Override
	protected ChatGroupDto convert(Cursor cursor) {
		ChatGroupDto dto = new ChatGroupDto();

		int column = cursor.getColumnIndex("group_id");
		if (column != -1) {
			dto.groupId = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("parent_group_id");
		if (column != -1) {
			dto.parentGroupId = cursor.getInt(column);
		}
		column = cursor.getColumnIndex("group_name");
		if (column != -1) {
			dto.groupName = cursor.getString(column);
		}
		column = cursor.getColumnIndex("favorite_register_date");
		if (column != -1) {
			dto.favoriteRegisterDate = cursor.getString(column);
		}
		return dto;
	}

	public List<ChatGroupDto> getAllGroups() {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT group_id ");
		sql.append("        ,parent_group_id ");
		sql.append("        ,group_name ");
		sql.append("        ,favorite_register_date ");
		sql.append(" FROM m_chat_group");
		return rawQueryGetDtoList(sql.toString(), null, ChatGroupDto.class);
	}

	public List<ChatGroupDto> getUserGroups(Integer shopMemberId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT cg.group_id ");
		sql.append("        ,cg.parent_group_id ");
		sql.append("        ,cg.group_name ");
		sql.append("        ,cg.favorite_register_date ");
		sql.append(" FROM m_chat_group cg INNER JOIN r_shop_member_group rsmg ON cg.group_id = rsmg.group_id");
		sql.append(" WHERE rsmg.shop_member_id = ?");
		return rawQueryGetDtoList(sql.toString(), new String[] {"" + shopMemberId}, ChatGroupDto.class);
	}

	public List<String> getMyGroupPathList() {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("     CASE WHEN (grandparentgroup.group_name IS NOT NULL) THEN grandparentgroup.group_name || ' / ' || parentgroup.group_name || ' / ' || mygroup.group_name ");
		sql.append("          WHEN (parentgroup.group_name IS NOT NULL) THEN parentgroup.group_name || ' / ' || mygroup.group_name ");
		sql.append("          ELSE mygroup.group_name ");
		sql.append("     END AS group_path_list ");
		sql.append(" FROM m_chat_group mygroup ");
		sql.append(" LEFT JOIN m_chat_group parentgroup ON mygroup.parent_group_id = parentgroup.group_id ");
		sql.append(" LEFT JOIN m_chat_group grandparentgroup ON parentgroup.parent_group_id = grandparentgroup.group_id ");
		sql.append(" WHERE mygroup.group_id IN (SELECT group_id FROM m_shop_member sm INNER JOIN r_shop_member_group rmg ON sm.shop_member_id = rmg.shop_member_id WHERE sm.self_flg = 1)");
		return rawQueryGetStringList(sql.toString(), null);
	}

	public List<ChatGroupDto> getUserGroupPathList(Integer shopMemberId) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT ");
		sql.append("     CASE WHEN (grandparentgroup.group_name IS NOT NULL) THEN grandparentgroup.group_name || ' / ' || parentgroup.group_name || ' / ' || usergroup.group_name ");
		sql.append("          WHEN (parentgroup.group_name IS NOT NULL) THEN parentgroup.group_name || ' / ' || usergroup.group_name ");
		sql.append("          ELSE usergroup.group_name ");
		sql.append("     END AS group_path, usergroup.group_id ");
		sql.append(" FROM m_chat_group usergroup ");
		sql.append(" LEFT JOIN m_chat_group parentgroup ON usergroup.parent_group_id = parentgroup.group_id ");
		sql.append(" LEFT JOIN m_chat_group grandparentgroup ON parentgroup.parent_group_id = grandparentgroup.group_id ");
		sql.append(" WHERE usergroup.group_id IN  ");
		sql.append("                                (SELECT group_id  ");
		sql.append("                                 FROM m_shop_member sm INNER JOIN r_shop_member_group rsmg ON sm.shop_member_id = rsmg.shop_member_id ");
		sql.append("                                 WHERE sm.shop_member_id = ? ) ");
		return rawQueryGetDtoList(sql.toString(), new String[] { "" + shopMemberId}, ChatGroupDto.class);
	}

	public void insertGroup(int groupId, int parentGroupId, String groupName) {
		StringBuffer sql = new StringBuffer();
		sql.append(" INSERT OR IGNORE INTO m_chat_group (group_id, parent_group_id, group_name) ");
		sql.append(" VALUES (?,?,?) ");
		try {
			beginTransaction();
			insert(sql.toString(), new Object[] { groupId, parentGroupId, groupName });
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertChatGroup failed.", e);
			throw new RuntimeException(e);
		}
		Logger.v(TAG, "sql=%s", sql);
	}


	public void updateGroup(int groupId, int parentGroupId, String groupName) {
		StringBuffer sql = new StringBuffer();
		sql.append(" UPDATE m_chat_group ");
		sql.append("    SET group_name = ? ");
		sql.append("  WHERE group_id = ? ");
		try {
			beginTransaction();
			update(sql.toString(), new Object[] { groupName, groupId });
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertChatGroup failed.", e);
			throw new RuntimeException(e);
		}
		Logger.v(TAG, "sql=%s", sql);
	}

	public void deleteGroup(ChatGroupDto dto, int defaultGroupId) {
		String[] keyValues = dto.getKeyValues();
		StringBuffer sql = new StringBuffer();
		try{
			beginTransaction();
			delete("r_shop_member_group", "group_id=?", keyValues);
			delete("m_chat_group", "group_id=?", keyValues);
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("deleteChatGroup failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}

	public void deleteChatGroup() {
		try {
			beginTransaction();
			delete("r_shop_member_group", null, null);
			delete("m_chat_group", null, null);
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("deleteChatMessage failed.", e);
			throw new RuntimeException(e);
		} finally {
		}
	}

	public ChatGroupDto getGroup(int groupId) {
		return rawQueryGetDto("SELECT * FROM m_chat_group WHERE group_id = " + groupId, null, ChatGroupDto.class);
	}

	public List<ChatGroupDto> getFavoriteGroup() {
		return rawQueryGetDtoList("SELECT * FROM m_chat_group WHERE favorite_register_date IS NOT NULL ORDER BY favorite_register_date ", null, ChatGroupDto.class);
	}

	public List<ChatGroupDto> getGroupByName(String[] keywords) {
		StringBuffer sql = new StringBuffer();
		sql.append(" SELECT *  ");
		sql.append(" FROM m_chat_group ");
		ArrayList<String> whereSqlList = new ArrayList<String>();
		for (String keyword : keywords) {
			if (StringUtil.isNullOrEmpty(keyword)) {
				continue;
			}
			String whereSql = "group_name LIKE '%"+keyword+"%'";
			whereSqlList.add(whereSql);
		}
		if (CollectionUtil.isNotEmpty(whereSqlList)) {
			sql.append("WHERE " + StringUtil.join(" AND ", whereSqlList));
		}
		sql.append(" ORDER BY group_name ASC ");
		return rawQueryGetDtoList(sql.toString(), null, ChatGroupDto.class);
	}

	public List<ChatGroupDto> getGroupTree(Integer groupId) {
		List<ChatGroupDto> list;
		StringBuffer sql = new StringBuffer();
		sql.append(" WITH group_loop AS ");
		sql.append(" ( ");
		sql.append("     SELECT 1 rownum, group_id, group_name, parent_group_id FROM m_chat_group WHERE group_id = ? ");
		sql.append("     UNION ALL ");
		sql.append("     SELECT rownum+1 rownum, parent.group_id, parent.group_name, parent.parent_group_id ");
		sql.append("     FROM group_loop inner JOIN m_chat_group parent ON parent.group_id = group_loop.parent_group_id ");
		sql.append("  ) ");
		sql.append("  SELECT * FROM group_loop ORDER BY group_loop.rownum DESC ");

		list = rawQueryGetDtoList(sql.toString(), new String[]{""+ groupId}, ChatGroupDto.class);
		return list;
	}

	public List<ChatGroupDto> getGroupChildList(int parentId) {
		List<ChatGroupDto> list;
		list = rawQueryGetDtoList("SELECT * FROM m_chat_group mg WHERE (mg.parent_group_id =  ?) ORDER BY group_name", new String[]{""+ parentId}, ChatGroupDto.class);
		return list;
	}

	public ChatGroupDto getRootGroup() {
		return rawQueryGetDto("SELECT * FROM m_chat_group mg WHERE (mg.parent_group_id =  0)",null, ChatGroupDto.class);
	}

	public ChatGroupDto getParentGroup(Integer childGroupId) {
		return rawQueryGetDto("SELECT parent.* FROM m_chat_group parent INNER JOIN m_chat_group child ON child.parent_group_id = parent.group_id WHERE child.group_id = ?", new String[]{""+ childGroupId}, ChatGroupDto.class);
	}

	public boolean isExistParent(int baseId) {
		return rawQueryGetInt("SELECT * FROM m_chat_group WHERE parent_group_id = " + baseId, null) > 0;
	}

	public List<ChatGroupDto> getMyGroups() {
		return rawQueryGetDtoList("SELECT * FROM m_chat_group WHERE group_id IN (SELECT group_id FROM m_shop_member sm INNER JOIN r_shop_member_group rmg ON sm.shop_member_id = rmg.shop_member_id WHERE sm.self_flg = 1) ORDER BY group_name ASC", null, ChatGroupDto.class);
	}

	public void insertGroupList(List<ChatGroupDto> groupList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" INSERT INTO m_chat_group (group_id, parent_group_id, group_name) ");
		sql.append(" VALUES (?,?,?) ");
		try {
			beginTransaction();
			for (ChatGroupDto groupDto : groupList) {
				insert(sql.toString(), new Object[] { groupDto.groupId, groupDto.parentGroupId, groupDto.groupName});
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("insertGroupList failed.", e);
			throw new RuntimeException(e);
		}
	}


	public void updateGroupList(List<ChatGroupDto> groupList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" UPDATE m_chat_group ");
		sql.append("    SET group_name = ? ");
		sql.append("      , parent_group_id = ? ");
		sql.append("  WHERE group_id = ? ");
		try {
			beginTransaction();
			for (ChatGroupDto groupDto : groupList) {
				update(sql.toString(), new Object[] { groupDto.groupName, groupDto.parentGroupId, groupDto.groupId });
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("updateGroupList failed.", e);
			throw new RuntimeException(e);
		}
	}

	public void updateFavoriteGroupList(List<ChatGroupDto> groupList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" UPDATE m_chat_group ");
		sql.append(" SET favorite_register_date = ? ");
		sql.append(" WHERE group_id = ? ");
		try {
			beginTransaction();
			for (ChatGroupDto groupDto : groupList) {
				update(sql.toString(), new Object[] { groupDto.favoriteRegisterDate, groupDto.groupId });
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("updateFavoriteGroupList failed.", e);
			throw new RuntimeException(e);
		}
	}

	public void deleteGroupList(List<ChatGroupDto> groupList) {
		StringBuffer sql = new StringBuffer();
		sql.append(" DELETE FROM m_chat_group ");
		sql.append("  WHERE group_id = ? ");
		try {
			beginTransaction();
			for (ChatGroupDto groupDto : groupList) {
				update(sql.toString(), new Object[] { groupDto.groupId });
				update("DELETE FROM r_shop_member_group WHERE group_id = ? ", new Object[] { groupDto.groupId });
			}
			commit();
		} catch (Exception e) {
			rollback();
			Logger.e("deleteChatGroupList failed.", e);
			throw new RuntimeException(e);
		}
	}
}
