/*
 * Decompiled with CFR 0.152.
 */
package org.chromium.media;

import android.annotation.TargetApi;
import android.media.AudioTrack;
import android.media.MediaCodec;
import android.media.MediaCrypto;
import android.media.MediaFormat;
import android.os.Build;
import android.os.Bundle;
import android.view.Surface;
import java.nio.ByteBuffer;
import org.chromium.base.Log;
import org.chromium.base.annotations.CalledByNative;
import org.chromium.base.annotations.JNINamespace;
import org.chromium.base.annotations.MainDex;
import org.chromium.media.MediaCodecUtil;

@JNINamespace(value="media")
class MediaCodecBridge {
    private static final String TAG = "cr_media";
    private static final int MEDIA_CODEC_OK = 0;
    private static final int MEDIA_CODEC_DEQUEUE_INPUT_AGAIN_LATER = 1;
    private static final int MEDIA_CODEC_DEQUEUE_OUTPUT_AGAIN_LATER = 2;
    private static final int MEDIA_CODEC_OUTPUT_BUFFERS_CHANGED = 3;
    private static final int MEDIA_CODEC_OUTPUT_FORMAT_CHANGED = 4;
    private static final int MEDIA_CODEC_INPUT_END_OF_STREAM = 5;
    private static final int MEDIA_CODEC_OUTPUT_END_OF_STREAM = 6;
    private static final int MEDIA_CODEC_NO_KEY = 7;
    private static final int MEDIA_CODEC_ABORT = 8;
    private static final int MEDIA_CODEC_ERROR = 9;
    private static final long MAX_PRESENTATION_TIMESTAMP_SHIFT_US = 100000L;
    private static final int PCM16_BYTES_PER_SAMPLE = 2;
    private static final String KEY_CROP_LEFT = "crop-left";
    private static final String KEY_CROP_RIGHT = "crop-right";
    private static final String KEY_CROP_BOTTOM = "crop-bottom";
    private static final String KEY_CROP_TOP = "crop-top";
    private ByteBuffer[] mInputBuffers;
    private ByteBuffer[] mOutputBuffers;
    private MediaCodec mMediaCodec;
    private AudioTrack mAudioTrack;
    private byte[] mPendingAudioBuffer;
    private boolean mFlushed;
    private long mLastPresentationTimeUs;
    private String mMime;
    private boolean mAdaptivePlaybackSupported;

    private MediaCodecBridge(MediaCodec mediaCodec, String mime, boolean adaptivePlaybackSupported) {
        assert (mediaCodec != null);
        this.mMediaCodec = mediaCodec;
        this.mPendingAudioBuffer = null;
        this.mMime = mime;
        this.mLastPresentationTimeUs = 0L;
        this.mFlushed = true;
        this.mAdaptivePlaybackSupported = adaptivePlaybackSupported;
    }

    @CalledByNative
    private static MediaCodecBridge create(String mime, boolean isSecure, int direction, boolean requireSoftwareCodec) {
        MediaCodecUtil.CodecCreationInfo info = new MediaCodecUtil.CodecCreationInfo();
        try {
            if (direction == 1) {
                info.mediaCodec = MediaCodec.createEncoderByType((String)mime);
                info.supportsAdaptivePlayback = false;
            } else {
                info = MediaCodecUtil.createDecoder(mime, isSecure, requireSoftwareCodec);
            }
        }
        catch (Exception e) {
            Log.e(TAG, "Failed to create MediaCodec: %s, isSecure: %s, direction: %d", mime, isSecure, direction, e);
        }
        if (info.mediaCodec == null) {
            return null;
        }
        return new MediaCodecBridge(info.mediaCodec, mime, info.supportsAdaptivePlayback);
    }

    @CalledByNative
    private void release() {
        try {
            String codecName = "unknown";
            if (Build.VERSION.SDK_INT >= 18) {
                codecName = this.mMediaCodec.getName();
            }
            Log.w(TAG, "calling MediaCodec.release() on " + codecName, new Object[0]);
            this.mMediaCodec.release();
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Cannot release media codec", e);
        }
        this.mMediaCodec = null;
        if (this.mAudioTrack != null) {
            this.mAudioTrack.release();
        }
        this.mPendingAudioBuffer = null;
    }

    @CalledByNative
    private boolean start() {
        try {
            this.mMediaCodec.start();
            if (Build.VERSION.SDK_INT <= 19) {
                this.mInputBuffers = this.mMediaCodec.getInputBuffers();
                this.mOutputBuffers = this.mMediaCodec.getOutputBuffers();
            }
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Cannot start the media codec", e);
            return false;
        }
        catch (IllegalArgumentException e) {
            Log.e(TAG, "Cannot start the media codec", e);
            return false;
        }
        return true;
    }

    @CalledByNative
    private DequeueInputResult dequeueInputBuffer(long timeoutUs) {
        int status = 9;
        int index = -1;
        try {
            int indexOrStatus = this.mMediaCodec.dequeueInputBuffer(timeoutUs);
            if (indexOrStatus >= 0) {
                status = 0;
                index = indexOrStatus;
            } else if (indexOrStatus == -1) {
                status = 1;
            } else {
                Log.e(TAG, "Unexpected index_or_status: " + indexOrStatus, new Object[0]);
                assert (false);
            }
        }
        catch (Exception e) {
            Log.e(TAG, "Failed to dequeue input buffer", e);
        }
        return new DequeueInputResult(status, index);
    }

    @CalledByNative
    private int flush() {
        try {
            this.mFlushed = true;
            if (this.mAudioTrack != null) {
                this.mAudioTrack.pause();
                this.mAudioTrack.flush();
                this.mPendingAudioBuffer = null;
            }
            this.mMediaCodec.flush();
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to flush MediaCodec", e);
            return 9;
        }
        return 0;
    }

    @CalledByNative
    private void stop() {
        this.mMediaCodec.stop();
        if (this.mAudioTrack != null) {
            this.mAudioTrack.pause();
        }
    }

    @CalledByNative
    private GetOutputFormatResult getOutputFormat() {
        MediaFormat format = null;
        int status = 0;
        try {
            format = this.mMediaCodec.getOutputFormat();
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to get output format", e);
            status = 9;
        }
        return new GetOutputFormatResult(status, format);
    }

    @CalledByNative
    private ByteBuffer getInputBuffer(int index) {
        if (Build.VERSION.SDK_INT > 19) {
            try {
                return this.mMediaCodec.getInputBuffer(index);
            }
            catch (IllegalStateException e) {
                Log.e(TAG, "Failed to get input buffer", e);
                return null;
            }
        }
        return this.mInputBuffers[index];
    }

    @CalledByNative
    private ByteBuffer getOutputBuffer(int index) {
        if (Build.VERSION.SDK_INT > 19) {
            try {
                return this.mMediaCodec.getOutputBuffer(index);
            }
            catch (IllegalStateException e) {
                Log.e(TAG, "Failed to get output buffer", e);
                return null;
            }
        }
        return this.mOutputBuffers[index];
    }

    @CalledByNative
    private int queueInputBuffer(int index, int offset, int size, long presentationTimeUs, int flags) {
        this.resetLastPresentationTimeIfNeeded(presentationTimeUs);
        try {
            this.mMediaCodec.queueInputBuffer(index, offset, size, presentationTimeUs, flags);
        }
        catch (Exception e) {
            Log.e(TAG, "Failed to queue input buffer", e);
            return 9;
        }
        return 0;
    }

    @TargetApi(value=19)
    @CalledByNative
    private void setVideoBitrate(int bps) {
        Bundle b = new Bundle();
        b.putInt("video-bitrate", bps);
        this.mMediaCodec.setParameters(b);
    }

    @TargetApi(value=19)
    @CalledByNative
    private void requestKeyFrameSoon() {
        Bundle b = new Bundle();
        b.putInt("request-sync", 0);
        this.mMediaCodec.setParameters(b);
    }

    @CalledByNative
    private int queueSecureInputBuffer(int index, int offset, byte[] iv, byte[] keyId, int[] numBytesOfClearData, int[] numBytesOfEncryptedData, int numSubSamples, long presentationTimeUs) {
        this.resetLastPresentationTimeIfNeeded(presentationTimeUs);
        try {
            MediaCodec.CryptoInfo cryptoInfo = new MediaCodec.CryptoInfo();
            cryptoInfo.set(numSubSamples, numBytesOfClearData, numBytesOfEncryptedData, keyId, iv, 1);
            this.mMediaCodec.queueSecureInputBuffer(index, offset, cryptoInfo, presentationTimeUs, 0);
        }
        catch (MediaCodec.CryptoException e) {
            if (e.getErrorCode() == 1) {
                Log.d(TAG, "Failed to queue secure input buffer: CryptoException.ERROR_NO_KEY");
                return 7;
            }
            Log.e(TAG, "Failed to queue secure input buffer, CryptoException with error code " + e.getErrorCode(), new Object[0]);
            return 9;
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to queue secure input buffer, IllegalStateException " + e, new Object[0]);
            return 9;
        }
        return 0;
    }

    @CalledByNative
    private void releaseOutputBuffer(int index, boolean render) {
        try {
            this.mMediaCodec.releaseOutputBuffer(index, render);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Failed to release output buffer", e);
        }
    }

    @CalledByNative
    private DequeueOutputResult dequeueOutputBuffer(long timeoutUs) {
        MediaCodec.BufferInfo info = new MediaCodec.BufferInfo();
        int status = 9;
        int index = -1;
        try {
            int indexOrStatus = this.mMediaCodec.dequeueOutputBuffer(info, timeoutUs);
            if (info.presentationTimeUs < this.mLastPresentationTimeUs) {
                info.presentationTimeUs = this.mLastPresentationTimeUs;
            }
            this.mLastPresentationTimeUs = info.presentationTimeUs;
            if (indexOrStatus >= 0) {
                status = 0;
                index = indexOrStatus;
            } else if (indexOrStatus == -3) {
                assert (Build.VERSION.SDK_INT <= 19);
                this.mOutputBuffers = this.mMediaCodec.getOutputBuffers();
                status = 3;
            } else if (indexOrStatus == -2) {
                int newSampleRate;
                status = 4;
                MediaFormat newFormat = this.mMediaCodec.getOutputFormat();
                if (this.mAudioTrack != null && newFormat.containsKey("sample-rate") && this.mAudioTrack.setPlaybackRate(newSampleRate = newFormat.getInteger("sample-rate")) != 0) {
                    status = 9;
                }
            } else if (indexOrStatus == -1) {
                status = 2;
            } else {
                Log.e(TAG, "Unexpected index_or_status: " + indexOrStatus, new Object[0]);
                assert (false);
            }
        }
        catch (IllegalStateException e) {
            status = 9;
            Log.e(TAG, "Failed to dequeue output buffer", e);
        }
        return new DequeueOutputResult(status, index, info.flags, info.offset, info.presentationTimeUs, info.size);
    }

    @CalledByNative
    private boolean configureVideo(MediaFormat format, Surface surface, MediaCrypto crypto, int flags, boolean allowAdaptivePlayback) {
        try {
            if (!allowAdaptivePlayback) {
                this.mAdaptivePlaybackSupported = false;
            }
            if (this.mAdaptivePlaybackSupported) {
                format.setInteger("max-width", format.getInteger("width"));
                format.setInteger("max-height", format.getInteger("height"));
            }
            this.maybeSetMaxInputSize(format);
            this.mMediaCodec.configure(format, surface, crypto, flags);
            return true;
        }
        catch (IllegalArgumentException e) {
            Log.e(TAG, "Cannot configure the video codec, wrong format or surface", e);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Cannot configure the video codec", e);
        }
        catch (MediaCodec.CryptoException e) {
            Log.e(TAG, "Cannot configure the video codec: DRM error", new Object[]{e});
        }
        catch (Exception e) {
            Log.e(TAG, "Cannot configure the video codec", e);
        }
        return false;
    }

    @CalledByNative
    private static MediaFormat createAudioFormat(String mime, int sampleRate, int channelCount) {
        return MediaFormat.createAudioFormat((String)mime, (int)sampleRate, (int)channelCount);
    }

    @CalledByNative
    private static MediaFormat createVideoDecoderFormat(String mime, int width, int height) {
        return MediaFormat.createVideoFormat((String)mime, (int)width, (int)height);
    }

    private void maybeSetMaxInputSize(MediaFormat format) {
        int minCompressionRatio;
        int maxPixels;
        if (format.containsKey("max-input-size")) {
            return;
        }
        int maxHeight = format.getInteger("height");
        if (this.mAdaptivePlaybackSupported && format.containsKey("max-height")) {
            maxHeight = Math.max(maxHeight, format.getInteger("max-height"));
        }
        int maxWidth = format.getInteger("width");
        if (this.mAdaptivePlaybackSupported && format.containsKey("max-width")) {
            maxWidth = Math.max(maxHeight, format.getInteger("max-width"));
        }
        switch (format.getString("mime")) {
            case "video/avc": {
                if ("BRAVIA 4K 2015".equals(Build.MODEL)) {
                    return;
                }
                maxPixels = (maxWidth + 15) / 16 * ((maxHeight + 15) / 16) * 16 * 16;
                minCompressionRatio = 2;
                break;
            }
            case "video/x-vnd.on2.vp8": {
                maxPixels = maxWidth * maxHeight;
                minCompressionRatio = 2;
                break;
            }
            case "video/hevc": 
            case "video/x-vnd.on2.vp9": {
                maxPixels = maxWidth * maxHeight;
                minCompressionRatio = 4;
                break;
            }
            default: {
                return;
            }
        }
        int maxInputSize = maxPixels * 3 / (2 * minCompressionRatio);
        format.setInteger("max-input-size", maxInputSize);
    }

    @CalledByNative
    private static MediaFormat createVideoEncoderFormat(String mime, int width, int height, int bitRate, int frameRate, int iFrameInterval, int colorFormat) {
        MediaFormat format = MediaFormat.createVideoFormat((String)mime, (int)width, (int)height);
        format.setInteger("bitrate", bitRate);
        format.setInteger("frame-rate", frameRate);
        format.setInteger("i-frame-interval", iFrameInterval);
        format.setInteger("color-format", colorFormat);
        return format;
    }

    @CalledByNative
    private boolean isAdaptivePlaybackSupported(int width, int height) {
        return this.mAdaptivePlaybackSupported;
    }

    @CalledByNative
    private static void setCodecSpecificData(MediaFormat format, int index, byte[] bytes) {
        String name;
        switch (index) {
            case 0: {
                name = "csd-0";
                break;
            }
            case 1: {
                name = "csd-1";
                break;
            }
            case 2: {
                name = "csd-2";
                break;
            }
            default: {
                name = null;
            }
        }
        if (name != null) {
            format.setByteBuffer(name, ByteBuffer.wrap(bytes));
        }
    }

    @CalledByNative
    private static void setFrameHasADTSHeader(MediaFormat format) {
        format.setInteger("is-adts", 1);
    }

    @CalledByNative
    private boolean configureAudio(MediaFormat format, MediaCrypto crypto, int flags, boolean playAudio) {
        try {
            int channelCount;
            int sampleRate;
            this.mMediaCodec.configure(format, null, crypto, flags);
            return !playAudio || this.createAudioTrack(sampleRate = format.getInteger("sample-rate"), channelCount = format.getInteger("channel-count"));
        }
        catch (IllegalArgumentException e) {
            Log.e(TAG, "Cannot configure the audio codec", e);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "Cannot configure the audio codec", e);
        }
        catch (MediaCodec.CryptoException e) {
            Log.e(TAG, "Cannot configure the audio codec: DRM error", new Object[]{e});
        }
        catch (Exception e) {
            Log.e(TAG, "Cannot configure the audio codec", e);
        }
        return false;
    }

    @CalledByNative
    private boolean createAudioTrack(int sampleRate, int channelCount) {
        Log.v(TAG, "createAudioTrack: sampleRate:" + sampleRate + " channelCount:" + channelCount);
        int channelConfig = this.getAudioFormat(channelCount);
        int minBufferSize = AudioTrack.getMinBufferSize((int)sampleRate, (int)channelConfig, (int)2);
        int minBufferSizeInFrames = minBufferSize / 2 / channelCount;
        int bufferSize = (int)(1.5 * (double)minBufferSizeInFrames) * 2 * channelCount;
        if (this.mAudioTrack != null) {
            this.mAudioTrack.release();
        }
        this.mAudioTrack = new AudioTrack(3, sampleRate, channelConfig, 2, bufferSize, 1);
        if (this.mAudioTrack.getState() == 0) {
            Log.e(TAG, "Cannot create AudioTrack", new Object[0]);
            this.mAudioTrack = null;
            return false;
        }
        return true;
    }

    @CalledByNative
    private long playOutputBuffer(byte[] buf, boolean postpone) {
        if (this.mAudioTrack == null) {
            return 0L;
        }
        if (postpone) {
            assert (this.mPendingAudioBuffer == null);
            this.mPendingAudioBuffer = buf;
            return 0L;
        }
        if (3 != this.mAudioTrack.getPlayState()) {
            this.mAudioTrack.play();
        }
        int size = 0;
        if (this.mPendingAudioBuffer != null) {
            size = this.mAudioTrack.write(this.mPendingAudioBuffer, 0, this.mPendingAudioBuffer.length);
            if (this.mPendingAudioBuffer.length != size) {
                Log.i(TAG, "Failed to send all data to audio output, expected size: " + this.mPendingAudioBuffer.length + ", actual size: " + size, new Object[0]);
            }
            this.mPendingAudioBuffer = null;
        }
        if (buf.length != (size = this.mAudioTrack.write(buf, 0, buf.length))) {
            Log.i(TAG, "Failed to send all data to audio output, expected size: " + buf.length + ", actual size: " + size, new Object[0]);
        }
        return 0xFFFFFFFFL & (long)this.mAudioTrack.getPlaybackHeadPosition();
    }

    @CalledByNative
    private void setVolume(double volume) {
        if (this.mAudioTrack != null) {
            this.mAudioTrack.setStereoVolume((float)volume, (float)volume);
        }
    }

    private void resetLastPresentationTimeIfNeeded(long presentationTimeUs) {
        if (this.mFlushed) {
            this.mLastPresentationTimeUs = Math.max(presentationTimeUs - 100000L, 0L);
            this.mFlushed = false;
        }
    }

    private int getAudioFormat(int channelCount) {
        switch (channelCount) {
            case 1: {
                return 4;
            }
            case 2: {
                return 12;
            }
            case 4: {
                return 204;
            }
            case 6: {
                return 252;
            }
            case 8: {
                if (Build.VERSION.SDK_INT >= 23) {
                    return 6396;
                }
                return 1020;
            }
        }
        return 1;
    }

    @MainDex
    private static class GetOutputFormatResult {
        private final int mStatus;
        private final MediaFormat mFormat;

        private GetOutputFormatResult(int status, MediaFormat format) {
            this.mStatus = status;
            this.mFormat = format;
        }

        private boolean formatHasCropValues() {
            return this.mFormat.containsKey(MediaCodecBridge.KEY_CROP_RIGHT) && this.mFormat.containsKey(MediaCodecBridge.KEY_CROP_LEFT) && this.mFormat.containsKey(MediaCodecBridge.KEY_CROP_BOTTOM) && this.mFormat.containsKey(MediaCodecBridge.KEY_CROP_TOP);
        }

        @CalledByNative(value="GetOutputFormatResult")
        private int status() {
            return this.mStatus;
        }

        @CalledByNative(value="GetOutputFormatResult")
        private int width() {
            return this.formatHasCropValues() ? this.mFormat.getInteger(MediaCodecBridge.KEY_CROP_RIGHT) - this.mFormat.getInteger(MediaCodecBridge.KEY_CROP_LEFT) + 1 : this.mFormat.getInteger("width");
        }

        @CalledByNative(value="GetOutputFormatResult")
        private int height() {
            return this.formatHasCropValues() ? this.mFormat.getInteger(MediaCodecBridge.KEY_CROP_BOTTOM) - this.mFormat.getInteger(MediaCodecBridge.KEY_CROP_TOP) + 1 : this.mFormat.getInteger("height");
        }

        @CalledByNative(value="GetOutputFormatResult")
        private int sampleRate() {
            return this.mFormat.getInteger("sample-rate");
        }

        @CalledByNative(value="GetOutputFormatResult")
        private int channelCount() {
            return this.mFormat.getInteger("channel-count");
        }
    }

    @MainDex
    private static class DequeueOutputResult {
        private final int mStatus;
        private final int mIndex;
        private final int mFlags;
        private final int mOffset;
        private final long mPresentationTimeMicroseconds;
        private final int mNumBytes;

        private DequeueOutputResult(int status, int index, int flags, int offset, long presentationTimeMicroseconds, int numBytes) {
            this.mStatus = status;
            this.mIndex = index;
            this.mFlags = flags;
            this.mOffset = offset;
            this.mPresentationTimeMicroseconds = presentationTimeMicroseconds;
            this.mNumBytes = numBytes;
        }

        @CalledByNative(value="DequeueOutputResult")
        private int status() {
            return this.mStatus;
        }

        @CalledByNative(value="DequeueOutputResult")
        private int index() {
            return this.mIndex;
        }

        @CalledByNative(value="DequeueOutputResult")
        private int flags() {
            return this.mFlags;
        }

        @CalledByNative(value="DequeueOutputResult")
        private int offset() {
            return this.mOffset;
        }

        @CalledByNative(value="DequeueOutputResult")
        private long presentationTimeMicroseconds() {
            return this.mPresentationTimeMicroseconds;
        }

        @CalledByNative(value="DequeueOutputResult")
        private int numBytes() {
            return this.mNumBytes;
        }
    }

    @MainDex
    private static class DequeueInputResult {
        private final int mStatus;
        private final int mIndex;

        private DequeueInputResult(int status, int index) {
            this.mStatus = status;
            this.mIndex = index;
        }

        @CalledByNative(value="DequeueInputResult")
        private int status() {
            return this.mStatus;
        }

        @CalledByNative(value="DequeueInputResult")
        private int index() {
            return this.mIndex;
        }
    }
}

