package jp.agentec.abook.abv.ui.viewer.view;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.Toast;

import org.json.adf.JSONObject;

import java.util.List;

import jp.agentec.abook.abv.bl.acms.client.json.MarkingJson;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.abook.abv.bl.dto.MarkingSettingDto;
import jp.agentec.abook.abv.bl.logic.AbstractLogic;
import jp.agentec.abook.abv.bl.logic.ContentLogic;
import jp.agentec.abook.abv.bl.logic.MarkingLogic;
import jp.agentec.abook.abv.bl.websocket.MeetingManager;
import jp.agentec.abook.abv.cl.helper.ContentMarkingFileHelper;
import jp.agentec.abook.abv.cl.util.PreferenceUtil;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType;
import jp.agentec.abook.abv.ui.common.dialog.ABookAlertDialog;
import jp.agentec.abook.abv.ui.common.dialog.ContentColorPickerView;
import jp.agentec.abook.abv.ui.common.util.ABVToastUtil;
import jp.agentec.abook.abv.ui.common.util.AlertDialogUtil;
import jp.agentec.abook.abv.ui.Interface.MovePageInterface;
import jp.agentec.abook.abv.ui.Interface.UnAuthorizedContentListener;
import jp.agentec.abook.abv.ui.viewer.view.ZoomRelativeLayout.ViewMargin;

/**
 * タッチされた場所にマーキングを描画するクラスです。
 * 
 * @author tauchi
 *
 */
public class MarkingLayout extends RelativeLayout {
	private final static String TAG = "MarkingLayout";

	private MarkingLogic markingLogic = AbstractLogic.getLogic(MarkingLogic.class);

	public interface OnMarkingListener {
		void close();
	}

	private Activity context; // 呼出元アクティビティ
	private ContentDto contentDto; // コンテンツDto
	private int pageNum; // ページ番号
	private int pageWidth; // ページ幅（イメージ保存用）
	private int pageHeight; // ページ高さ（イメージ保存用）
	
	private OnMarkingListener markingListener; // リスナー
	private AlertDialog alertDialog = null; // アラートダイアログ

	private Paint penPaint; // ペン用ペイント
	private Paint markerPaint; // マーカー用ペイント
	private Paint eraserPaint; // 消しゴム用ペイント

	private boolean paletteVisible = true; // メニュー表示・非表示フラグ
	private TouchDrawView touchDrawView; // タッチドロービュー
	private ImageButton btnPallete; // メニューサイズ変更ボタン
	private View palette;

	private ImageButton penButton; // ペン選択ラジオボタン
	private ImageButton markerButton; // マーカー選択ラジオボタン
	private ImageButton eraserButton; // 消しゴム選択ラジオボタン

	private LinearLayout linearPaletteItems1; // メニュー用レイアウト
	private LinearLayout linearPaletteItems2; // メニュー用レイアウト

	private boolean allClearFlag = false; // すべてクリアフラグ

	private boolean isOpenPalette = true;
	private ContentMarkingFileHelper contentMarkingFileHelper = new ContentMarkingFileHelper();
	private MeetingManager meetingManager;
	public boolean chairmanMarkingShareFlag = false;

	public ViewMargin margin;

	private UnAuthorizedContentListener unAuthorizedContentListener;
	private MovePageInterface movePageListener;
	/**
	 * NoUse
	 */
	public MarkingLayout(Context context) {
		super(context);
	}
	// 初期化
	public MarkingLayout(Context context, ContentDto contentDto, int pageNum, int pageWidth, int pageHeight, ViewMargin margin, ContentLogic contentLogic, OnMarkingListener markingListener) {
		super(context);

		this.context = (Activity) context;
		this.contentDto = contentDto;
		this.pageNum = pageNum;
		this.pageWidth = pageWidth;
		this.pageHeight = pageHeight;
		this.markingListener = markingListener;
		this.margin = margin;

		meetingManager = MeetingManager.getInstance();
		List<MarkingSettingDto> listDto = markingLogic.getMarkingSettingList();

		// ペン、マーカー、消しゴムペイント初期化
		penPaint = new Paint();
		markerPaint = new Paint();
		eraserPaint = new Paint();

		for (MarkingSettingDto dto : listDto) {
			if (dto.markingSettingId == TouchDrawView.PEN) {
				initPenPaint(dto.color, dto.width, 255);
			} else if (dto.markingSettingId == TouchDrawView.MARKER) {
				initMarkerPaint(dto.color, dto.width, 127);
			} else if (dto.markingSettingId == TouchDrawView.ERASER) {
				initEraserPaint(dto.color, dto.width);
			}
		}

		// タッチドロービュー
		RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
		params.setMargins(margin.left, margin.top, margin.right, margin.bottom);
		touchDrawView = new TouchDrawView(context, loadFromFile(), new TouchDrawView.OnDrawMarkingListener() {
			@Override
			public void onDrawMarking(int action, float x, float y) {
				if (meetingManager.isSendable()) {
					sendJson(action, x, y);
				}
			}
		});
		
		addView(touchDrawView, params);
		
		craetePalette();
		
		// 参加者で一時停止でないときはリードオンリーにセット
		setReadOnly(meetingManager.isSubscribed() && !meetingManager.isPaused());
		
		// 初期はペンモードにセット
		setPenMode();
	}

	public void setUnAuthorizedContentListener(UnAuthorizedContentListener listener) {
		unAuthorizedContentListener = listener;
	}

	public void setMovePageListener(MovePageInterface listener) {
		movePageListener = listener;
	}

	public void setReadOnly(boolean readOnly) {
		if (readOnly) {
			touchDrawView.drawable = false;
			palette.setVisibility(View.INVISIBLE);
		} else {
			touchDrawView.drawable = true;
			palette.setVisibility(View.VISIBLE);

			// 初期化
			List<MarkingSettingDto> listDto = markingLogic.getMarkingSettingList();

			for (MarkingSettingDto dto : listDto) {
				if (dto.markingSettingId == TouchDrawView.PEN) {
					initPenPaint(dto.color, dto.width, 255);
				} else if (dto.markingSettingId == TouchDrawView.MARKER) {
					initMarkerPaint(dto.color, dto.width, 127);
				} else if (dto.markingSettingId == TouchDrawView.ERASER) {
					initEraserPaint(dto.color, dto.width);
				}
			}
			setPenMode();
		}
	}

	private void craetePalette(){
		palette = context.getLayoutInflater().inflate(R.layout.marking, null);
		addView(palette);

		// メニューレイアウト
		linearPaletteItems1 = (LinearLayout) findViewById(R.id.linear_palette_items1);
		linearPaletteItems2 = (LinearLayout) findViewById(R.id.linear_palette_items2);

		// メニューサイズ変更ボタン
		btnPallete = (ImageButton) findViewById(R.id.btn_palette);
		btnPallete.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View view) {
				changePaletteSize();
			}
		});

		// ペン選択ラジオボタン
		penButton = (ImageButton) findViewById(R.id.radio_btn_pen);
		penButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (touchDrawView.mode == TouchDrawView.PEN) {
					showColorPickerDialog(TouchDrawView.PEN);
				} else {
					setPenMode();
				}
			}
		});

		// マーカー選択ラジオボタン
		markerButton = (ImageButton) findViewById(R.id.radio_btn_marker);
		markerButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (touchDrawView.mode == TouchDrawView.MARKER) {
					showColorPickerDialog(TouchDrawView.MARKER);
				} else {
					setMarkerMode();
				}
			}
		});

		// 消しゴム選択ラジオボタン
		eraserButton = (ImageButton) findViewById(R.id.radio_btn_eraser);
		eraserButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (touchDrawView.mode == TouchDrawView.ERASER) {
					showColorPickerDialog(TouchDrawView.ERASER);
				} else {
					setEraserMode();
				}
			}
		});

		// 保存ボタン
		Button btnSave = (Button) findViewById(R.id.btn_save);
		btnSave.setOnClickListener(new View.OnClickListener() {
			@Override
            public void onClick(View v) {
				if (contentDto.isUnAuthorizedContent) {
					unAuthorizedContentListener.unAuthorizedContentNotification(getResources().getString(R.string.marking_unusable));
				} else {
					saveClose();
				}
			}
		});

		// 終了ボタン
		Button btnEnd = (Button) findViewById(R.id.btn_end);
		btnEnd.setOnClickListener(new View.OnClickListener() {
			@Override
            public void onClick(View v) {
				notSaveClose();
			}
		});

		// クリアボタン
		Button btnClear = (Button) findViewById(R.id.btn_clear);
		btnClear.setOnClickListener(new View.OnClickListener() {
			@Override
            public void onClick(View v) {
				clear();
			}
		});

		Button btnNext = (Button) findViewById(R.id.nextPage) ;
		btnNext.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				movePageListener.nextPage();
			}
		});
		Button btnBack = (Button) findViewById(R.id.backPage);
		btnBack.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				movePageListener.backPage();
			}
		});
	}

	@Override
	public boolean onTouchEvent(MotionEvent event) {
		return (!meetingManager.isSubscribed() || meetingManager.isPaused());
	}

	// ペン初期化
	private void initPenPaint(int color, int width, int alpha) {
		penPaint.setAntiAlias(true);
		penPaint.setDither(true);
		penPaint.setColor(color);
		penPaint.setStyle(Paint.Style.STROKE);
		penPaint.setStrokeJoin(Paint.Join.ROUND);
		penPaint.setStrokeCap(Paint.Cap.ROUND);
		penPaint.setStrokeWidth(width);
		penPaint.setAlpha(alpha);
		penPaint.setXfermode(null);
	}

	private void initPenPaint(int red, int green, int blue, int width, int alpha) {
		penPaint.setAntiAlias(true);
		penPaint.setDither(true);
		penPaint.setStyle(Paint.Style.STROKE);
		penPaint.setStrokeJoin(Paint.Join.ROUND);
		penPaint.setStrokeCap(Paint.Cap.ROUND);
		penPaint.setStrokeWidth(width);
		penPaint.setARGB(alpha, red, green, blue);
		penPaint.setXfermode(null);
	}

	// マーカー初期化
	private void initMarkerPaint(int color, int width, int alpha) {
		markerPaint.setAntiAlias(true);
		markerPaint.setDither(true);
		markerPaint.setColor(color);
		markerPaint.setStyle(Paint.Style.STROKE);
		markerPaint.setStrokeJoin(Paint.Join.ROUND);
		markerPaint.setStrokeCap(Paint.Cap.SQUARE);
		markerPaint.setStrokeWidth(width);
		markerPaint.setAlpha(alpha);
		markerPaint.setXfermode(null);
	}

	private void initMarkerPaint(int red, int green, int blue, int width, int alpha) {
		markerPaint.setAntiAlias(true);
		markerPaint.setDither(true);
		markerPaint.setStyle(Paint.Style.STROKE);
		markerPaint.setStrokeJoin(Paint.Join.ROUND);
		markerPaint.setStrokeCap(Paint.Cap.SQUARE);
		markerPaint.setStrokeWidth(width);
		markerPaint.setARGB(alpha, red, green, blue);
		markerPaint.setXfermode(null);
	}

	// 消しゴム初期化
	private void initEraserPaint(int color, int width) {
		eraserPaint.setAntiAlias(true);
		eraserPaint.setDither(true);
		eraserPaint.setColor(color);
		eraserPaint.setStyle(Paint.Style.STROKE);
		eraserPaint.setStrokeJoin(Paint.Join.ROUND);
		eraserPaint.setStrokeCap(Paint.Cap.ROUND);
		eraserPaint.setStrokeWidth(width);
		eraserPaint.setAlpha(255);
		eraserPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
	}

	private void initEraserPaint(int red, int green, int blue, int width) {
		eraserPaint.setAntiAlias(true);
		eraserPaint.setDither(true);
		eraserPaint.setStyle(Paint.Style.STROKE);
		eraserPaint.setStrokeJoin(Paint.Join.ROUND);
		eraserPaint.setStrokeCap(Paint.Cap.ROUND);
		eraserPaint.setStrokeWidth(width);
		eraserPaint.setARGB(255, red, green, blue);
		eraserPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
	}

	// ペンモード
	private void setPenMode() {
		penButton.setImageResource(R.drawable.icon_pen_on);
		markerButton.setImageResource(R.drawable.icon_marker_off);
		eraserButton.setImageResource(R.drawable.icon_eraser_off);
		touchDrawView.setPaint(penPaint, TouchDrawView.PEN);
	}

	// マーカーモード
	private void setMarkerMode() {
		penButton.setImageResource(R.drawable.icon_pen_off);
		markerButton.setImageResource(R.drawable.icon_marker_on);
		eraserButton.setImageResource(R.drawable.icon_eraser_off);
		touchDrawView.setPaint(markerPaint, TouchDrawView.MARKER);
	}

	// 消しゴムモード
	private void setEraserMode() {
		penButton.setImageResource(R.drawable.icon_pen_off);
		markerButton.setImageResource(R.drawable.icon_marker_off);
		eraserButton.setImageResource(R.drawable.icon_eraser_on);
		touchDrawView.setPaint(eraserPaint, TouchDrawView.ERASER);
	}

	// 保存しないで閉じる
	private void notSaveClose() {
		if (meetingManager.isSendable()) {
			JSONObject json = new JSONObject();
			json.put(MeetingManager.MARKING_ACTION, MeetingManager.MARKING_ACTION_END);
			meetingManager.sendWs(MeetingManager.CMD_MARKING, contentDto.contentId, pageNum, null, json);
		}
		markingListener.close();
	}

	// 保存して閉じる
	public void saveClose() {
		if (allClearFlag) {
			allFileDelete();
		}
		saveToFile();
		if (meetingManager.isSendable()) {
			JSONObject json = new JSONObject();
			json.put(MeetingManager.MARKING_ACTION, MeetingManager.MARKING_ACTION_SAVE);
			if (PreferenceUtil.get(getContext(), AppDefType.UserPrefKey.CHAIRMAN_MARKING_SHARE, false)) {
				json.put(MeetingManager.MARKING_SHARE, 1);
			} else {
				json.put(MeetingManager.MARKING_SHARE, 0);
			}

			meetingManager.sendWs(MeetingManager.CMD_MARKING, contentDto.contentId, pageNum, null, json);
		}
		markingListener.close();
	}

	// ファイル保存
	private void saveToFile() {
		try {
			contentMarkingFileHelper.saveMarkingFile(context, contentDto.contentId, pageNum, pageWidth, pageHeight, touchDrawView.mBitmap);
			ABVToastUtil.showMakeText(context, String.format(context.getResources().getString(R.string.save_success), ""), Toast.LENGTH_SHORT);
			boolean meetingMarkingAutoSaveFlag = PreferenceUtil.get(getContext(), AppDefType.UserPrefKey.MEETING_MARKING_AUTO_SAVE, false);

			// 連動中は会議中マーキングを保存
			if ((meetingManager.isSendable() && meetingMarkingAutoSaveFlag) || (meetingManager.isSubscribed() && chairmanMarkingShareFlag && meetingMarkingAutoSaveFlag)) {
				contentMarkingFileHelper.saveMarkingOnMeeting(contentDto.contentId, pageNum);
				ABVToastUtil.showMakeText(context, String.format(context.getResources().getString(R.string.save_success), ""), Toast.LENGTH_SHORT);
			}
		} catch (Exception e) {
			Logger.e("Exception", e.toString());
			ABVToastUtil.showMakeText(context, String.format(context.getResources().getString(R.string.save_fail), ""), Toast.LENGTH_SHORT);
		}
	}

	public void clear(){
		touchDrawView.clear();
		if (meetingManager.isSendable()) {
			JSONObject json = new JSONObject();
			json.put(MeetingManager.MARKING_ACTION, MeetingManager.MARKING_ACTION_CLEAR);
			meetingManager.sendWs(MeetingManager.CMD_MARKING, contentDto.contentId, pageNum, null, json);
		}
	}

	private Bitmap loadFromFile() {
		Bitmap image = null;
		try {
			image = contentMarkingFileHelper.getMarkingFile(contentDto.contentId, pageNum);
		} catch (Exception e) {
			Logger.e("Exception", e.toString());
			ABVToastUtil.showMakeText(context, String.format(context.getResources().getString(R.string.load_fail), ""), Toast.LENGTH_SHORT);
		}
		return image;
	}

	// マーキングフォルダ下全て削除
	private void allFileDelete() {
		try {
			contentMarkingFileHelper.deleteMarkingFile(contentDto.contentId);
		} catch (Exception e) {
			Logger.e("Exception", e.toString());
			ABVToastUtil.showMakeText(context, context.getResources().getString(R.string.clear), Toast.LENGTH_SHORT);
		}
	}

	// マージンセット
	public void setTouchDrawViewMargin(ViewMargin margin) {
		RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams)touchDrawView.getLayoutParams();
		params.setMargins(margin.left, margin.top, margin.right, margin.bottom);
	}

	/**
	 * マーキングで描く色を設定します。
	 * 
	 * @param color
	 *            色
	 */
	public void setPaintColor(int color, int type) {
		if (type == TouchDrawView.PEN) {
			setColor(penPaint, color);
		} else if (type == TouchDrawView.MARKER) {
			setColor(markerPaint, color);
		} else if (type == TouchDrawView.ERASER) {
			setColor(eraserPaint, color);
		}
	}

	private void setColor(Paint paint, int color) {
		// 色を設定する前に既存の透明度保存
		int alpha = paint.getAlpha();
		// 色セット
		paint.setColor(color);
		// 透明度再セット
		paint.setAlpha(alpha);
	}

	/**
	 * マーキングで描く色を設定します。
	 * 
	 * @return ペイントの色
	 */
	public int getPaintColor(int type) {
		int color = 0;

		if (type == TouchDrawView.PEN) {
			color = penPaint.getColor();
		} else if (type == TouchDrawView.MARKER) {
			color = markerPaint.getColor();
		} else if (type == TouchDrawView.ERASER) {
			color = eraserPaint.getColor();
		}
		return color;
	}

	/**
	 * マーキングで描く太さを設定します。
	 * 
	 * @param size
	 */
	public void setPaintStrokeWidth(int size, int type) {
		if (type == TouchDrawView.PEN) {
			penPaint.setStrokeWidth(size);
		} else if (type == TouchDrawView.MARKER) {
			markerPaint.setStrokeWidth(size);
		} else if (type == TouchDrawView.ERASER) {
			eraserPaint.setStrokeWidth(size);
		}
	}

	/**
	 * マーキングで描く太さを取得します。
	 * 
	 * @return ペイントの太さ
	 */
	public int getPaintStrokeWidth(int mode) {
		int width = 0;

		if (mode == TouchDrawView.PEN) {
			width = (int) penPaint.getStrokeWidth();
		} else if (mode == TouchDrawView.MARKER) {
			width = (int) markerPaint.getStrokeWidth();
		} else if (mode == TouchDrawView.ERASER) {
			width = (int) eraserPaint.getStrokeWidth();
		}
		return width;
	}

	/**
	 * カラーピッカーダイアログ表示
	 */
	private void showColorPickerDialog(int mode) {
		int mPenSize = getPaintStrokeWidth(mode);
		// 既存の設定されている色の透明度を255に変換
		int mColor = getPaintColor(mode) | 0xFF000000;

		final ABookAlertDialog dialog = AlertDialogUtil.createABookAlertDialog(context);
		final ContentColorPickerView colorPicker = new ContentColorPickerView();
		View view = colorPicker.createColorPickerView(context, mode, mPenSize, mColor);
		dialog.setView(view);
		dialog.setNegativeButton(R.string.cancel, null);
		dialog.setNeutralButton(R.string.confirm, new DialogInterface.OnClickListener() {

			@Override
            public void onClick(DialogInterface dialog, int whichButton) {
				int penSize = colorPicker.getPenSize();
				int color = colorPicker.getColor();
				int type = colorPicker.getType();
				setPaintStrokeWidth(penSize, type);
				setPaintColor(color, type);
				try {
					markingLogic.updateMarkingSetting(type, color, penSize);
				} catch (Exception e) {
					Logger.e("Marking", e.toString());
				}
			}
		});
		showAlertDialog(dialog);
	}

	/**
	 * メニューボタン及びメニューレイアウトの可視を設定します。
	 * 
	 * @param visible
	 */
	public void setPaletteVisible(boolean visible) {
		paletteVisible = visible;
		if (visible) {
			btnPallete.setVisibility(View.VISIBLE);
			openPalette();
		} else {
			btnPallete.setVisibility(View.GONE);
			closePalette();
		}
	}

	/**
	 * メニューボタン及びメニューレイアウトの可視を返します。
	 * 
	 * @return
	 */
	public boolean getPaletteVisible() {
		return paletteVisible;
	}

	/**
	 * 保存ダイアログを表示します表示
	 * 
	 */
	public void showSaveDialog() {
		// 編集していない場合、ダイアログを表示せずに保存せずに終了
		if (touchDrawView.editFlg) {
			final ABookAlertDialog aBookAlertDialog = AlertDialogUtil.createAlertDialog(context, R.string.end);
			aBookAlertDialog.setPositiveButton(R.string.save_end, new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int which) {
					saveClose();
				}
			});
			aBookAlertDialog.setNeutralButton(R.string.not_save_end, new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int which) {
					notSaveClose();
				}
			});
			aBookAlertDialog.setNegativeButton(R.string.cancel, new DialogInterface.OnClickListener() {
				@Override
				public void onClick(DialogInterface dialog, int which) {
				}
			});
			showAlertDialog(aBookAlertDialog);
		} else {
			notSaveClose();
		}
	}

	private void changePaletteSize() {
		if (isOpenPalette) {
			closePalette();
		} else {
			openPalette();
		}
	}

	private void openPalette() {
		isOpenPalette = true;
		btnPallete.setImageResource(R.drawable.icon_palette_on);
		linearPaletteItems1.setVisibility(View.VISIBLE);
		linearPaletteItems2.setVisibility(View.VISIBLE);
	}

	private void closePalette() {
		isOpenPalette = false;
		btnPallete.setImageResource(R.drawable.icon_palette_off);
		linearPaletteItems1.setVisibility(View.GONE);
		linearPaletteItems2.setVisibility(View.GONE);
	}

	// アラートダイアログ表示
	private void showAlertDialog(AlertDialog alertDialog) {
		if (this.alertDialog != null && this.alertDialog.isShowing()) {
			// 表示しているダイアログを非表示(二重表示防止)
			this.alertDialog.dismiss();
		}
		this.alertDialog = alertDialog;
		alertDialog.show();
	}

	private synchronized void sendJson(int action, float x, float y) {
		float scale = (float)pageWidth / (getWidth() -  margin.left - margin.right);
		JSONObject json = new JSONObject();
		json.put(MeetingManager.MARKING_ACTION, MeetingManager.MARKING_ACTION_DRAW);
		json.put(MeetingManager.MARKING_DRAW_ACTION, action);
		if(action == MotionEvent.ACTION_DOWN){
			json.put(MeetingManager.MARKING_DRAW_MODE, touchDrawView.mode);
			json.put(MeetingManager.MARKING_DRAW_WIDTH, scale * touchDrawView.mPaint.getStrokeWidth());
			json.put(MeetingManager.MARKING_DRAW_ALPHA, touchDrawView.mPaint.getAlpha());
			json.put(MeetingManager.MARKING_DRAW_RED, Color.red(touchDrawView.mPaint.getColor()));
			json.put(MeetingManager.MARKING_DRAW_GREEN, Color.green(touchDrawView.mPaint.getColor()));
			json.put(MeetingManager.MARKING_DRAW_BLUE, Color.blue(touchDrawView.mPaint.getColor()));
		}
		
		if(action != MotionEvent.ACTION_UP){
			json.put(MeetingManager.X, (int) (x * scale + 0.5));
			json.put(MeetingManager.Y, (int) (y * scale + 0.5));
		}
		meetingManager.sendWs(MeetingManager.CMD_MARKING, contentDto.contentId, pageNum, null, json);
	}

	public synchronized void drawRemoteMarkig(MarkingJson json) {
		if (json.drawAction == MotionEvent.ACTION_DOWN) {
			switch (json.mode) {
			case TouchDrawView.MARKER:
				initMarkerPaint(json.red, json.green, json.blue, json.width, json.alpha);
				setMarkerMode();
				break;
			case TouchDrawView.ERASER:
				initEraserPaint(json.red, json.green, json.blue, json.width);
				setEraserMode();
				break;
			default:
				initPenPaint(json.red, json.green, json.blue, json.width, json.alpha);
				setPenMode();
				break;
			}
		}
		touchDrawView.remoteDrawMarking(json.drawAction, json.pointX, json.pointY);
		touchDrawView.invalidate();
	}

	public void setBackPageStatus(boolean isEnable) {
		findViewById(R.id.backPage).setEnabled(isEnable);
	}

	public void setNextPageStatus(boolean isEnable) {
		findViewById(R.id.nextPage).setEnabled(isEnable);
	}

}
