package jp.agentec.abook.abv.bl.data.dao;

import java.util.List;

import jp.agentec.abook.abv.bl.common.db.Cursor;
import jp.agentec.abook.abv.bl.dto.SppDeviceDto;

/**
 * SPP通信の端末情報を管理する
 */

public class SppDeviceDao extends AbstractDao {

    private static final String TAG = "SppDeviceDao";

    @Override
    protected SppDeviceDto convert(Cursor cursor) {
        SppDeviceDto dto = new SppDeviceDto();

        int column = cursor.getColumnIndex("spp_device_id");
        if (column != -1) {
            dto.sppDeviceId = cursor.getInt(column);
        }
        column = cursor.getColumnIndex("spp_device_name");
        if (column != -1) {
            dto.sppDeviceName = cursor.getString(column);
        }

        column = cursor.getColumnIndex("data_start_index");
        if (column != -1) {
            dto.dataStartIndex = cursor.getInt(column);
        }
        column = cursor.getColumnIndex("data_end_index");
        if (column != -1) {
            dto.dataEndIndex = cursor.getInt(column);
        }
        column = cursor.getColumnIndex("pairing_device_name");
        if (column != -1) {
            dto.pairingDeviceName = cursor.getString(column);
        }
        column = cursor.getColumnIndex("pairing_device_address");
        if (column != -1) {
            dto.pairingDeviceAddress = cursor.getString(column);
        }
        return dto;
    }

    /**
     * 登録処理
     * @param dto
     */
    public void insert(SppDeviceDto dto) {
        insert("insert into t_spp_device "
                        + "(spp_device_id, "
                        + "spp_device_name, "
                        + "data_start_index, "
                        + "data_end_index, "
                        + "pairing_device_name, "
                        + "pairing_device_address) "
                        + "values "
                        + "(?,?,?,?,?,?)",
                dto.getInsertValues());
    }

    /**
     * SPP通信端末を全て取得
     * @return
     */
    public List<SppDeviceDto> getAllSppDevice() {
        return rawQueryGetDtoList("SELECT * FROM t_spp_device ORDER BY spp_device_id DESC", null, SppDeviceDto.class);
    }

    /**
     * SPP通信端末をIDで検索
     * @param sppDeviceId
     * @return
     */
    public SppDeviceDto getSppDeviceById(Integer sppDeviceId) {
        return rawQueryGetDto("SELECT * FROM t_spp_device WHERE spp_device_id = ?", new String[] { "" + sppDeviceId }, SppDeviceDto.class);
    }

    /**
     * SPP通信端末IDを全て取得
     * @return
     */
    public List<Integer> getSppDeviceIdList() {
        return rawQueryGetIntegerList("SELECT spp_device_id FROM t_spp_device ORDER BY spp_device_id DESC", null);
    }

    /**
     * SPP通信端末でペアリング済みデータ取得
     * @return
     */
    public List<SppDeviceDto> getPairingDeviceList() {
        return rawQueryGetDtoList("SELECT * FROM t_spp_device WHERE pairing_device_name is NOT NULL AND pairing_device_address is NOT NULL order by spp_device_id DESC", null, SppDeviceDto.class);
    }

    /**
     * SPP通信端末の更新処理
     * @param dto
     * @return
     */
    public boolean updateSppDevice(SppDeviceDto dto) {
        long count = update("UPDATE t_spp_device SET spp_device_name=?, data_start_index=?, data_end_index=?, pairing_device_name=?, pairing_device_address=? WHERE spp_device_id=?", dto.getUpdateValues());

        return count > 0;
    }

    /**
     * ペアリングの端末アドレスがユニーク前提
     * ペアリング情報をアドレスで取得
     * @param deviceAddress
     * @return
     */
    public SppDeviceDto getPairingDeviceByAddress(String deviceAddress) {
        StringBuffer sql = new StringBuffer();
        sql.append("    SELECT DISTINCT * ");
        sql.append("    FROM t_spp_device");
        sql.append("    WHERE pairing_device_address = ?");
        return rawQueryGetDto(sql.toString(), new String[] { deviceAddress }, SppDeviceDto.class);
    }

    /**
     * ペアリング情報をクリアする
     * @param deviceAddress
     */
    public void clearPairingDevice(String deviceAddress) {
        SppDeviceDto sppDeviceDto = getPairingDeviceByAddress(deviceAddress);
        // ペアリング情報を空にして更新
        sppDeviceDto.pairingDeviceName = null;
        sppDeviceDto.pairingDeviceAddress = null;

        updateSppDevice(sppDeviceDto);
    }

    /**
     * SPP端末IDでDB情報を削除
     * @param sppDeviceId
     */
    public void deleteById(Integer sppDeviceId) {
        delete("t_spp_device", "spp_device_id=?", new String[] { "" + sppDeviceId });
    }
}
