package jp.agentec.abook.abv.bl.logic;

import java.util.List;
import java.util.UUID;

import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.ContentDao;
import jp.agentec.abook.abv.bl.data.dao.ContentMemoDao;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.abook.abv.bl.dto.ContentMemoDto;

public class MemoLogic extends AbstractLogic {
	private ContentMemoDao contentMemoDao = AbstractDao.getDao(ContentMemoDao.class);
	private ContentDao contentDao = AbstractDao.getDao(ContentDao.class);


	/**
	 * Memo情報を追加します。
	 * @param dto ContentMemoDto
	 * @throws Exception
	 * @since 1.0.0
	 */
	public void insertContentMemo(ContentMemoDto dto) {
		// UUIDをメモIDとして自動作成
		dto.memoId = UUID.randomUUID().toString();
		contentMemoDao.insertMemo(dto);
	}

	/**
	 * 指定したメモをコピーされた状態に設定します。isCutをtrueにすると切り取り状態になります。
	 * @param dto コピーするメモ情報を格納したdtoです。
	 * @param isCut コピーだけならfalseを、切り取りならtrueを設定します。
	 * @since 1.1.3
	 */
	public boolean setContentMemoToCopy(ContentMemoDto dto, boolean isCut) {
		dto.copyFlg = true;
		dto.delFlg = isCut;
		return contentMemoDao.setCopiedMemo(dto);
	}

	/**
	 * メモをコンテンツ間でコピーします。
	 * 
	 * @param srcContentId コピー元コンテンツID
	 * @param dstContentId コピー先コンテンツID
	 * @throws Exception
	 */
	public void copyContentMemo(long srcContentId, long dstContentId) {
		List<ContentMemoDto> list = contentMemoDao.getMemoList(srcContentId);
		ContentDto dstContentDto = contentDao.getContent(dstContentId);

		for (ContentMemoDto contentMemoDto : list) {
			contentMemoDto.contentId = dstContentId;
			// コピー先のページ数より大きい場合、最後のページに移動
			if (dstContentDto.allPageNum - 1 < contentMemoDto.pageNum) {
				contentMemoDto.pageNum = dstContentDto.allPageNum - 1;
			}
			// UUIDをメモIDとして自動作成
			contentMemoDto.memoId = UUID.randomUUID().toString();
		}

		try {
			contentMemoDao.beginTransaction();
			contentMemoDao.insertMemos(list);
			contentMemoDao.commit();
		} catch (Exception e) {
			contentMemoDao.rollback();
			throw new RuntimeException(e);
		}
	}
}
