package jp.agentec.abook.abv.cl.util;

import android.Manifest;
import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothManager;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.ContextMenu;
import android.view.LayoutInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.BaseAdapter;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import jp.agentec.abook.abv.bl.common.log.Logger;

import static android.bluetooth.le.ScanSettings.CALLBACK_TYPE_ALL_MATCHES;
import static android.bluetooth.le.ScanSettings.MATCH_MODE_AGGRESSIVE;
import static android.bluetooth.le.ScanSettings.SCAN_MODE_LOW_LATENCY;
import static org.chromium.base.ThreadUtils.runOnUiThread;


public class TR41BluetoothUtil {
    public static final String TAG = "TR41BluetootUtil";

    private BluetoothManager mBluetoothManager ;
    public BluetoothAdapter mBluetoothAdapter ;
    private BluetoothLeScanner mBluetoothLeScanner;

    private TR41BluetoothUtilLeScanner mListener;
    private Context mContext;
    private Handler mHandler;

    private boolean mIsScaning;

    private static final long SCAN_PERIOD = (20 * 1000); // スキャン時間。単位はミリ秒

    /**
     *
     */
    public interface TR41BluetoothUtilLeScanner {
        void onScanFailed(int errorCode);
        void onScanResult(String serialNo, String strTemperature); // デバイスから温度を渡す
        void onScanStop(); // スキャンを停止した
    }

    public TR41BluetoothUtil(Context context, TR41BluetoothUtilLeScanner listener)
    {
        mContext = context;
        mListener = listener;
        mHandler = new Handler();

        mIsScaning = false;
    }
    // ************************************************************************
    // void Sb_Init_Ble_Scan()
    // 概要 Bluetoothの初期化
    // 引数  :IN :
    //       :OUT:
    //       :RET:
    // ************************************************************************
    private void Sb_Init_Ble_Scan()
    {
        if (mBluetoothAdapter == null){
            return;
        }

        boolean btEnable = mBluetoothAdapter.isEnabled() ;
        if(btEnable) {
            // Bluetoothアダプタ作成
            mBluetoothManager = (BluetoothManager) mContext.getSystemService( Context.BLUETOOTH_SERVICE );
            mBluetoothLeScanner = mBluetoothAdapter.getBluetoothLeScanner() ;
        }
    }

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // 検索関連
    ///////////////////////////////////////////////////////////////////////////////////////////////

    /** BLE機器を検索する */
    public void Sb_StartScan()
    {
        Sb_Init_Ble_Scan() ;

        // スキャンフィルタの作成
        List<ScanFilter> mScanFilters = new ArrayList<>() ;
        // スキャンモードの作成
        ScanSettings.Builder mScanSettingBuiler = new ScanSettings.Builder();

        /*
         * CALLBACK_TYPE_ALL_MATCHES
         * 検出されたすべてのBluetoothアドバタイズのコールバックをトリガーし、フィルター基準に一致します。
         * アクティブなフィルターがない場合、すべてのアドバタイズパケットが報告されます。　
         * ... 500msに１回ほどスキャンしている
         *
         * MATCH_MODE_AGGRESSIVE
         * アグレッシブモードでは、信号強度が弱く、継続時間中に目撃/一致がほとんどない場合でも、
         * ハードウェアは一致をより早く判断します。
         * ... ALL_MATCHESよりも高速にスキャンしている。（遠くの弱い信号も受信している。遠くの弱い信号は必要ない）
         */
        // 近くのTR41を検索するのであれば、ALL_MATCHESで問題ない。
        mScanSettingBuiler.setMatchMode(CALLBACK_TYPE_ALL_MATCHES);

        /*
         * SCAN_MODE_LOW_LATENCY
         * 最高のデューティサイクルを使用してスキャンします。 アプリケーションがフォアグラウンドで実行されている場合にのみ、このモードを使用することをお勧めします。
         */
        mScanSettingBuiler.setScanMode(SCAN_MODE_LOW_LATENCY) ;

        ScanSettings mScanSettings = mScanSettingBuiler.build() ;

        // 作成したスキャンフィルタとモードでスキャン開始
        mBluetoothLeScanner.startScan(mScanFilters, mScanSettings, mScanCallback) ;

        mHandler.postDelayed( new Runnable() {
            @Override
            public void run() {
                // SCAN_PERIOD 秒後に呼ばれた時にスキャン中であれば端末を検索＆取得できなかったとみなす。
                if (mIsScaning) {
                    mListener.onScanFailed(-390831);
                    Sb_StopScan();
                    Logger.d(TAG, "scan in 20 sec");
                }
            }
        }, SCAN_PERIOD );

        mIsScaning = true;
    }

    /**
     *  スキャンを止める。mHandlerから全てのコールバックを削除しているので、
     *
     */
    public void Sb_StopScan()
    {
        if (mIsScaning) {
            try {
                mIsScaning = false;
                mBluetoothLeScanner.stopScan(mScanCallback);
                mHandler.removeCallbacksAndMessages(null);
            } catch (Exception e) {
                Logger.e(TAG, e.toString());
            }
        }
    }

    private ScanCallback mScanCallback = new ScanCallback()
    {
        @Override
        protected Object clone() throws CloneNotSupportedException {
            return super.clone();
        }

        @Override
        public void onBatchScanResults(final List<ScanResult> results) {
            Logger.d(TAG, "onBatchScanResults");
        };

        @Override
        public void onScanFailed(final int errorCode) {
            Logger.d(TAG, "onScanFailed　errorCode = " + errorCode);
            runOnUiThread( new Runnable() {
                @Override
                public void run() {
                    mListener.onScanFailed(errorCode);
                }
            });
        };

        @Override
        public void onScanResult(int callbackType, ScanResult result) {
            Logger.d(TAG, "onScanResult　callbackType = " + callbackType);
            if(!mIsScaning){
                Logger.d(TAG, "mIsScaning is false");
                return;
            }
            BluetoothDevice device = result.getDevice() ;
            byte[] scanRecord = result.getScanRecord().getBytes();

            if (device.getBondState() != BluetoothDevice.BOND_BONDED) {
                long serial[] = new long[1] ;
                int security[] = new int[1] ;
                int ch1Data[] = new int[1] ;
                int ch2Data[] = new int[1] ;
                int batLevel[] = new int[1] ;
                StringBuffer localName = new StringBuffer() ;
                int rtn ;

                String devName = device.getName() ;
                if(devName == null) {
                    Logger.d(TAG,"deviceName is null");
                    return;
                }

                // 検索範囲を調整したい場合は、rssiの閾値を調整する（ getRssiは、電波強度を返す。dBm マイナスが小さいほど電波が強い）
                //if(result.getRssi() < -50)
                //    return ;
                Logger.d(TAG,"Rssi = " + result.getRssi());

                rtn = Sb_Parse_ScanRecord(scanRecord, serial, localName, security, ch1Data, ch2Data, batLevel) ;
                if(rtn == 0) {

                    final String serialStr = String.format("%X", (int)(serial[0]));

                    // 温度
                    final String ch1Str ;
                    if(ch1Data[0] == 0xEEEE) {
                        //ch1Str = "----";
                        return;
                    } else if(ch1Data[0] == 0xEEE0) {
                        //ch1Str = "";
                        return;
                    } else {
                        ch1Str = String.format("%.1f", (double) (ch1Data[0] - 1000) / 10.0);
                    }

                    runOnUiThread( new Runnable() {
                        @Override
                        public void run() {
                            mListener.onScanResult(serialStr, ch1Str);
                        }
                    });
                }
            }
        };
    };


    // ************************************************************************
    // int Sb_Parse_ScanRecord(final byte[] scanRecord, long serialNo[],
    //                         StringBuffer devName, int security[],
    //                         int ch1Data[], int ch2Data[], int batLevel[])
    // 概要 スキャンデータを解析し、TR-4シリーズかどうかを調べる。
    // 引数  :IN : scanRecord  : スキャンデータ
    //       :OUT: serialNo    : 機器シリアル番号
    //             devName     : 機器名称
    //             security    : セキュリティ
    //             ch1Data     : Ch1データ(素データ形式)
    //             ch2Data     : Ch2データ(素データ形式)=未使用
    //             batLevel    : 電池残量(1-5)
    //       :RET:             : 0=TR-4シリーズ -1=違う機器
    // ************************************************************************
    private int Sb_Parse_ScanRecord(final byte[] scanRecord, long serialNo[],
                                    StringBuffer devName, int security[],
                                    int ch1Data[], int ch2Data[], int batLevel[])
    {
        // [長さ][AD Type][データ][長さ][AD Type][データ]
        //    02     01     06     1b      ff     .....
        int pt = 0 ;
        int companyCode ; // 会社コード
        int formatCode ; // データフォーマット番号

        try {
            pt = (int)scanRecord[0] + 1 ;

            // 会社コード
            pt += 2 ;
            companyCode = Sb_Conv2ByteToInt(scanRecord[pt], scanRecord[pt+1]) ;
            if(companyCode != 0x0392) {
                //Logger.d(TAG,"CompanyCode Error");
                return -1;
            }

            // シリアル番号
            pt += 2 ;
            serialNo[0] = Sb_Conv4ByteToLong(scanRecord[pt], scanRecord[pt+1], scanRecord[pt+2], scanRecord[pt+3]) ;
            int devType = Sb_Get_DeviceType(serialNo[0]) ;
            if(devType != 0x16 && devType != 0x17 && devType != 0x18) {
                //Logger.d(TAG,"serialNo Error");
                return -1;
            }

            // セキュリティ
            pt += 4 ;
            security[0] = (int)scanRecord[pt] ;

            // データフォーマット番号
            pt += 1 ;
            formatCode = (int)scanRecord[pt] ;
            if(formatCode != 1) {
                //Logger.d(TAG,"formatCode Error");
                return -1;
            }

            // Ch1データ
            pt ++ ;
            ch1Data[0] = Sb_Conv2ByteToInt(scanRecord[pt], scanRecord[pt+1]) ;

            // Ch2データ(未使用)
            pt += 2 ;
            ch2Data[0] = Sb_Conv2ByteToInt(scanRecord[pt], scanRecord[pt+1]) ;

            // 電池残量
            pt += 2 ;
            batLevel[0] = (int)scanRecord[pt] ;

            // 機器名称
            devName.append(Sb_Get_Str_from_Byte(scanRecord, 33, 33+16, "UTF-8")) ;

            return 0 ;
        }
        catch (Exception e) {
            Logger.d(TAG,"" + e.getMessage());
            return -1 ;
        }

    }

    // ************************************************************************
    // int Sb_Conv2ByteToInt(byte b1, byte b2)
    // 概要 2ByteのunsignedリトルデータをINTへ変換
    // 引数  :IN :  b1      : 1Byte目
    //              b2      : 2Byte目
    //       :OUT:
    //       :RET:          : 変換した値
    // ************************************************************************
    private int Sb_Conv2ByteToInt(byte b1, byte b2)
    {
        int c1, c2 ;

        c1 = b1 ;
        c2 = b2 ;

        c1 = c1 & 0xFF ;
        c2 = c2 & 0xFF ;

        int val = c1 + (c2 << 8) ;

        return val ;
    }

    // ************************************************************************
    // int Sa_Conv4ByteToLong(byte b1, byte b2, byte b3, byte b4)
    // 概要 4Byteのunsignedリトルデータをintへ変換
    // 引数  :IN :  b1      : 1Byte目
    //              b2      : 2Byte目
    //              b3      : 3Byte目
    //              b4      : 4Byte目
    //       :OUT:
    //       :RET:          : 変換後の値
    // ************************************************************************
    private long Sb_Conv4ByteToLong(byte b1, byte b2, byte b3, byte b4)
    {
        long c1, c2, c3, c4 ;

        c1 = b1 ;
        c2 = b2 ;
        c3 = b3 ;
        c4 = b4 ;

        c1 = c1 & 0xFF ;
        c2 = c2 & 0xFF ;
        c3 = c3 & 0xFF ;
        c4 = c4 & 0xFF ;

        long val = c1 + (c2 << 8) + (c3 << 16) + (c4 << 24) ;

        return val ;
    }

    // ************************************************************************
    // int Sa_Get_DeviceType(long serial)
    // 概要 引数のシリアル番号から、TR4の機種タイプを返す。
    // 引数  :IN :  serial      : シリアル番号
    //       :OUT:
    //       :RET:              : TR41=0x16 TR42=0x17 TR45=0x18
    // ************************************************************************
    private int Sb_Get_DeviceType(long serial)
    {
        return (int) ((serial >> 17) & 0x7F) ;
    }

    // ************************************************************************
    // String Sb_Get_Str_from_Byte(final byte[] data, int start, int size,
    //                              String charsetName)
    // 概要 引数のバイト列からsize分の文字列を返す。
    // 引数  :IN :  data         : byte[]
    //              start        : 開始位置
    //              size         : 切り取るサイズ(byte)
    //              charsetName  : 文字コード("SJIS","UTF-8"など)
    //       :OUT:
    //       :RET:               : 切り出した文字列
    // ************************************************************************
    private String Sb_Get_Str_from_Byte(final byte[] data, int start,
                                        int size, String charsetName)
    {
        byte[] bTmp = new byte[size] ;
        String stTmp ;
        int i ;
        int c = 0 ;

        for(i=start; i<start+size; i++) {
            if(data[i] == 0x00)
                break ;
            else
                bTmp[c] = data[i] ;
            c++ ;
        }

        try {
            byte[] bStr = new byte[c] ;
            for(i=0; i<bStr.length; i++)
                bStr[i] = bTmp[i] ;
            stTmp = new String(bStr, charsetName) ;
        } catch (UnsupportedEncodingException e) {
            return "" ;
        }

        return stTmp ;
    }

    // ************************************************************************
    // String Sb_GetTimeStr(long timeData, String format)
    // 概要 UNIX時間から文字列時刻を返す。
    // 引数  :IN :  timeData    : UNIX時間(秒)
    //              format      : 日時フォーマット("MMM dd,yyyy HH:mm:ss" など)
    //       :OUT:
    //       :RET:              : 時刻文字列
    // ************************************************************************
    private String Sb_GetTimeStr(long timeData, String format)
    {
        String retStr = "" ;

        SimpleDateFormat sdf = new SimpleDateFormat(format) ;
        Date orgTime = new Date(timeData*1000) ;
        retStr = sdf.format(orgTime) ;

        return retStr ;
    }
}