package jp.agentec.abook.abv.bl.acms.client.parameters;

import java.sql.Timestamp;

import jp.agentec.abook.abv.bl.acms.type.DownloadStatusType;
import jp.agentec.adf.util.DateTimeFormat;
import jp.agentec.adf.util.DateTimeUtil;

/**
 * ダウンロードコンテンツに対するログを記録するためのパラメータです。
 * @author Taejin Hong
 * @version 1.0.0
 */
public class ContentDownloadLogParameters extends AcmsContentParameters {
	private Timestamp downloadTime;
	private Timestamp downloadEndTime;
	private int resourceVersion;
	private Long downloadSize;
	private int deviceTypeId;
	private DownloadStatusType downloadStatus;
	private String loginId; // Reader
	private String key; // Reader
	
	/**
	 * {@link ContentDownloadLogParameters} クラスのインスタンスを初期化します。
	 * @param sid ログインした時のセッションIDです。
	 * @param contentId　ダウンロードしたコンテンツのIDです。
	 * @param downloadTime　ダウンロードの開始時刻です。
	 * @param resourceVersion　ダウンロードしたコンテンツのリソースバージョンです。
	 * @param deviceTypeId　端末タイプIDです。（iPad:1, iPhone:2, Android:3）
	 * @param downloadStatus ダウンロード状態です。
	 * @since 1.0.0
	 */
	public ContentDownloadLogParameters(String sid, long contentId, Timestamp downloadTime, int resourceVersion, int deviceTypeId, DownloadStatusType downloadStatus) {
		super(sid, contentId);

		this.downloadTime = downloadTime;
		this.downloadEndTime = null;
		this.resourceVersion = resourceVersion;
		this.downloadSize = null;
		this.deviceTypeId = deviceTypeId;
		this.downloadStatus = downloadStatus;
	}
	
	public void setDownloadEndtime(Timestamp endTime) {
		this.downloadEndTime = endTime;
	}
	
	public void setDownloadSize(Long downloadSize) {
		this.downloadSize = downloadSize;
	}
	
	public void setDownloadStatus(DownloadStatusType downloadStatus) {
		this.downloadStatus = downloadStatus;
	}

	/**
	 * ダウンロードの開始時刻を返します。
	 * @return　ダウンロードの開始時刻です。
	 * @since 1.0.0
	 */
	public String getDownloadTime() {
		if (downloadTime == null) {
			return null;
		}
		return DateTimeUtil.toStringInTimeZone(downloadTime, DateTimeFormat.yyyyMMddHHmmss_hyphen, "GMT") + ",GMT";
	}

	/**
	 * ダウンロードの終了時刻を返します。
	 * @return　ダウンロードの終了時刻です。
	 * @since 1.0.0
	 */
	public String getDownloadEndTime() {
		if (downloadEndTime == null) {
			return null;
		}
		return DateTimeUtil.toStringInTimeZone(downloadEndTime, DateTimeFormat.yyyyMMddHHmmss_hyphen, "GMT") + ",GMT";
	}

	/**
	 * ダウンロードしたコンテンツのリソースバージョンを返します。
	 * @return　ダウンロードしたコンテンツのリソースバージョンです。
	 * @since 1.0.0
	 */
	public int getResourceVersion() {
		return resourceVersion;
	}

	/**
	 * ダウンロードしたサイズ（バイト）を返します。
	 * @return　ダウンロードしたサイズ（バイト）です。
	 * @since 1.0.0
	 */
	public Long getDownloadSize() {
		return downloadSize;
	}

	/**
	 * 端末タイプIDを返します。
	 * @return　端末タイプIDです。（iPad:1, iPhone:2, Android:3）
	 * @since 1.0.0
	 */
	public int getDeviceTypeId() {
		return deviceTypeId;
	}

	/**
	 * ダウンロード状態を返します。
	 * @return　ダウンロード状態です。
	 * @see #setDownloadStatus
	 * @since 1.0.0
	 */
	public String getDownloadStatus() {
		switch (downloadStatus) {
			case Succeeded:
				return "1";
			case Waiting:
			case Downloading:
			case Paused:
			case AutoPaused:
			case Initializing:
			case Failed:
			case Canceled:
			case None:
			default:
				return "0";
		}
	}

	public String getLoginId() {
		return loginId;
	}

	public void setLoginId(String loginId) {
		this.loginId = loginId;
	}

	public String getKey() {
		return key;
	}

	public void setKey(String key) {
		this.key = key;
	}
	
}
