package jp.agentec.abook.abv.ui.viewer.activity.theta;

import android.content.Intent;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.content.res.ResourcesCompat;
import android.util.TypedValue;
import android.view.KeyEvent;
import android.view.View;
import android.widget.Button;
import android.widget.SeekBar;
import android.widget.TextView;

import com.theta.view.MJpegInputStream;
import com.theta.view.MJpegView;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import jp.agentec.abook.abv.bl.common.constant.ABookKeys;
import jp.agentec.abook.abv.bl.common.constant.ABookValues;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.viewer.activity.theta.task.GetOptionExposureTask;
import jp.agentec.abook.abv.ui.viewer.activity.theta.task.SetOptionExposureTask;
import jp.agentec.abook.abv.ui.viewer.activity.theta.task.ShootTask;
import jp.agentec.abook.abv.ui.viewer.activity.theta.task.ShowLiveViewTask;

/**
 * THETAカメラ画面
 * @version 1.2.300
 * @since 2020/05/19
 * @author 金鎭星
 */
public class ThetaCameraActivity extends ThetaActivity {
    private static final String TAG = "ThetaCameraActivity";
    //画面表示後、ライブビューア取得を0.5秒後に取得
    private static final int LIVE_VIEW_START_DELAY = 500;
    private static final int LIVE_VIEW_FAIL_START_DELAY = 1000;
    private static final int EV_TEXT_VIEW_TEXT_SIZE_PHONE = 16;
    private static final int LIVE_VIDEO_FAIL_MAX_COUNT = 3;
    private MJpegView mLiveView;
    private ShowLiveViewTask mLivePreviewTask;
    private Button mShootBtn;
    private Button mImageListBtn;
    private TextView mExposureTextView;
    private SeekBar mExposureSeekBar;
    private int mCurrentSeekBarProgress;
    private int mLiveViewFailCount;
    private boolean mInitOnResumeFlg = false;
    private static final List<String> mExposureValues = new ArrayList<>(Arrays.asList("-2.0", "-1.7", "-1.3", "-1.0", "-0.7", "-0.3", "0.0", "0.3", "0.7", "1.0", "1.3", "1.7", "2.0"));

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.ac_theta_camera);
        mLiveView = findViewById(R.id.live_view);
        // 閉じるボタン
        final Button closeBtn = findViewById(R.id.btn_close);
        closeBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                Logger.v(TAG, "CloseBtn.onClick");
                buttonDoubleTapControl(closeBtn);
                mLiveView.stopPlay();
                if (deviceWifiEnable()) {
                    closeThetaCameraActivity();
                } else {
                    finish();
                }
            }
        });

        //撮影ボタン
        mShootBtn = findViewById(R.id.btn_theta_camera_shoot);
        mShootBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                buttonDoubleTapControl(mShootBtn);
                showProgressPopup();
                new ShootTask(ThetaCameraActivity.this).execute();
                mLiveView.setSource(null);
                mLiveViewFailCount = 0;
            }
        });

        //画像選択ボタン
        mImageListBtn = findViewById(R.id.btn_theta_camera_image_list);
        mImageListBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                buttonDoubleTapControl(mImageListBtn);
                Intent intent = new Intent();
                intent.setClassName(getPackageName(), ThetaImageListActivity.class.getName());
                startActivity(intent);
                mLiveViewFailCount = 0;
            }
        });
        //露出シークバー
        settingSeekbar();
        showProgressPopup();
        startLiveCameraTask(false);
        mLiveViewFailCount = 0;
        mLiveView.setLiveCameraListenerListener(new MJpegView.LiveCameraListener() {
            @Override
            public void liveCameraPlayFail() {
                mLiveViewFailCount++;
                Logger.e(TAG,"mLiveViewFailCount = " + mLiveViewFailCount);
                if (mLiveViewFailCount == LIVE_VIDEO_FAIL_MAX_COUNT) {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            mLiveView.setSource(null);
                            thetaConnectError(R.string.msg_theta_live_image_fail);
                        }
                    });
                    return;
                }

                startLiveCameraTask(true);
            }
        });
    }

    private void startLiveCameraTask(final boolean isFailRetry) {
        int delayTime = LIVE_VIEW_START_DELAY;
        if (isFailRetry) { //失敗時にはリトライを1秒後に行うように設定
            delayTime = LIVE_VIEW_FAIL_START_DELAY;
        }
        handler.postDelayed(new Runnable() {
            @Override
            public void run() {
                if (isFailRetry) {
                    showProgressPopup();
                }
                if (mLivePreviewTask != null) {
                    mLivePreviewTask.cancel(true);
                }
                mLivePreviewTask = new ShowLiveViewTask(ThetaCameraActivity.this);
                mLivePreviewTask.execute();
            }
        }, delayTime);
    }
    //端末の戻るボタン禁止
    @Override
    public boolean dispatchKeyEvent(KeyEvent event) {
        Logger.v(TAG, "dispatchKeyEvent %s", event);
            if (event.getAction()==KeyEvent.ACTION_UP) { // 戻るボタンを抑止
                if (event.getKeyCode() == KeyEvent.KEYCODE_BACK) {
                    return true;
                }
        }
        return super.dispatchKeyEvent(event);
    }

    @Override
    protected void onPause() {
        super.onPause();
        mLiveView.setSource(null);
    }

    @Override
    protected void onResume() {
        super.onResume();
        Logger.d(TAG, "super.onResume()");
        if (mInitOnResumeFlg) { //初回呼ばれた時は実行しない。（初期表示時）
            showProgressPopup();
            startLiveCameraTask(false);
        }
        mInitOnResumeFlg = true;
    }
    @Override
    protected void onDestroy() {
        if (mLivePreviewTask != null) {
            mLivePreviewTask.cancel(true);
        }
        super.onDestroy();
    }

    /**
     * シークバーを描画及び初期値設定
     */
    private void settingSeekbar() {
        mExposureTextView = findViewById(R.id.tv_exposure);

        //スマートフォンの場合、テキストサイズを小さく変更
        if (isNormalSize()) {
            mExposureTextView.setTextSize(TypedValue.COMPLEX_UNIT_SP, EV_TEXT_VIEW_TEXT_SIZE_PHONE);
        }

        //Android6.0以下の場合、古いシークバー利用
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.M) {
            mExposureSeekBar = findViewById(R.id.sb_exposure_old);
            findViewById(R.id.sb_exposure).setVisibility(View.GONE);
        } else {
            mExposureSeekBar = findViewById(R.id.sb_exposure);
            findViewById(R.id.sb_exposure_old).setVisibility(View.GONE);
            if (isNormalSize()) {
                Drawable icon = ResourcesCompat.getDrawable(getResources(),
                        R.drawable.ic_seek_bar_thumb_phone, null);
                mExposureSeekBar.setThumb(icon);
            }
        }

        mExposureSeekBar.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
                mExposureTextView.setText(getString(R.string.title_theta_exposure_value, mExposureValues.get(progress)));
            }
            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                int progress = seekBar.getProgress();
                if (progress != mCurrentSeekBarProgress) {
                    mCurrentSeekBarProgress = progress;
                    String exposure = mExposureValues.get(progress);
                    //サーバ通信
                    new SetOptionExposureTask(ThetaCameraActivity.this).execute(exposure);
                    Logger.d(TAG, "exposure" + exposure);
                }
                Logger.d(TAG, "progress = " + progress + ", mCurrentSeekBarProgress = " + mCurrentSeekBarProgress);
            }
            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {}

        });
    }

    /**
     * THETAカメラの撮影するShootTaskスレッド終了後に呼ばれる
     * @param fileId ファイルID
     */
    public void shootTaskFinish(final String fileId) {
        handler.post(new Runnable() {
            @Override
            public void run() {
                mShootBtn.setEnabled(true);
                if (fileId.equals(ABookValues.FAIL)) {
                    thetaConnectError(R.string.msg_theta_shoot_fail);
                } else {
                    closeProgressPopup();
                    Intent intent = new Intent();
                    intent.putExtra(ABookKeys.THETA_FILE_ID, fileId);
                    intent.setClassName(getPackageName(), ThetaImagePreviewActivity.class.getName());
                    startActivity(intent);
                }
            }
        });
    }

    /**
     * THETAカメラのライブ映像情報を取得するShowLiveViewTaskスレッド終了後に呼ばれる
     * @param mJpegInputStream Live画像データ
     */
    public void showLiveViewTaskFinish(MJpegInputStream mJpegInputStream) {

        if (mJpegInputStream != null) {
            mLiveView.setSource(mJpegInputStream);
            new GetOptionExposureTask(ThetaCameraActivity.this).execute();
        } else {
            Logger.e(TAG, "failed to start live view");
            thetaConnectError(R.string.msg_theta_live_image_fail);
        }
    }

    /**
     * THETAカメラのWi-Fi機能OFFにするFinishWlanTaskスレッド終了後に呼ばれる
     * @param result Wi-Fi設定OFF結果
     */
    public void finishWlanTaskFinish(String result) {
        if (result.equals(ABookValues.FAIL)) {
            Logger.e(TAG, "finishWlanTaskFinish fail");
        }
        closeProgressPopup();
        finish();
    }

    /**
     * THETAカメラの露出値を取得するGetOptionExposureTaskの終了時呼ばれる
     * @param exposure 露出値
     */
    public void getOptionExposureTaskFinish(String exposure) {
        if (exposure != null) {
            closeProgressPopup();
            int progress = mExposureValues.indexOf(exposure);
            if (progress != 1) {
                mExposureSeekBar.setProgress(progress);
            }
            mExposureTextView.setText(getString(R.string.title_theta_exposure_value, exposure));
        } else {
            thetaConnectError(R.string.msg_theta_get_exposure_fail);
            Logger.e(TAG, "getOptionExposureTaskFinish fail");
        }
    }

    /**
     *HETAカメラの露出値を設定するSetOptionExposureTaskの終了時呼ばれる
     * @param result 送信結果（Success:成功、fail:失敗）
     */
    public void setOptionExposureTaskFinish(String result) {
        if (!ABookValues.SUCCESS.equals(result)) {
            thetaConnectError(R.string.msg_theta_set_exposure_fail);
            Logger.e(TAG, "setOptionExposureTaskFinish fail");
        }
    }

    /**
     * 撮影したあと、THETAカメラと通信失敗時、
     * Wi-Fiが接続状態の場合、ライブ映像を再表示
     */
    @Override
    protected void thetaCameraWifiConnected() {
        if (mLiveView.getSource() == null) {
            if (mLivePreviewTask != null) {
                mLivePreviewTask.cancel(true);
                mLivePreviewTask = new ShowLiveViewTask(ThetaCameraActivity.this);
                mLivePreviewTask.execute();
            }
            mLiveView.play();
        }
    }
}
