package jp.agentec.abook.abv.bl.common;

import java.io.File;
import java.security.NoSuchAlgorithmException;
import java.util.Date;

import jp.agentec.abook.abv.bl.acms.client.json.CmsUrlJSON;
import jp.agentec.abook.abv.bl.acms.type.ContentZipType;
import jp.agentec.abook.abv.bl.common.log.LogLevel;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.common.nw.NetworkAdapter;
import jp.agentec.abook.abv.bl.logic.AbstractLogic;
import jp.agentec.abook.abv.bl.logic.UserAuthenticateLogic;
import jp.agentec.adf.security.cryptography.MD5;
import jp.agentec.adf.util.DateTimeUtil;
import jp.agentec.adf.util.FileUtil;

/**
 * アプリの定数・環境変数などを格納します。
 * @author Taejin Hong
 * @version 1.1.1
 */
public class ABVEnvironment {
	private static final String TAG = "ABVEnvironment";
	private static volatile ABVEnvironment instance = null;
	private boolean initialized = false;
	public String appVersion = null;
	public String acmsAddress = null;
	public String downloadServerAddress = null;
	public String defaultLogName = null;
	private LogLevel logLevel = null;
	public String mainFolderName = null;
	public boolean isContentProtected;
	public boolean isPdfThumbnailOutput;

	public boolean disableLogSend; // ログを送信しないフラグ（現状false固定）
	
	public NetworkAdapter networkAdapter;
	public String deviceId;
	public int deviceIdType;
	public String rootDirectory;
	public String externalStorageDirectory;
	public String cacheDirectory;
	public String encryptKey;
	public String screenSizeForAcmsParam;
	public String websocketServerHttpUrl;
	public String websocketServerWsUrl;
	public boolean websocketDebug;
	public int minPageLogSecond = 3;
//	public boolean logSend;
	public String allSubscriptionProductId; // 決済用に使用（定期購読（全可）の現在購入可能なプロダクトID）

	public int cacheSize;
	public String aspectType; // 16対9または4対3

	public boolean isCheckInvalidPasswordLimit = true;//　パスワード失敗回数チェックを行うか否か (R.bool.is_check_invalid_password_limit)

    // Serverから取得したcontentVersion時のlastFetchDateを一時的に保存するための変数
    // ダウンロード作業が完了したあと、テーブルに値を更新する。その後、再びクリアするようにする
	public String tempContentVersionLastFetchDate = "";

	// Serverから取得したcontentVersion時のリソースパターンを一時的に保存するための変数
	public int resourcePatternType;
	public boolean operationGroupMasterClearFlg;

	//////////////////////////////    定数    //////////////////////////////////
	
	private static final String ServerTimeKey = "ABook";
	public static final String ZipPasswordPrefix = "ABook+";		//	ABook+コンテンツID
	public static final String ZipPasswordPrefixForReader = "Reader+";		//	Reader用パスワードプリフィックス
	public static final String AbkDefaultZipPassword = "default";		//	Readerデフォルトパスワード
	public static final String LoginPasswordAESKey = "ABook+!\"#$+kooBA";	//	桁数を変更しないこと（１６桁）
	public static final String MacAddressAESKey = "aAg+b2o#n0T}ke3c";	//	桁数を変更しないこと（１６桁）
	public static final String APK_FILE_NAME = "abook_plus.apk";
	public static final String REPORT_SEND = "reportSend";	//	作業報告のディレクトリ名
	
	public static final int AppId = 3;
	public static final int DeviceTypeId = 3;
	
//	public static final String Content3DViewDirectoryName = "3d";
	
	public static final String ContentDirectoryFormat = "%s/ABook/contents/%d";
	public static final String ContentDetailDirectoryFormat = ContentDirectoryFormat + "/detail";
	public static final String ContentResourcesDirectoryFormat = ContentDirectoryFormat + "/resources";
	public static final String ContentMarkingDirectoryFormat = ContentDirectoryFormat + "/marking";
	public static final String ContentRemoteMarkingDirectoryFormat = ContentDirectoryFormat + "/remote";
	public static final String ContentDownloadsDirectoryFormat = "%s/ABook/downloads";
	public static final String ContentCacheDirectoryFormat = "%s/%d";
    public static final String WebCacheDirectoryFormat = "%s/web";
	
	public static final String Content3DzipNameFormat = "%s/images.zip";//	3Dzipファイル
	public static final String PDFThumbnailDirectoryFormat = "%s/pdfThumbnail";   //	PDF小サムネイルディレクトリ
	
	public static final String ContentVersionInfoJsonName = "json.txt";
	
	public static final String DatabaseExportNameFormat = "%s/ABook/export/database/ABVJE.db";
	public static final String LogExportName = "log/ExportLog.log";
	public static final String LogZipExportName = "ExportLog.zip";
	public static final String LogDeviceInfoPath = "log/deviceInfo.txt";
	public static final String AppLogPath = "log/appLogcat_%s.log";
	public static final String LogExportNameFormat = "%s/ABook/export/";
	public static final String ImportNameFormat = "%s/ABook/import/";
	public static final String LogUncaughtErrorName = "log/UncaughtErrReport.txt";
	public static final String LogDebugInfoName = "log/debugInfo.txt";
	public static final String LogHProfPath = "log/UncaughtErrReport.hprof";
	public static final String LogUncaughtErrorEncName = "UncaughtErrReport.zip";
	
	public static final String BackupFileDirectoryFormat = "%s/ABook/backup";
	public static final String TimeFileDirectoryFormat = "%s/ABook/time";
	
	public static final String CommonDirectoryFormat = "%s/ABook/common";

	public static final String OperationDirFormat = "%s/ABook/operation/%d";
	public static final String OperationTaskKeyDirFormat = "%s/ABook/operation/%d/%s";
	public static final String PanoImageDirFormat = "%s/panoImage";
	public static final String PanoImageFileFormat = "%s/panoImage.zip";
    public static final String TaskPdfDirFormat = "%s/taskPdf";
    public static final String TaskPdfFileFormat = "%s/taskPdf.zip";
	public static final String TaskListDirFormat = "%s/taskList";
	public static final String TaskListFileFormat = "%s/taskList.zip";
    public static final String OzdDirectoryFormat = "%s/ABook/contents/ozd/%d";

    public static final String OperationDirectionOrReportDirFormat = OperationDirFormat + "/%s/%s";
	public static final String CacheTempFormat = "%s/temp";
	public static final String TempTaskDirFormat = "%s/%d/temp";
    public static final String TempTaskKeyDirFormat = TempTaskDirFormat + "/%s";

    public static final String DefaultOzFileFormat = ContentCacheDirectoryFormat + "/%s";
    public static final String TempTaskFileFormat = TempTaskKeyDirFormat + "/%s";
	public static final String AttachedMoviesDirFormat = "%s/%d/attachedMovie";
    public static final String AttachedMoviesTaskTypeDir = AttachedMoviesDirFormat + "/%s/%s";
    public static final String AttachedMoviesRoutineTaskTypeDir = AttachedMoviesDirFormat + "/%s/%s/%d/%s";

	public static final String CacheTempAttachedImageFormat = "%s/temp/attachedImage";

	// #32926 作業報告画面改善 start
	// 作業報告レベル(taskReportLevel)によってディレクトリを追加する
	public static final String TaskReportLevelDir = "/%s";
	public static final String SavedOzFileFormat = OperationDirectionOrReportDirFormat + TaskReportLevelDir;
	public static final String RoutineTaskSavedOzFileFormat = OperationDirectionOrReportDirFormat + TaskReportLevelDir + "/%s/%s";
	public static final String OperationTaskKeyReportSendDirFormat = OperationTaskKeyDirFormat + "/reportSend/%d";
	public static final String OperationTaskKeyRoutineTaskReportDirFormat = OperationTaskKeyDirFormat + "/%d/%s/%s";
	// #32926 作業報告画面改善 end

	/**
	 * コンテンツのファイルを暗号化するときに、暗号化するサイズをバイト単位で指定します。
	 * @since 1.0.0
	 */
	public static final long ContentHeaderBytes = 100;
	
	/**
	 * コンテンツをダウンロードするときにの最低限必要な空き容量です。単位はMbyteです。
	 * @since 1.0.0
	 */
	public static final int MinFreeStorageSpaceMbyte = 100;

	/**
	 * パスワード有効期間。
	 * @since 1.0.0
	 */
	public static final int PasswordExpiryDate = 90;
	/**
	 * 最後のログインから次のログインまでに有効な期間
	 */
	public static final int LastLoginExpiryDate = 90;
	
	public static final int InvalidPasswordLimit = 5;
	/**
	 * ネットワークエラー以外で送信できなかった閲覧ログの保持日数
	 */
	public static final int SendErrorLogRetentionPeriod  = 30;
	
	/**
	 * コンテンツダウンロードをデバッグする時にtrueに設定します。
	 * trueにすると、zipファイルをSDカードにダウンロードし、解凍後削除しません。
	 * @since 1.0.0
	 */
	public static final boolean DebugContentDownload = false;

    public boolean enableToastMessage = true;

	// マスタ参照パス
	public static final String OperationMasterDataDirFormat = "%s/ABook/operation/master";

	private ABVEnvironment() {
	}
	
	public static ABVEnvironment getInstance() {
		if (instance == null) {
			synchronized (ServerTimeKey) {
				if (instance == null) {
					instance = new ABVEnvironment();
				}
			}
		}
		return instance;
	}
	
	///////////////////////////////////////　　　　   以下Getter/Setter　　　　 ///////////////////////////////////////

	public void setLogLevel(LogLevel logLevel) {
		if (logLevel != null) {
			this.logLevel = logLevel;
		} else {
			this.logLevel = LogLevel.info;
		}
	}

	public boolean isInitialized() {
		return initialized;
	}
	
	public void setInitialized(boolean initialized) {
		this.initialized = initialized;
	}

	public String getServerTimeKey()  {
		try {
			return MD5.getMd5Hash(ServerTimeKey).toUpperCase();
		} catch (NoSuchAlgorithmException e) {
			Logger.e(TAG, "getMd5Hash error", e);
			throw new RuntimeException(e);
		}
	}

	public LogLevel getLogLevel() {
		if (logLevel == null) {
			return LogLevel.debug;
		}
		
		return logLevel;
	}

	
	///////////////////////////////////////　　　　   以下ファイルパス関係　　　　 ///////////////////////////////////////
	
	
	/**
	 * コンテンツをビューアーに表示するために、一時的にリソースを展開するディレクトリを取得します。このディレクトリはセキュリティレベルが高いディレクトリを指定することをお勧めします。<br>
	 * Androidの場合、内臓メモリーが当てはまるでしょう。
	 * @return コンテンツをビューアーに表示するために、一時的にリソースを展開するディレクトリを取得します。
	 * @since 1.0.0
	 */
	public File getContentCacheDirectory(long contentId) {
		return new File(getContentCacheDirectoryPath(contentId));
	}
	
	/**
	 * コンテンツをビューアーに表示するために、一時的にリソースを展開するディレクトリを取得します。このディレクトリはセキュリティレベルが高いディレクトリを指定することをお勧めします。<br>
	 * Androidの場合、内臓メモリーが当てはまるでしょう。
	 * @return コンテンツをビューアーに表示するために、一時的にリソースを展開するディレクトリを取得します。
	 * @since 1.0.0
	 */
	public String getContentCacheDirectoryPath(long contentId) {
		if (!isContentProtected) {
			return getContentResourcesDirectoryPath(contentId, false);
		}
		return String.format(ContentCacheDirectoryFormat, cacheDirectory, contentId);
	}
	
	/**
	 * コンテンツファイルのルートディレクトリを取得します。ここには、コンテンツのサムネールファイルが保存されます。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツファイルのルートディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getContentDirectory(long contentId, boolean externalStorage) {
		return new File(getContentDirectoryPath(contentId, externalStorage));
	}

	/**
	 * OZDファイルの変更して保存した際のディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param createDir フォルダの作成有無
	 * @return
	 */
	public String getSavedOzdDirectoryPath(long contentId, boolean createDir) {
		if (createDir) {
			getOzdDirectory(OzdDirectoryFormat, contentId);
		}
		return String.format(OzdDirectoryFormat, rootDirectory, contentId);
	}

	private String getOzdDirectory(String dirFormat, long contentId) {
		String path = String.format(dirFormat, rootDirectory, contentId);
		FileUtil.createNewDirectory(path);
		return path;
	}
	/**
	 * コンテンツファイルのルートディレクトリを取得します。ここには、コンテンツのサムネールファイルが保存されます。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツファイルのルートディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getContentDirectoryPath(long contentId, boolean externalStorage) {
		return getContentDirectory(ContentDirectoryFormat, contentId, externalStorage);
	}

	/**
	 * コンテンツ詳細のページ毎のサムネールが保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのページ毎のサムネールが保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getContentDetailDirectory(long contentId, boolean externalStorage) {
		return new File(getContentDetailDirectoryPath(contentId, externalStorage));
	}
	
	/**
	 * コンテンツ詳細のページ毎のサムネールが保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのページ毎のサムネールが保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getContentDetailDirectoryPath(long contentId, boolean externalStorage) {
		return getContentDirectory(ContentDetailDirectoryFormat, contentId, externalStorage);
	}

	/**
	 * コンテンツのリソース（PDF等の素材）が保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのリソース（PDF等の素材）が保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getContentResourcesDirectory(long contentId, boolean externalStorage) {
		return new File(getContentResourcesDirectoryPath(contentId, externalStorage));
	}
	
	/**
	 * コンテンツのリソース（PDF等の素材）が保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのリソース（PDF等の素材）が保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getContentResourcesDirectoryPath(long contentId, boolean externalStorage) {
		return getContentDirectory(ContentResourcesDirectoryFormat, contentId, externalStorage);
	}
	
	/**
	 * コンテンツのマーキング画像ファイル情報が保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのPDFファイルのサムネールが保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getContentMarkingDirectory(long contentId, boolean externalStorage) {
		return new File(getContentMarkingDirectoryPath(contentId, externalStorage));
	}

	/**
	 * コンテンツのマーキング画像ファイルが保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのリソース（PDF等の素材）が保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getContentMarkingDirectoryPath(long contentId, boolean externalStorage) {
		return getContentDirectory(ContentMarkingDirectoryFormat, contentId, externalStorage);
	}
	
	/**
	 * コンテンツのマーキング画像ファイル情報が保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのPDFファイルのサムネールが保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getContentRemoteMarkingDirectory(long contentId, boolean externalStorage) {
		return new File(getContentRemoteMarkingDirectoryPath(contentId, externalStorage));
	}

	/**
	 * コンテンツのマーキング画像ファイルが保存されるディレクトリを取得します。
	 * @param contentId コンテンツIDを指定します。
	 * @param externalStorage 外部ストレージ（SDカード等）を利用する場合、trueを設定します。
	 * @return コンテンツのリソース（PDF等の素材）が保存されるディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getContentRemoteMarkingDirectoryPath(long contentId, boolean externalStorage) {
		return getContentDirectory(ContentRemoteMarkingDirectoryFormat, contentId, externalStorage);
	}
	
	/**
	 * コンテンツのダウンロードディレクトリを取得します。
	 * @return コンテンツのダウンロードディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getContentDownloadsDirectory() {
		return new File(getContentDownloadsDirectoryPath());
	}
	
	/**
	 * コンテンツのダウンロードディレクトリを取得します。
	 * @return コンテンツのダウンロードディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getContentDownloadsDirectoryPath() {
		String path = String.format(ContentDownloadsDirectoryFormat, rootDirectory);
		FileUtil.createNewDirectory(path);
		
		return path;
	}
	
	public String getBackupFileDirectoryPath() {
		String path = String.format(BackupFileDirectoryFormat, rootDirectory);
		FileUtil.createNewDirectory(path);
		return path;
	}
	
	public String getBackupTimeDirectoryPath() {
		String path = String.format(TimeFileDirectoryFormat, rootDirectory);
		FileUtil.createNewDirectory(path);
		
		return path;
	}

	public String getCommonDirectoryPath() {
		String path = String.format(CommonDirectoryFormat, rootDirectory);
		FileUtil.createNewDirectory(path);

		return path;
	}

	public String getContentDownloadPath(long contentId) {
		return String.format("%s/%s_%s.zip", getContentDownloadsDirectoryPath(), ContentZipType.ContentDownload, contentId);
	}

	private String getContentDirectory(String dirFormat, long contentId, boolean externalStorage) { // FIXME: externalStorageが無視されている
		String path = String.format(dirFormat, rootDirectory, contentId);
		
		FileUtil.createNewDirectory(path);
		
		return path;
	}

	/**
	 * データベースをエクスポートするディレクトリを取得します。
	 * @return データベースをエクスポートするディレクトリを返します。
	 * @since 1.0.0
	 */
	public File getDatabaseExportFile() {
		return new File(getDatabaseExportFileName());
	}

	/**
	 * データベースをエクスポートするディレクトリを取得します。
	 * @return データベースをエクスポートするディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getDatabaseExportFileName() {
		String path = String.format(DatabaseExportNameFormat, externalStorageDirectory);
		
		try {
			FileUtil.createParentDirectory(path);
		} catch (Exception e) {
			//	エラーになっても、ここでは何もしない。
		}
		
		return path;
	}
	
	/**
	 * Logをエクスポートするディレクトリを取得します。
	 * @return Logをエクスポートするディレクトリを返します。
	 * @since 1.0.0
	 */
	public String getLogExportFileName() {
		String path = String.format(LogExportNameFormat, externalStorageDirectory);
		
		try {
			FileUtil.createParentDirectory(path);
		} catch (Exception e) {
			//	エラーになっても、ここでは何もしない。
		}
		
		return path;
	}
	
	/**
	 * ３Ｄzipファイルの名前を収得します。
	 * @return ３Ｄzipファイルの名前を返します。
	 * @since 1.0.0
	 */
	public String getContent3DzipName(String contentDirectory) {
		return String.format(Content3DzipNameFormat, contentDirectory);
	}

	/**
	 * ３Ｄzipファイルのディレクトリを収得します。
	 * @return ３Ｄzipファイルのディレクトリを返します。
	 * @since 1.0.0
	 */
//	public String getContent3DDirectory(String contentDirectory) {
//		return String.format(Content3DDirectoryFormat, contentDirectory);
//	}

	/**
	 * PDFサムネイル（小）のディレクトリを取得します
	 * 
	 * @param contentId
	 * @return
	 */
	public String getPdfThumbnailDirectory(long contentId) {
		String path = String.format(PDFThumbnailDirectoryFormat, getContentCacheDirectoryPath(contentId));
		FileUtil.createNewDirectory(path);
		return path;
	}

	/**
	 * コンテンツ用のディレクトリを返す。
	 * （ディレクトリ作成は行わない）
	 * 
	 * @param contentId
	 * @return
	 */
	public String getContentDirectory(long contentId) {
		return String.format(ContentDirectoryFormat, rootDirectory, contentId);
	}

	/**
	 * abkファイルインポート用ディレクトリを取得
	 * 
	 * @return
	 */
	public String getImportDirectory() {
		return String.format(ImportNameFormat, rootDirectory);
	}
	
	public void setAddress(CmsUrlJSON cmsUrlJSON) {
		if (cmsUrlJSON != null) {
			acmsAddress = cmsUrlJSON.url;
			downloadServerAddress = cmsUrlJSON.url;
			websocketServerHttpUrl = cmsUrlJSON.syncHttpUrl;
			websocketServerWsUrl = cmsUrlJSON.syncWsUrl;
		}
		else {
			acmsAddress = null;
			downloadServerAddress = null;
			websocketServerHttpUrl = null;
			websocketServerWsUrl = null;
		}
	}

    public String getWebCacheDirectory() {
        String webCacheDir = String.format(WebCacheDirectoryFormat, cacheDirectory);
        FileUtil.createNewDirectory(webCacheDir);
        return webCacheDir;
    }

    public Date getCurrentTime(boolean isOnline) {
        Date serverTime;
        if (isOnline) {
            try {
                serverTime = AbstractLogic.getLogic(UserAuthenticateLogic.class).getServerTime();
            } catch (Exception e) {
                serverTime = DateTimeUtil.getCurrentDate();
            }
        } else {
            serverTime = DateTimeUtil.getCurrentDate();
        }
        return serverTime;
    }

	public String getPanoImageName(String contentDirectory) {
		return String.format(PanoImageFileFormat, contentDirectory);
	}

	public String getPanoImageDirName(String contentDirectory) {
		return String.format(PanoImageDirFormat, contentDirectory);
	}

    public String getTaskPdfDirName(String contentDirectory) {
        return String.format(TaskPdfDirFormat, contentDirectory);
    }

    public String getTaskPdfFileName(String contentDirectory) {
        return String.format(TaskPdfFileFormat, contentDirectory);
    }

	public String getTaskListDirName(String contentDirectory) {
		return String.format(TaskListDirFormat, contentDirectory);
	}

	public String getTaskListName(String contentDirectory) {
		return String.format(TaskListFileFormat, contentDirectory);
	}

	public String getOperationDirFile(long operationId) {
		return String.format(OperationDirFormat, rootDirectory, operationId);
	}

    // #32926 作業報告画面改善 start
	/**
	 * 保存されたOZファイルパス
	 * @param operationId
	 * @param taskKey
	 * @param fileName
	 * @param taskReportLevel
	 * @return ファイルパス
	 */
	public String getSavedOzFileTaskReportLevelPath(long operationId, String taskKey, String fileName, String taskReportLevel) {
		return String.format(SavedOzFileFormat, rootDirectory, operationId, taskKey, taskReportLevel, fileName);
	}
	// #32926 作業報告画面改善 end

    /**
     * 保存されたOZファイルパス、定期点検用
	 * 例）root/files/ABook/operation/[operation_id]/[task_key]/{taskReportLevel}/{taskReportId}/{reportStartDate}
     * @param operationId
     * @param taskKey
     * @param taskReportId
     * @param reportStartDate
     * @param fileName
     * @return
     */
    public String getRoutineTaskSavedOzFilePath(long operationId, String taskKey, int taskReportId, String reportStartDate, String fileName, String taskReportLevel) {
		return String.format(RoutineTaskSavedOzFileFormat, rootDirectory, operationId, taskKey, taskReportLevel, taskReportId, reportStartDate, fileName);
    }

    /**
     * テンプレートOZファイルパス
     * @param contentId
     * @param taskKey
     * @param fileName
     * @return ファイルパス
     */
    public String getDefaultOzFilePath(long contentId, String taskKey, String fileName) {
        return String.format(DefaultOzFileFormat, cacheDirectory, contentId, fileName);
    }

    /**
     * 臨時保存したファイルパス
     * @param contentId
     * @param taskKey
     * @param fileName
     * @return ファイルパス
     */
    public String getTempFilePath(long contentId, String taskKey, String fileName) {
		return String.format(TempTaskFileFormat, cacheDirectory, contentId, taskKey, fileName);
	}

	/**
	 * * 臨時保存したタスクキーのディレクトリファイルパス
	 * 例）../caches/[contentId]/temp/[task_key]
	 * @param contentId
	 * @param taskKey
	 * @return
	 */
	public String getTempTaskDirPath(long contentId, String taskKey) {
		return String.format(TempTaskKeyDirFormat, cacheDirectory, contentId, taskKey);
	}

	/**
	 * * 臨時保存したディレクトリファイルパス
	 * 例）../caches/[contentId]/temp
	 * @param contentId
	 * @return
	 */
	public String getTempDirPath(long contentId) {
		return String.format(TempTaskDirFormat, cacheDirectory, contentId);
	}

	/**
	 * プロジェクトの作業ディレクトリパス
	 * 例）root/files/ABook/operation/[operation_id]/[task_key]
	 * @param operationId
	 * @param taskKey
	 * @return
	 */
	public String getOperationTaskDirFilePath(long operationId, String taskKey) {
		return String.format(OperationTaskKeyDirFormat, rootDirectory, operationId, taskKey);
	}


	/**
	 * 報告送信ディレクトリパス
	 * 例）root/files/ABook/operation/[operation_id]/[task_key]/reportSend/{taskReportSendId}
	 * @param operationId
	 * @param taskKey
	 * @param taskReportSendId
	 * @return
	 */
	// taskReportLevelの値によってディレクトリを構成
    public String getOperationTaskReportSendDirFilePath(long operationId, String taskKey, int taskReportSendId) {
        return String.format(OperationTaskKeyReportSendDirFormat, rootDirectory, operationId, taskKey, taskReportSendId);
    }


	/**
	 * 定期点検で添付ファイルのパス
	 * 例）root/files/ABook/operation/[operation_id]/[task_key]/0/{taskReportId}/{reportStartDate}
	 * @param operationId
	 * @param taskKey
	 * @param taskReportId
	 * @param reportStartDate
	 * @return
	 */
    public String getRoutineTaskReportDirFilePath(long operationId, String taskKey, int taskReportId, String reportStartDate) {
        return String.format(OperationTaskKeyRoutineTaskReportDirFormat, rootDirectory, operationId, taskKey, Constant.TaskReportLevel.ReportType, taskReportId, getReportDate(reportStartDate));
    }

	/**
	 * 添付ファイル管理パス
	 * 例）root/files/ABook/operation/[operation_id]/[task_key]/{task_report_level}
	 * @param operationId
	 * @param taskKey
	 * @param taskReportLevel
	 * @return
	 */
	public String getOperationDirectionOrReportDirPath(long operationId, String taskKey, int taskReportLevel) {
		return String.format(OperationDirectionOrReportDirFormat, rootDirectory, operationId, taskKey, String.valueOf(taskReportLevel));
	}

	/**
	 * 作業報告レベルのディレクトリパス
	 * 例）root/files/ABook/operation/[operation_id]/[task_key]/[taskReportLevel]
	 * @param operationId
	 * @param taskKey
	 * @param taskReportLevel
	 * @return
	 */
	public String getOperationTaskReportLevelDirPath(long operationId, String taskKey, int taskReportLevel) {
		return String.format(OperationDirectionOrReportDirFormat, rootDirectory, operationId, taskKey, String.valueOf(taskReportLevel));
	}

	/**
	 * 添付動画の臨時保存パス
	 * 例）root/cache/{contentId}/attachedMovie
	 * @param contentId
	 * @return
	 */
	public String getAttachedMoviesFilePath(long contentId) {
		Logger.d(TAG, "cacheDirectory : " + cacheDirectory);
		return String.format(AttachedMoviesDirFormat, cacheDirectory, contentId);
	}

	// #32926 作業報告画面改善 start
	// taskReportLevelの値によってディレクトリを構成
	public String getAttachedDirectionOrReportDirPath(long contentId, String taskKey, int taskReportLevel) {
		return String.format(AttachedMoviesTaskTypeDir, cacheDirectory, contentId, taskKey, String.valueOf(taskReportLevel));
	}
	// #32926 作業報告画面改善 end

	/**
	 * 例）root/cache/{contentId}/attachedMovie/[task_key]/{task_report_level}/{task_report_id}/{report_start_date}
	 * @param contentId
	 * @param taskKey
	 * @param taskReportId
	 * @param reportStartDate
	 * @return
	 */
    public String getAttachedRoutineTaskReportDirPath(long contentId, String taskKey, int taskReportId, String reportStartDate) {
        return String.format(AttachedMoviesRoutineTaskTypeDir, cacheDirectory, contentId, taskKey, Constant.TaskReportLevel.ReportType,  taskReportId, getReportDate(reportStartDate));
    }

	public String getCacheTempAttachedImageDirPath() {
		return String.format(CacheTempAttachedImageFormat, cacheDirectory);
	}

	private String getReportDate(String reportDate) {
        if (reportDate.length() > 12) {
            return reportDate.substring(0, 12);
        } else {
            return reportDate;
        }
    }

    // MasterDataに対したJSONファイルの位置
	// 経路：root/files/operation/master
	public String getMasterFilePath() {
		return String.format(OperationMasterDataDirFormat, rootDirectory);
	}


	/**
	 * 作業種別のクリア判定フラグをセット
	 * @param clearFlg
	 */
	public void setOperationGroupMasterClearFlg(boolean clearFlg) {
		this.operationGroupMasterClearFlg = clearFlg;
	}

	/**
	 * 作業種別のクリア判定フラグ取得
	 * @return
	 */
	public boolean getOperationGroupMasterClearFlg() {
		return operationGroupMasterClearFlg;
	}
}
