package jp.agentec.abook.abv.bl.logic;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import jp.agentec.abook.abv.bl.acms.client.AcmsClient;
import jp.agentec.abook.abv.bl.acms.client.json.AcmsMessageJSON;
import jp.agentec.abook.abv.bl.acms.client.json.FixPushMessageJSON;
import jp.agentec.abook.abv.bl.acms.client.json.PushMessageListJSON;
import jp.agentec.abook.abv.bl.acms.client.parameters.AcmsParameters;
import jp.agentec.abook.abv.bl.acms.client.parameters.ReadPushMessageParameters;
import jp.agentec.abook.abv.bl.acms.client.parameters.SendPushMessageParameters;
import jp.agentec.abook.abv.bl.common.constant.ABookKeys;
import jp.agentec.abook.abv.bl.common.exception.AcmsException;
import jp.agentec.abook.abv.bl.common.exception.NetworkDisconnectedException;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.PushMessageDao;
import jp.agentec.abook.abv.bl.dto.FixPushMessageDto;
import jp.agentec.abook.abv.bl.dto.PushMessageDto;

/**
 * Created by kim jinsung on 2018/09/17.
 */

public class PushMessageLogic extends AbstractLogic {
    private static final String TAG = "OperationRelatedContentActivity";
    private PushMessageDao mPushMessageDao = AbstractDao.getDao(PushMessageDao.class);

    /**
     * サーバ側からプッシュメッセージを取得し、DB更新する。
     * @throws AcmsException ACMS関連例外
     * @throws NetworkDisconnectedException ネットワーク接続例外
     */
    public void initializePushMessage() throws AcmsException, NetworkDisconnectedException {
        AcmsParameters param = new AcmsParameters(cache.getMemberInfo().sid);
        PushMessageListJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getPushMessageList(param);
        List<PushMessageDto> serverPushMessageList = json.pushMessageList;
        List<PushMessageDto> localPushMessageList = mPushMessageDao.selectAll();
        //追加
        for (PushMessageDto serverPushMessageDto : serverPushMessageList) {
            boolean insertFlg = true;
            for (PushMessageDto localPushMessageDto : localPushMessageList) {
                if (serverPushMessageDto.pushMessageId.equals(localPushMessageDto.pushMessageId)) {
                    if (serverPushMessageDto.readingFlg && !localPushMessageDto.readingFlg) {
                        updateReadingFlg(localPushMessageDto.pushMessageId);
                    }
                    insertFlg = false;
                    break;
                }
            }
            if (insertFlg) {
                mPushMessageDao.insert(serverPushMessageDto);
            }
        }

        //削除
        List<PushMessageDto> deleteListDtoArray = new ArrayList();
        for (PushMessageDto localPushMessageDto : localPushMessageList) {
            boolean deleteFlg = true;
            for (PushMessageDto serverPushMessageDto : serverPushMessageList) {
                if (serverPushMessageDto.pushMessageId.equals(localPushMessageDto.pushMessageId)) {
                    deleteFlg = false;
                    break;
                }
            }
            if (deleteFlg) {
                mPushMessageDao.delete(localPushMessageDto.pushMessageId);
            }
        }
    }

    /**
     * 作業一覧上部に表示する未読プッシュメッセージ３つを返す
     * @return プッシュメッセージ情報配列
     */
    public List<PushMessageDto> getUnreadPushMessageList() {
        List<PushMessageDto> savedPushMessageList = mPushMessageDao.selectUnreadPushMessage();
        List<PushMessageDto> pushMessageList = new ArrayList();
        if (savedPushMessageList != null && savedPushMessageList.size() > 0) {
            for (PushMessageDto pushMessageDto : savedPushMessageList) {
                if (!pushMessageDto.readingFlg) {
                    pushMessageList.add(pushMessageDto);
                }
            }
        }
        return pushMessageList;
    }
    /**
     * プッシュメッセージの定型文取得
     * @return 定型文リスト
     * @throws IOException
     * @throws AcmsException
     */
    public List<FixPushMessageDto> getFixPushMessage() throws IOException, AcmsException, NetworkDisconnectedException {
        FixPushMessageJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).getFixPushMessage(cache.getMemberInfo().sid);
        return json.fixPushMessageDtoList;
    }

    /**
     * プッシュメッセージ送信依頼
     * @return 定型文リスト
     * @throws IOException
     * @throws AcmsException
     */
    public boolean sendPushMessageRequest(int sendType, String message, long operationId) throws IOException, AcmsException, NetworkDisconnectedException {
        SendPushMessageParameters param = new SendPushMessageParameters(cache.getMemberInfo().sid, operationId, sendType, message);
        AcmsMessageJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).sendPushMessageRequest(param);
        Logger.d(TAG, " status=" + json.httpStatus);
        if (json.httpStatus == ABookKeys.HTTP_STATUS_SUCCESS) {
            Logger.v(TAG, "sendPushMessageRequest httpStatus=" + json.httpStatus);
            return true;
        }
        return false;
    }

    public List<PushMessageDto> getAllPushMessageList() {
        return mPushMessageDao.selectAll();
    }

    public PushMessageDto getPushMessage(long pushMessageId) {
        return mPushMessageDao.select(pushMessageId);
    }

    public void updateReadingFlg(long pushMessageId) {
        mPushMessageDao.updateReadingFlg(pushMessageId);
    }

    /**
     * プッシュメッセージ既読情報を送信
     * @return
     * @throws IOException
     * @throws AcmsException
     */
    public boolean sendReadPushMessage(Long pushMessageId) throws IOException, AcmsException, NetworkDisconnectedException {
        ReadPushMessageParameters param = new ReadPushMessageParameters(cache.getMemberInfo().sid, pushMessageId);
        AcmsMessageJSON json = AcmsClient.getInstance(cache.getUrlPath(), networkAdapter).sendReadPushMessageRequest(param);
        Logger.d(TAG, " status=" + json.httpStatus);
        if (json.httpStatus == ABookKeys.HTTP_STATUS_SUCCESS) {
            Logger.v(TAG, "sendPushMessageRequest httpStatus=" + json.httpStatus);
            return true;
        }
        return false;
    }
}
