package jp.agentec.abook.abv.launcher.android;

import java.util.ArrayList;

import jp.agentec.abook.abv.bl.common.Constant.DeviceType;
import jp.agentec.abook.abv.bl.common.Constant.ReportType;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.dto.BluetoothPairingDeviceInfoDto;
import jp.agentec.abook.abv.bl.dto.OperationGroupMasterDto;
import jp.agentec.abook.abv.cl.util.PreferenceUtil;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType.DefPrefKey;
import jp.agentec.abook.abv.ui.common.appinfo.AppDefType.UserPrefKey;
import jp.agentec.adf.util.DateTimeUtil;
import jp.agentec.adf.util.StringUtil;

import android.content.Context;
import java.util.List;
import java.util.Stack;

public class ABVUIDataCache {
	private static final String TAG = "ABVUIDataCache";

	private Context context;
	
	// ホーム画面項目
	private int viewMode; // パンネル、リスト形式

	private ArrayList<Integer> mReportTypes;
	
	public long lastUpdateTime = -1;
	public String searchText;
	
	public boolean isFirstLaunching = true; // ログアウトした時もtrueに変える
	public boolean checkLocationServiceFlg = true; // アプリ起動一回につき一度位置情報アラートを表示する
	
	public boolean preIsCatalogEdition;

	// ユーザ情報コピー元コンテンツID
	public long srcContentId = -1;

	//コンテンツビューからコンテンツビューへ戻る時のコンテンツ情報リスト
	public List<long[]> returnContentIdList = new ArrayList<>();

	/**
	 * 最上部に表示されているActivityのクラス名、つまりユーザに見えているActivityのクラス名です。<br>
	 * Activityにfocusがあるかどうかをもとにアプリがバックグラウンドになったかを判断します。
	 */
	public String visibleActivityName = null;
	

    public ABVUIDataCache(Context context) {
    	this.context = context;
    	init();
    }

	private void init() {
    	viewMode = loadViewMode();
		operationGroupMasterStack = new Stack<>();
	}

    public long saveLastUpdateTime() {
    	lastUpdateTime = DateTimeUtil.getCurrentDate().getTime();
    	PreferenceUtil.putUserPref(context, UserPrefKey.LAST_UPDATE_TIME, lastUpdateTime);
    	return lastUpdateTime;
    }

	public void clear() {
		Logger.i(TAG, "clear all data");
		init();
		
	}

	public boolean getPermissionAccessLocation() {
		return PreferenceUtil.getUserPref(context, DefPrefKey.PERMISSION_ACCESS_LOCATION, false);
	}

	public boolean isShowedPermissionAccessLocationAlert() {
		return PreferenceUtil.getUserPref(context, UserPrefKey.SHOWED_PERMISSION_ACCESS_LOCATION_ALERT, false);
	}

	public int getViewMode() {
    	return PreferenceUtil.getUserPref(context, UserPrefKey.VIEW_MODE, 0);
//		return viewMode;
	}

	public void setViewMode(int viewMode) {
		this.viewMode = viewMode;
		PreferenceUtil.putUserPref(context, UserPrefKey.VIEW_MODE, viewMode);
	}

	private int loadViewMode() {
		return PreferenceUtil.getUserPref(context, UserPrefKey.VIEW_MODE, AppDefType.ViewMode.PANEL);
	}

	/**
	 * コンテンツビューからコンテンツビューへ
	 * 戻る時のコンテンツIDとページのセット
	 * @param contentId
	 * @param page
	 */
	public void setReturnContentIdList(long contentId, int page) {
			long contentInfo[] =new long[]{contentId,page};
			returnContentIdList.add(contentInfo);
	}

	//戻る時用のコンテンツIDとページ情報をリストで返す
	public List<long[]> getReturnContentIdList() { return returnContentIdList; }

	/**
	 * 戻る用コンテンツIDリストをリセット
	 */
	public void resetReturnContentIdList() {
		if (returnContentIdList.size() != 0) {
			returnContentIdList.clear();
		}
	}

	/**
	 * フィルターのタイプIDの値セット
	 * @param operationReportTypes
	 */
	public void setOperationReportTypes(ArrayList<Integer> operationReportTypes) {
		this.mReportTypes = operationReportTypes;
		String val = null;
		if (operationReportTypes.size() > 0) {
			// カンマ区切りで保存
			StringBuffer sb = new StringBuffer();
			for (int contentType : operationReportTypes) {
				sb.append(contentType);
				sb.append(",");
			}
			val = sb.substring(0, sb.length() - 1);
		}
		PreferenceUtil.putUserPref(context, UserPrefKey.OPERATION_REPORT_TYPES, val);
	}

	/**
	 * フィルターID情報リスト取得
	 * @return フィルター用のタイプIDリスト
	 */
	public ArrayList<Integer> getOperationReportTypes() {
		String operationReportTypesStr = PreferenceUtil.getUserPref(context, UserPrefKey.OPERATION_REPORT_TYPES, null);

		if (operationReportTypesStr != null) {
			ArrayList<Integer> operationReportTypes = new ArrayList<>();
			String[] operationReportTypesStrList = operationReportTypesStr.split(",");
			for (String contentType : operationReportTypesStrList) {
				if (!StringUtil.isNullOrEmpty(contentType)) {
					operationReportTypes.add(Integer.parseInt(contentType));
				}
			}
			mReportTypes = operationReportTypes;
		} else {
			mReportTypes = new ArrayList<Integer>();
			mReportTypes.add(ReportType.Report);
			mReportTypes.add(ReportType.RoutineTask);
			mReportTypes.add(ReportType.ReportReply);
		}

		return mReportTypes;
	}

	/**
	 * アプリ内で保存されてるbluetoothのペアリング情報の
	 * BluetoothPairingDeviceInfoDtoで返す。
	 *
	 * 情報が存在しない場合はnullで返す。
	 *
	 * @param bluetoothDeviceType  取得するbluetoothのデバイスタイプ
	 * @return BluetoothPairingDeviceInfoDto ペアリング情報
	 */
	public BluetoothPairingDeviceInfoDto getPairingBluetoothDeviceInfo(int bluetoothDeviceType) {
		BluetoothPairingDeviceInfoDto dto = new BluetoothPairingDeviceInfoDto();
		String deviceName = null;
		String deviceAddress = null;
		if (bluetoothDeviceType == DeviceType.centerThermomete) {
			// 中心温度計
			deviceName = PreferenceUtil.getUserPref(context, UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_NAME, "");
			deviceAddress = PreferenceUtil.getUserPref(context, UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_ADDRESS, "");
		} else if (bluetoothDeviceType == DeviceType.radiationThermomete) {
			// 放射温度計
			deviceName = PreferenceUtil.getUserPref(context, UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_NAME, "");
			deviceAddress = PreferenceUtil.getUserPref(context, UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_ADDRESS, "");
		}

		// deviceNameとdeviceAddressがセットなので、どっちかの値が存在しないとnullで返す。
		if (StringUtil.isNullOrEmpty(deviceName) || StringUtil.isNullOrEmpty(deviceAddress)) {
			return null;
		}

		dto.deviceName = deviceName;
		dto.deviceAddress = deviceAddress;
		dto.deviceType = bluetoothDeviceType;
		return dto;
	}

	/**
	 * 引数のデバイスタイプでローカルに保存されてるbluetoothのデバイスアドレス取得
	 * @param bluetoothDeviceType
	 * @return
	 */
	public String getPairingBluetoothDeviceAddress(int bluetoothDeviceType) {
		String deviceAddressKey = null;
		if (bluetoothDeviceType == DeviceType.centerThermomete) {
			deviceAddressKey = UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_ADDRESS;
		} else if (bluetoothDeviceType == DeviceType.radiationThermomete) {
			deviceAddressKey = UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_ADDRESS;
		}
		if (deviceAddressKey == null) {
			// 引数のデバイスタイプが定義した以外の値が入った場合
			return null;
		}

		return PreferenceUtil.getUserPref(context, deviceAddressKey, "");
	}

	/**
	 * 引数であるデバイスタイプリストに対するアプリ内で保存されてる
	 * bluetoothのペアリング情報リストを返す。
	 * @param deviceTypeList
	 * @return List<BluetoothPairingDeviceInfoDto> ペアリング情報リスト
	 */
	public List<BluetoothPairingDeviceInfoDto> getPairingBluetoothDeviceInfoList(List<Integer> deviceTypeList) {
		List<BluetoothPairingDeviceInfoDto> bluetoothPairingDeviceInfoDtoList = new ArrayList<BluetoothPairingDeviceInfoDto>();
		for (Integer deviceType : deviceTypeList) {
			// 引数のデバイスタイプのリスト全部を１つずつデータ取得
			BluetoothPairingDeviceInfoDto dto = getPairingBluetoothDeviceInfo(deviceType);
			if (dto != null) {
				// dtoがnullではない場合のみリストに追加
				bluetoothPairingDeviceInfoDtoList.add(dto);
			}
		}
		return bluetoothPairingDeviceInfoDtoList;
	}

	/**
	 * ペアリング情報をローカルに保存する
	 * deviceType毎に保存するキーが異なる
	 * @param pairingDeviceInfoDto
	 */
	public void setPairingBluetoothDeviceInfo(BluetoothPairingDeviceInfoDto pairingDeviceInfoDto) {
		if (pairingDeviceInfoDto.deviceType.equals(DeviceType.centerThermomete)) {
			// 中心温度計
			PreferenceUtil.putUserPref(context, UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_NAME, pairingDeviceInfoDto.deviceName);
			PreferenceUtil.putUserPref(context, UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_ADDRESS, pairingDeviceInfoDto.deviceAddress);
		} else if (pairingDeviceInfoDto.deviceType.equals(DeviceType.radiationThermomete)) {
			// 放射温度計
			PreferenceUtil.putUserPref(context, UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_NAME, pairingDeviceInfoDto.deviceName);
			PreferenceUtil.putUserPref(context, UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_ADDRESS, pairingDeviceInfoDto.deviceAddress);
		}
	}

	/**
	 * 引数であるデバイスタイプの
	 * ペアリング情報を削除する
	 * @param deviceType
	 */
	public void removePairingBluetoothDeviceInfo(int deviceType) {
		if (deviceType == DeviceType.centerThermomete) {
			// 中心温度計
			PreferenceUtil.removeUserPref(context, UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_NAME);
			PreferenceUtil.removeUserPref(context, UserPrefKey.BLE_DEVICE_CENTER_TEMPERATURE_ADDRESS);
		} else if (deviceType == DeviceType.radiationThermomete) {
			// 放射温度計
			PreferenceUtil.removeUserPref(context, UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_NAME);
			PreferenceUtil.removeUserPref(context, UserPrefKey.BLE_DEVICE_RADIATION_TEMPERATURE_ADDRESS);
		}
	}

    // 作業種別絞り込み用のスタック
    public Stack<OperationGroupMasterDto> operationGroupMasterStack;
	/**
	 * xmlにファイル書き込みされた
	 * 通常・作業種別のモード切り替えの値を取得
	 * @return
	 */
	public int getOperationGroupMasterMode() {
		return PreferenceUtil.getUserPref(context, UserPrefKey.OPERATION_GROUP_MASERT_MODE, 0);
	}

	/**
	 * 通常・作業種別のモード切り替えの値をxmlにファイル書き込みする
	 * @param operationGroupMasterMode
	 */
	public void setOperationGroupMasterMode(int operationGroupMasterMode) {
		PreferenceUtil.putUserPref(context, UserPrefKey.OPERATION_GROUP_MASERT_MODE, operationGroupMasterMode);

	}

	/**
	 * xmlに書き込みされた作業種別IDを取得
	 * @return
	 */
	public int getOperationGroupMasterId() {
		return PreferenceUtil.getUserPref(context, UserPrefKey.OPERATION_GROUP_MASERT_ID, 0);
	}

	/**
	 * ソート条件を取得
	 * @return ソートタイプ（デフォルト : 1(作業期間が新しい順)）
	 */
	public int getSortCondition() {
		return PreferenceUtil.getUserPref(context, UserPrefKey.OPERATION_SORT_CONDITION, 1);
	}

	/**
	 * xmlに作業種別IDをファイル書き込みする
	 * @param operationGroupMasterId
	 */
	public void setOperationGroupMasterId(int operationGroupMasterId) {
		PreferenceUtil.putUserPref(context, UserPrefKey.OPERATION_GROUP_MASERT_ID, operationGroupMasterId);
	}

	/**
	 * 保持していた作業種別のデータをクリアする
	 */
	public void clearOperationGroupMaster() {
		setOperationGroupMasterMode(AppDefType.OperationLocationType.ALL);
		setOperationGroupMasterId(0);
		operationGroupMasterStack.clear();
	}
}
