package jp.agentec.abook.abv.ui.common.view;

import android.app.ProgressDialog;
import android.content.Context;
import android.graphics.PixelFormat;

import java.util.Stack;

import jp.agentec.abook.abv.bl.common.ABVEnvironment;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.ContentDao;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.abook.abv.bl.dto.OperationDto;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.home.activity.OperationListActivity;

/**
 * Created by leej on 2019/08/26.
 */

public class ABVBatchSyncView extends ProgressDialog {
    private static final String TAG = "ABVBatchSyncView";
    private Stack<OperationDto> mBatchSyncOperationStack = new Stack<>();
    private ContentDao contentDao = AbstractDao.getDao(ContentDao.class);
    private OperationListActivity mOperationListActivity;

    // Activityが破棄された場合、処理を中止するフラグ
    private boolean isDestroy = false;

    public ABVBatchSyncView(Context context) {
        super(context);
        mOperationListActivity = (OperationListActivity)context;
        init();
        isDestroy = false;
    }

    // 初期化
    private void init() {
        if (getWindow() != null) {
            setIndeterminate(false);
            // キャンセルできないように設定・ナビゲーションバーのバックボタンも無効してくれる
            setCancelable(false);
            setProgressStyle(ProgressDialog.STYLE_HORIZONTAL); //プログレスバー表示
            getWindow().setFormat(PixelFormat.TRANSPARENT);
            setMessage(getContext().getResources().getString(R.string.batch_syncing));
        }
    }

    /**
     * スタックをセット（同期対象情報）
     * @param operationDtoStack
     */
    public void setStack(Stack<OperationDto> operationDtoStack) {
        setMax(operationDtoStack.size());
        mBatchSyncOperationStack = operationDtoStack;
    }

    /**
     * 閉じる処理
     */
    public void closeProgressDialog() {
        // プログレス値を0に初期化
        setProgress(0);
        dismiss();
    }

    /**
     * スタック中の空チェック
     * @return
     */
    public boolean isStackEmpty() {
        return mBatchSyncOperationStack.empty();
    }

    /**
     * 一括同期処理（stackの中を全て同期する）
     * ベース資料ダウンロードチェックして、未ダウンロードであればダウンロードして同期
     */
    public void batchOperationSyncForCheckDonwload() {
        if (isDestroy || isStackEmpty()) {
            // Activity破棄フラグがtrue、またはスタックが存在しなければ、全て完了と見做す
            closeProgressDialog();
            Logger.d(TAG, "---batchSync is end");
            if (mOperationListActivity != null && mOperationListActivity.mAutoSyncingFlg) {
                mOperationListActivity.autoSyncOperationDone(true);
            }
            return;
        }
        // ベース資料のダウンロードチェック
        OperationDto peekOperationDto = mBatchSyncOperationStack.peek();
        ContentDto contentDto = contentDao.getContent(peekOperationDto.contentId);
        if (contentDto == null) {
            Logger.e(TAG, "contentDto is null !");
            closeProgressDialog();
            return;
        }
        if (!contentDto.downloadedFlg || contentDto.updatedFlg) {
            if (!mOperationListActivity.operationContentDownload(contentDto)) {
                // error
                showBatchSyncErrorAlert(peekOperationDto, mOperationListActivity.getString(R.string.msg_batch_sync_content_download_fail));
            }
            if (mOperationListActivity != null && mOperationListActivity.mAutoSyncingFlg) {
                mOperationListActivity.autoSyncOperationDone(false);
            }
            return;
        }
        // 同期処理
        batchOperationSync();
    }

    /**
     * 一括同期処理（ベース資料はダウンロード済み）
     */
    public void batchOperationSync() {
        final OperationDto operationDto = mBatchSyncOperationStack.pop();
        String errorMessage = mOperationListActivity.syncOperation(operationDto.operationId, operationDto.reportType, false);
        if (errorMessage != null) {
            showBatchSyncErrorAlert(operationDto, errorMessage);
        } else {
            // 正常
            // 次のスタックがなければ、カウントしない
            if (!isStackEmpty()) {
                setProgress(getProgress() + 1);
            }
            // 次の作業を同期処理
            batchOperationSyncForCheckDonwload();
        }
    }

    // Activityが破棄された場合呼ばれる
    public void setActivityDestroy() {
        isDestroy = true;
    }

    /**
     * 一括同期処理時、エラーアラート表示
     * 一括同期中のプログレスバーは非表示にする
     * @param operationDto
     * @param errorMessage
     */
    public void showBatchSyncErrorAlert(OperationDto operationDto, String errorMessage) {
        // 異常
        closeProgressDialog();
        if (ABVEnvironment.getInstance().networkAdapter.isNetworkConnected()) {
            String convertErrorMsg = String.format(mOperationListActivity.getString(R.string.msg_batch_sync_error), operationDto.operationName) + errorMessage;
            mOperationListActivity.showSimpleAlertDialog(mOperationListActivity.getString(R.string.app_name), convertErrorMsg);
        } else {
            // ネットワーク通信エラー
            mOperationListActivity.showSimpleAlertDialog(mOperationListActivity.getString(R.string.app_name), mOperationListActivity.getString(R.string.msg_batch_sync_disconnect_network));
        }
    }
}
