package jp.agentec.abook.abv.ui.viewer.view;

import java.io.File;
import java.io.IOException;

import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.common.util.ContentFileUtil;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.launcher.android.R.id;
import jp.agentec.abook.abv.ui.common.appinfo.AppColor;
import jp.agentec.abook.abv.ui.common.constant.ErrorCode;
import jp.agentec.abook.abv.ui.common.constant.ErrorMessage;
import jp.agentec.abook.abv.ui.home.activity.GuideViewActivity;
import jp.agentec.abook.abv.ui.viewer.activity.ContentViewActivity;
import jp.agentec.abook.abv.ui.viewer.foxitPdf.FoxitPdfCore;
import jp.agentec.abook.abv.ui.viewer.foxitPdf.AsyncTask;
import jp.agentec.abook.abv.ui.viewer.foxitPdf.PdfImageProvider;
import jp.agentec.abook.abv.ui.viewer.foxitPdf.PdfThumbnailProvider;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.BitmapFactory.Options;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Handler;
import android.os.Process;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowInsets;
import android.view.WindowManager;
import android.view.WindowMetrics;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;


/**
 * サムネイルバー表示クラス
 * （小サムネイルと中サムネイルを扱う。
 * 大サムネイルはContentViewActivityのmMainLayout上に載る）
 * 
 * @author tsukada
 *
 */
public class ThumbnailSeekBarLayout extends LinearLayout {
	private static final String TAG = "ThumbnailSeekBarLayout";

	public static final int NORMAL_HEIGHT = 35;
	public static final int LARGE_HEIGHT = 70;
	
	private int marginS = 2;
	private int marginM = 0;
	private int marginL = 10;
	
	private int landWidthDiffMS;
	private int portWidthDiffMS;
	
	private FoxitPdfCore mFoxitPdfCore;
	private PdfThumbnailProvider mPdfThumbnailProvider;
	private PdfImageProvider pdfImageProvider;
	private AsyncTask<Integer,Void,Bitmap>[] drawTasks;
	private AsyncTask<Integer,Void,Bitmap> midThumTask;
	private long contentId;
	private int maxPageCount;
	private boolean isLandscapeView;
	private int displayWidth;
	private int displayHeight;
	protected int width;
	protected int height;
	
	private RelativeLayout mPageNumberLayout;
	private RelativeLayout seekBarLayout;
	private LinearLayout linearLayout;
	private ImageView midImageView;
	private Bitmap dummyThumWhite;
	private Bitmap dummyThumTransparent;
	private int firstLeftX;
	private int firstCenterX = -1;
	private int lastCenterX;
	private float interval;

	private int newPage = -1;
	private int[] indexToPages;
	private int maxThumCount;

	private boolean isReady;
	private boolean mSeekingFlag = false;


	public ThumbnailSeekBarLayout(Context context, FoxitPdfCore foxitPdfCore, long contentId, int maxPageCount, boolean isLandscapeView, int displayWidth, int displayHeight, boolean isNormalSize, RelativeLayout pageNumberLayout) {
		super(context);
		this.maxPageCount = maxPageCount;
		this.mFoxitPdfCore = foxitPdfCore;
		this.isLandscapeView = isLandscapeView;
		this.displayWidth = displayWidth;
		this.displayHeight = displayHeight;
		this.contentId = contentId;
		this.mPageNumberLayout = pageNumberLayout;
		mPdfThumbnailProvider = new PdfThumbnailProvider(context, foxitPdfCore, contentId, isNormalSize);
		mPdfThumbnailProvider.setIsLandscape(isLandscapeView);

		landWidthDiffMS = mPdfThumbnailProvider.landWidthM - mPdfThumbnailProvider.landWidthS;
		portWidthDiffMS = mPdfThumbnailProvider.portWidthM - mPdfThumbnailProvider.portWidthS;
		marginL = (int) (marginL * mPdfThumbnailProvider.getRatio(isNormalSize));
		marginS = (int) (marginS * mPdfThumbnailProvider.getRatio(isNormalSize));

		setBackgroundColor(context.getResources().getColor(AppColor.getSeekView()));
		
		LayoutInflater layoutInflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		seekBarLayout = (RelativeLayout)layoutInflater.inflate(R.layout.thumb_seekbar, null);
		seekBarLayout.setFitsSystemWindows(true);
		addView(seekBarLayout);
		
		linearLayout = (LinearLayout)seekBarLayout.findViewById(id.seekbar_liner);
		linearLayout.setFitsSystemWindows(true);
		
		width = isLandscapeView? mPdfThumbnailProvider.landWidthS: mPdfThumbnailProvider.portWidthS;
		height = isLandscapeView? mPdfThumbnailProvider.landHeightS: mPdfThumbnailProvider.portHeightS;
				
		maxThumCount = (displayWidth - marginS * 4 - 20) / (width + marginS * 2); // -20は余裕
		if (maxThumCount > maxPageCount) {
			maxThumCount = maxPageCount;
		}
		
		calcPageMapping(maxPageCount, maxThumCount); // サムネイル番号とページの対応
		dummyThumWhite = mPdfThumbnailProvider.createDummyThum(width, height, true); // ダミーのサムネイル作成(白色)
		dummyThumTransparent = mPdfThumbnailProvider.createDummyThum(width, height, false); // ダミーのサムネイル作成（透明）
		addSmallThumView(); // 小サムネイルビュー生成
	}

	public void setPdfImageProvider(PdfImageProvider pdfImageProvider) {
		this.pdfImageProvider = pdfImageProvider;
		mPdfThumbnailProvider.setPdfImageProvider(pdfImageProvider);
	}

	public void createThumbnail() {
		if (isReady) {
            return;
        }
		
		cancelTask();
		if (pdfImageProvider != null) {
			pdfImageProvider.setPauseTask(true);
		}

		drawTasks = new AsyncTask[maxThumCount];
		for (int i=0; i < maxThumCount; i++) {
			// Render the page in the background
			drawTasks[i] = new AsyncTask<Integer,Void,Bitmap>(Process.THREAD_PRIORITY_LOWEST) {
				private int index;
				@Override
                protected Bitmap doInBackground(Integer... v) {
					index = v[0];
					try {
						Thread.sleep(10); // 他のスレッドに実行権を渡す
						Thread.currentThread().setPriority(Thread.MIN_PRIORITY);
					} catch (InterruptedException e) {}
					try {
						return mPdfThumbnailProvider.getThumbnail(indexToPages[index], true);
					} catch (Exception e) {
						Logger.e(TAG, "PdfThumbnailProvider", e);
						ErrorMessage.showErrorMessageToast(ThumbnailSeekBarLayout.this.getContext(), ErrorCode.E107);
						return null;
					}
					
				}
				
				@Override
                protected void onPostExecute(Bitmap pageBitmap) {
					Logger.v(TAG, "onPostExecute");
					((ImageView)linearLayout.getChildAt(index)).setImageDrawable(null);
					((ImageView)linearLayout.getChildAt(index)).setImageBitmap(pageBitmap);
					linearLayout.getChildAt(index).invalidate();
					if (index == maxThumCount - 1) {
						isReady = true;
					}
					int nextIndex = index + 1;
					if (nextIndex < drawTasks.length) {
						drawTasks[nextIndex].executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, nextIndex);
					}
					else {
						if (pdfImageProvider != null) {
							pdfImageProvider.setPauseTask(false);
						}
					}
				}
			};
		}
		if (maxThumCount > 0) {
			drawTasks[0].executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, 0);
		}
	}
	
	public void cancelTask() {
		// Cancel pending render task
		if (drawTasks != null) {
			boolean hasCancel = false;
			for (AsyncTask<Integer,Void,Bitmap> drawTask : drawTasks) {
				if (drawTask.cancel(true)) {
					hasCancel = true;
				}
			}
			if (pdfImageProvider != null && hasCancel) {
				pdfImageProvider.setPauseTask(false);
			}
			
			drawTasks = null;
		}
	}

	/**
	 * 小サムネイルを追加する
	 *
	 */
	private void addSmallThumView() {
		LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
		params.setMargins(marginS, marginS, marginS, marginS);
		params.gravity = Gravity.CENTER_VERTICAL;

		int insetLeft;
		int insetTop;
		WindowManager wm = (WindowManager) getContext().getSystemService(Context.WINDOW_SERVICE);
		if (wm != null && Build.VERSION.SDK_INT > Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
			WindowMetrics metrics = wm.getCurrentWindowMetrics();
			WindowInsets insets = metrics.getWindowInsets();

			android.graphics.Insets systemBars = insets.getInsets(
					WindowInsets.Type.systemBars() | WindowInsets.Type.displayCutout()
			);
			insetLeft = systemBars.left;
			insetTop = systemBars.top;
		} else {
			insetTop = 0;
			insetLeft = 0;
		}

		for (int i=0; i < maxThumCount; i++) {
			ImageView smallImageView = new ImageView(getContext());
			final int page = indexToPages[i];
			smallImageView.setLayoutParams(params);
			smallImageView.setImageBitmap(dummyThumWhite);
			smallImageView.setFitsSystemWindows(true);

			smallImageView.setOnTouchListener(new OnTouchListener() {

				@Override
				public boolean onTouch(View v, MotionEvent ev) {
					float x = ev.getRawX() - insetLeft;
					float y = ev.getRawY() - insetTop;
					Logger.d(TAG, "[onTouch]:page:" + page + " action=" + str(ev.getAction()) + ", count=" + ev.getPointerCount()+ " x=" + x + " y=" + y);

				    switch (ev.getAction() & MotionEvent.ACTION_MASK) {
				    case MotionEvent.ACTION_DOWN:
				    	// 中サイズのサムネイル表示
				    	showMidThum(page, ((View)v.getParent()).getLeft() + v.getLeft() - (isLandscapeView?landWidthDiffMS:portWidthDiffMS) / 2);
						// 大サイズのサムネイル表示
						if (getContext() instanceof ContentViewActivity) {
							int left = ((View)v.getParent()).getLeft() + v.getLeft() + v.getWidth() / 2 - ((ContentViewActivity)getContext()).getLThumViewSize() / 2;
							showLargeThum(page, left);
						} else {
							int left = ((View)v.getParent()).getLeft() + v.getLeft() + v.getWidth() / 2 - ((GuideViewActivity)getContext()).getLThumViewSize() / 2;
							showLargeThum(page, left);

						}
				        break;
				    case MotionEvent.ACTION_MOVE:
				    	mSeekingFlag = true;
				    	if (firstCenterX <= 0) { // 最初・最後のサムネール、ページ間の間隔を設定する
				    		loadInterval();
				    	}
				    	if (newPage == -1) {
				    		newPage = page;
				    	}
				    	int pageX = (int) (firstCenterX + newPage * interval); // 現在ページ(大サムネイルの表示ページ)のx座標（さらに正確にするには各小サムネイル間のページ数で割って計算すべきだが）
				    	Logger.d(TAG, "newPage=" + newPage+ " x=" + x + " y=" + y + " firstX=" + firstCenterX + " lastX=" + lastCenterX + " interval=" + interval + " pageX=" + pageX);

				    	if (x < pageX - interval && newPage > 0) { // 前のページ
				    		newPage = (int) ((x - firstCenterX + interval) / interval);
				    	}
				    	else if (x > pageX + interval && newPage < maxPageCount - 1) { // 次のページ
				    		newPage = (int) ((x - firstCenterX) / interval);
				    	}
				    	else {
				    		break;
				    	}

				    	if (newPage >= 0 && newPage < maxPageCount) {
				    		// 中サイズのサムネイル表示
				    		showMidThum(newPage, (int) (x - (isLandscapeView? mPdfThumbnailProvider.landWidthM: mPdfThumbnailProvider.portWidthM) / 2));
				    		// 大サイズのサムネイル表示
							if (getContext() instanceof ContentViewActivity) {
								showLargeThum(newPage, (int)(x - ((ContentViewActivity)getContext()).getLThumViewSize() / 2));
							} else {
								showLargeThum(newPage, (int)(x - ((GuideViewActivity)getContext()).getLThumViewSize() / 2));
							}

				    	}

				        break;
				    case MotionEvent.ACTION_UP:
				    	if (getContext() instanceof ContentViewActivity) {
							((ContentViewActivity)getContext()).jumpToPage(newPage>=0 && newPage<maxPageCount? newPage: page); // ページ遷移
							((ContentViewActivity)getContext()).setToolbarVisable(false);
							((ContentViewActivity)getContext()).removeLargeThumView();
						} else {
							((GuideViewActivity)getContext()).jumpToPage(newPage>=0 && newPage<maxPageCount? newPage: page); // ページ遷移
							((GuideViewActivity)getContext()).setToolbarVisable(false);
							((GuideViewActivity)getContext()).removeLargeThumView();
						}
						v.post(() -> {
							mSeekingFlag = false;
						});
				    	clearImageView();
				    	newPage = -1;
				        break;
				    }
				    return true;
			    }

				private String str(int action) { // TODO later　 共通化
				    switch (action & MotionEvent.ACTION_MASK) {
				    case MotionEvent.ACTION_DOWN:
				    	return "ACTION_DOWN";
				    case MotionEvent.ACTION_MOVE:
				    	return "ACTION_MOVE";
				    case MotionEvent.ACTION_UP:
				    	return "ACTION_UP";
				    }
					return "" + action;
				}
			});
			
			linearLayout.addView(smallImageView);
		}
	}

	/**
	 *  大サイズのサムネイル表示 (パフォーマンスのため、元のPDFに対するアスペクト比を維持せず、LandscapeかPortraitかのみでサイズ固定する)
	 * 
	 * @param page
	 * @param left 
	 */
	private void showLargeThum(int page, int left) {
		if (mFoxitPdfCore == null) { // ベースファイル無しの場合作成しない
			return;
		}
		int largeWidth = isLandscapeView? mPdfThumbnailProvider.landWidthL: mPdfThumbnailProvider.portWidthL;
		int largeHeight = isLandscapeView? mPdfThumbnailProvider.landHeightL: mPdfThumbnailProvider.portHeightL;
		
		Bitmap bitmap = null;
		try {
			if (pdfImageProvider != null) {
				pdfImageProvider.setPauseTask(true);
			}
			
			bitmap = readFromFile(page, false);
			if (bitmap == null) {
				bitmap = Bitmap.createBitmap(largeWidth, largeHeight, Config.ARGB_8888);
				bitmap = mFoxitPdfCore.drawPage(page, largeWidth, largeHeight, 0, 0, largeWidth, largeHeight); // PDF描画
			}
		} finally {
			if (pdfImageProvider != null) {
				pdfImageProvider.setPauseTask(false);
			}
		}

		Drawable drawable = new BitmapDrawable(getResources() ,bitmap);
		int lThumViewSize = 0;
		if (getContext() instanceof ContentViewActivity) {
			lThumViewSize = ((ContentViewActivity)getContext()).getLThumViewSize();
		} else {
			lThumViewSize = ((GuideViewActivity)getContext()).getLThumViewSize();
		}

		int landHeightL = mPdfThumbnailProvider.landHeightL;
		int landWidthL = mPdfThumbnailProvider.landWidthL;
		int portWidthL = mPdfThumbnailProvider.portWidthL;
		int portHeightL = mPdfThumbnailProvider.portHeightL;
		if (isLandscapeView) {
			drawable.setBounds(marginL , (lThumViewSize  - landHeightL) / 2, marginL + landWidthL, (lThumViewSize - landHeightL) / 2 + landHeightL);
		}
		else {
			drawable.setBounds((lThumViewSize - portWidthL) / 2, marginL, (lThumViewSize - portWidthL) / 2 + portWidthL, marginL + portHeightL);
		}

		int stickCenter = lThumViewSize / 2; // 突起物の中央座標
		if (left > displayWidth - lThumViewSize) { // 右端
			int diff = left - (displayWidth - lThumViewSize);
			left = displayWidth - lThumViewSize;
			stickCenter = stickCenter + diff;
		}
		else if (left < 0) { // 左端
			stickCenter = stickCenter + left;
			left = 0;
		}
		
		int pageNumHSize = isLandscapeView? 0: 20; // ページ番号表示サイズ（高さ）
		int top = displayHeight - linearLayout.getHeight() - lThumViewSize - pageNumHSize - ContentViewActivity.THUMB_STICK_SIZE;
		if (getContext() instanceof ContentViewActivity) {
			((ContentViewActivity)getContext()).removeLargeThumView();
			((ContentViewActivity)getContext()).createLargeThumView(page, left, top, stickCenter, pageNumHSize, drawable);
		} else {
			((GuideViewActivity)getContext()).removeLargeThumView();
			((GuideViewActivity)getContext()).createLargeThumView(page, left, top, stickCenter, pageNumHSize, drawable);
		}
	}
	
	private void changePageNumberDisp(int pageNumber) {
		TextView pageNumberTextView = (TextView)mPageNumberLayout.findViewById(id.txt_page_number);
		//noinspection deprecation(API16から非推奨になった。無視)
		pageNumberTextView.setBackgroundDrawable(getResources().getDrawable(R.drawable.rounded_edittext));
		pageNumberTextView.setText((pageNumber + 1) + " / " + maxPageCount);
	}
	
	
	/**
	 * ファイルから読み込んで返す
	 * 
	 * @param page
	 * @param useSmall
	 * @return
	 */
	private Bitmap readFromFile(int page, boolean useSmall) {
		Bitmap bm = null;
		String imagePath = ContentFileUtil.getPdfImagePath(contentId, page); // まず大イメージ
		File file = new File(imagePath);
		if (file.exists() && file.length() > 1000) { // 
			Logger.v(TAG, "Found pdf image cache. path= " + imagePath);
			Options opt = new Options();
			opt.outWidth = width;
			opt.outHeight = height;
			try {
				bm = BitmapFactory.decodeFile(imagePath, opt);
			} catch (OutOfMemoryError e) {
				Logger.e(TAG, "readFromFile decodeFile error.", e); // ignore
			}
		}
		else {
			try {
				bm = mPdfThumbnailProvider.getThumbnail(page, false); // 続いて中イメージ
				if (bm == null && useSmall) { // なければ小サムネイル
					bm = mPdfThumbnailProvider.getThumbnail(page, true);
				}
			} catch (IOException e) {
				Logger.e(TAG, "getThumbnail error", e);
			}
		}
		return bm;
	}

	/**
	 * 中サムネイル表示
	 * 
	 * @param page
	 * @param left 
	 */
	private void showMidThum(final int page, int left) {
		final int midWidth = isLandscapeView? mPdfThumbnailProvider.landWidthM: mPdfThumbnailProvider.portWidthM;
		final int midHeight = isLandscapeView? mPdfThumbnailProvider.landHeightM: mPdfThumbnailProvider.portHeightM;
		
		if (midImageView == null) {
			midImageView = new ImageView(getContext());
			seekBarLayout.addView(midImageView);
		}
		else {
			clearImageView();
		}

		RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(midWidth, midHeight);
		params.setMargins(left, marginM, marginM, marginM);
		params.addRule(RelativeLayout.CENTER_VERTICAL, RelativeLayout.TRUE);
		midImageView.setLayoutParams(params);

		changePageNumberDisp(page); // ページ表示を更新する

		if (mFoxitPdfCore == null) { // ベースファイル無しの場合作成しない
			midImageView.setImageBitmap(dummyThumWhite);
			return;
		}
		
		midImageView.setImageBitmap(dummyThumTransparent);
		
		if (midThumTask != null) {
			midThumTask.cancel(true);
		}
		
		midThumTask = new AsyncTask<Integer,Void,Bitmap>(Process.THREAD_PRIORITY_DEFAULT){
			@Override
			protected Bitmap doInBackground(Integer... params) {
				Bitmap pageBitmap = null;
				try {
					if (pdfImageProvider != null) {
						pdfImageProvider.setPauseTask(true);
					}
					pageBitmap = readFromFile(page, true);
					if (pageBitmap == null) {
						pageBitmap = mFoxitPdfCore.drawPage(page, midWidth, midHeight, 0, 0, midWidth, midHeight); // PDF描画
					}
				} finally {
					if (pdfImageProvider != null) {
						pdfImageProvider.setPauseTask(false);
					}
				}
				return pageBitmap;
			}
			
			@Override
			protected void onPostExecute(Bitmap pageBitmap) {
				midImageView.setImageBitmap(pageBitmap);
				midImageView.invalidate();
			}
		};
		
		midThumTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, 0);
	}
	
	/**
	 * 指定ページの中サムネイルを表示する
	 * 
	 * @param page
	 */
	public void showMidThum(final int page) {
    	if (firstCenterX <= 0) { // 最初・最後のサムネール、ページ間の間隔を設定する
    		new Handler().postDelayed(new Runnable() {
				@Override
				public void run() {
		    		loadInterval();
		    		showMidThum(page, indexToPages);
				}}, 100); // TODO: later 描画後のタイミング、onDrawに実装するか？
    	}
    	else {
    		showMidThum(page, indexToPages);
    	}
	}

	/**
	 * 指定ページの中サムネイルを表示する
	 * 小サムネイルが表示されているページの場合は、その上に被さるように位置を合わせる。
	 * 小サムネイルが表示されていないページの場合は、ページ間のインターバルから計算する。
	 * 
	 * 
	 * @param page
	 * @param indexToPages
	 */
	private void showMidThum(final int page, int[] indexToPages) {
		int pageIndex = -1;
		for (int i = 0; i < indexToPages.length; i++) {
			if (indexToPages[i] == page) {
				pageIndex = i;
				break;
			}
		}
		if (pageIndex == -1) {
			showMidThum(page, (int) (linearLayout.getLeft() + firstCenterX + page * interval - (isLandscapeView? mPdfThumbnailProvider.landWidthM: mPdfThumbnailProvider.portWidthM) / 2));
		}
		else {
			ImageView imageView = (ImageView) linearLayout.getChildAt(pageIndex);
			showMidThum(page, linearLayout.getLeft() + imageView.getLeft() - (isLandscapeView?landWidthDiffMS:portWidthDiffMS) / 2);
		}
	}

	/**
	 * 表示可能なサムネイル枠は限度があるため、各サムネイル枠にどのページを載せるのかを決定する
	 * 
	 * @param maxPageCount
	 * @param maxThumCount
	 */
	private void calcPageMapping(int maxPageCount, int maxThumCount) {
		indexToPages = new int[maxThumCount];
		indexToPages[0] = 0; // 最初は最初のページ固定
		indexToPages[maxThumCount - 1] = maxPageCount - 1; //最後は最後のページ固定
		
		for (int i = 1; i < maxThumCount - 1; i++) {
			if (maxThumCount < maxPageCount) {
				indexToPages[i] = Math.round((maxPageCount - 1) * i / (maxThumCount -1));
			}
			else {
				indexToPages[i] = i;
			}
		}
	}

	private void loadInterval() {
		ImageView firstView = (ImageView) linearLayout.getChildAt(0);
		ImageView lastView = (ImageView) linearLayout.getChildAt(linearLayout.getChildCount() - 1);
		firstLeftX = firstView.getLeft();
		firstCenterX = firstLeftX + firstView.getWidth() / 2;
		lastCenterX = lastView.getLeft() + lastView.getWidth() / 2;
		interval = (lastCenterX - firstCenterX) / (float)(maxPageCount - 1);
	}

	public void clearImageView() {
		if (midImageView != null) {
			midImageView.setImageDrawable(null);
		}
	}

	public void destroy() {
		cancelTask();
		if (midThumTask != null) {
			midThumTask.cancel(true);
			midThumTask = null;
		}
		if (midImageView != null) {
			midImageView.setImageDrawable(null);
		}
		for (int i=0; i< linearLayout.getChildCount(); i++) {
			ImageView imageView = (ImageView) linearLayout.getChildAt(i);
			imageView.setImageDrawable(null);
		}
	}
	
	public boolean isSeeking() {
		return mSeekingFlag;
	}

}
