package jp.agentec.abook.abv.bl.logic;

import org.json.adf.JSONArray;
import org.json.adf.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.agentec.abook.abv.bl.common.constant.ABookCommConstants;
import jp.agentec.abook.abv.bl.data.dao.AbstractCommunicationDao;
import jp.agentec.abook.abv.bl.data.dao.AbstractDao;
import jp.agentec.abook.abv.bl.data.dao.ArchiveDao;
import jp.agentec.abook.abv.bl.data.dao.ChatGroupDao;
import jp.agentec.abook.abv.bl.data.dao.ChatMessageDao;
import jp.agentec.abook.abv.bl.data.dao.ChatRoomDao;
import jp.agentec.abook.abv.bl.data.dao.ContentDao;
import jp.agentec.abook.abv.bl.data.dao.ShopMemberDao;
import jp.agentec.abook.abv.bl.dto.ArchiveDto;
import jp.agentec.abook.abv.bl.dto.ChatGroupDto;
import jp.agentec.abook.abv.bl.dto.ChatMessageDto;
import jp.agentec.abook.abv.bl.dto.ChatRoomDto;
import jp.agentec.abook.abv.bl.dto.ContentDto;
import jp.agentec.abook.abv.bl.dto.ShopMemberDto;
import jp.agentec.adf.util.DateTimeUtil;
import jp.agentec.adf.util.StringUtil;

/**
 * @author Lee-mk
 */
public class CommunicationLogic extends AbstractLogic {
	private static final String TAG = "CommunicationLogic";

	private final ChatRoomDao chatRoomDao = AbstractCommunicationDao.getDao(ChatRoomDao.class);
	private final ChatMessageDao chatMessageDao = AbstractCommunicationDao.getDao(ChatMessageDao.class);
	private final ShopMemberDao shopMemberDao = AbstractCommunicationDao.getDao(ShopMemberDao.class);
	private final ChatGroupDao chatGroupDao = AbstractCommunicationDao.getDao(ChatGroupDao.class);
	private final ContentDao contentDao = AbstractDao.getDao(ContentDao.class);

	private String localFilePath;
	private final ArchiveDao archiveDao = AbstractCommunicationDao.getDao(ArchiveDao.class);

	/**
	 * {@link CommunicationLogic} クラスのインスタンスを初期化します。
	 * context　Androidとの互換性の為のプロパティです。Androidの android.content.Context のインスタンスを指定します。<br>
	 * UIがAndroidでない場合、何かのオブジェクトを指定しても、nullと見なします。
	 * @see AbstractLogic
	 * @since 1.2.3.6
	 */
	/*package*/ CommunicationLogic() {
	}

	public String getChatRoomList(Integer roomType, String keyword) {
		List<ChatRoomDto> chatRoomList;
		if (StringUtil.isNullOrEmpty(keyword) && !ABookCommConstants.FLAG.ROOM_TYPE.ALL.equals(roomType)) {
			chatRoomList = chatRoomDao.getAllChatRoom(roomType);
		} else {
			String[] replacedKeyword = keyword.replaceAll("　", " ").split(" ");
			chatRoomList = chatRoomDao.getAllChatRoombyKeyword(replacedKeyword);
		}
		JSONArray resultJsonArray = new JSONArray();
		for (ChatRoomDto chatRoomDto : chatRoomList) {
			Map<String, Object> chatRoomMap = new HashMap<String, Object>();
			chatRoomMap.put(ABookCommConstants.KEY.CHAT_ROOM_ID, chatRoomDto.chatRoomId);
			chatRoomMap.put(ABookCommConstants.KEY.CHAT_ROOM_NAME, chatRoomDto.chatRoomName);
			chatRoomMap.put(ABookCommConstants.KEY.TYPE, chatRoomDto.type);
			chatRoomMap.put(ABookCommConstants.KEY.FAVORITE_REGISTER_DATE, chatRoomDto.favoriteRegisterDate);
			chatRoomMap.put(ABookCommConstants.KEY.UNREAD_COUNT, chatRoomDto.unreadCount);
			if (ABookCommConstants.FLAG.MESSAGE_TYPE.PHOTO.equals(chatRoomDto.messageType) || ABookCommConstants.FLAG.MESSAGE_TYPE.VIDEO.equals(chatRoomDto.messageType)) {
				chatRoomMap.put(ABookCommConstants.KEY.MESSAGE, getFileUrlPath(chatRoomDto.message));
			} else {
				chatRoomMap.put(ABookCommConstants.KEY.MESSAGE, chatRoomDto.message);
			}
			chatRoomMap.put(ABookCommConstants.KEY.MESSAGE_TYPE, chatRoomDto.messageType);
			chatRoomMap.put(ABookCommConstants.KEY.INSERT_DATE, chatRoomDto.insertDate);
			chatRoomMap.put(ABookCommConstants.KEY.USER_COUNT, chatRoomDto.userCount);

			//ルームの参加者リスト取得。
			List<ShopMemberDto> userList = shopMemberDao.getUserInRoom(chatRoomDto.chatRoomId);
			JSONArray attendUsers = new JSONArray();
			for (ShopMemberDto user : userList) {
				Map<String, Object> userMap = new HashMap<String, Object>();
				userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, user.shopMemberId);
				userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(user.profileUrl));
				userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, user.shopMemberName);
				JSONObject jsonObject = new JSONObject(userMap);
				attendUsers.put(jsonObject);
			}

			chatRoomMap.put(ABookCommConstants.KEY.ATTEND_USERS, attendUsers);
			JSONObject jsonObject = new JSONObject(chatRoomMap);
			resultJsonArray.put(jsonObject);
		}
		return resultJsonArray.toString();
	}

	public String getUsersInRoom(Integer roomId) {
		List<ShopMemberDto> userList = shopMemberDao.getUserInRoom(roomId);
		JSONArray attendUsers = new JSONArray();
		for (ShopMemberDto user : userList) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, user.shopMemberId);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(user.profileUrl));
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, user.shopMemberName);
			JSONObject jsonObject = new JSONObject(userMap);
			attendUsers.put(jsonObject);
		}
		return attendUsers.toString();
	}
	public String getMessages(Integer roomId) {

		List<ChatMessageDto> chatMessageDtoList = chatMessageDao.getChatMessage(roomId);

		JSONArray resultJsonArray = new JSONArray();
		for (ChatMessageDto chatMessageDto : chatMessageDtoList) {
			Map<String, Object> chatMessageMap = new HashMap<String, Object>();
			chatMessageMap.put(ABookCommConstants.KEY.CHAT_ROOM_ID, chatMessageDto.chatRoomId);
			chatMessageMap.put(ABookCommConstants.KEY.LOGIN_ID, chatMessageDto.shopMemberName);
			chatMessageMap.put(ABookCommConstants.KEY.MESSAGE_ID, chatMessageDto.chatMessageId);
			chatMessageMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, chatMessageDto.shopMemberId);
			if (ABookCommConstants.FLAG.MESSAGE_TYPE.PHOTO.equals(chatMessageDto.messageType) || ABookCommConstants.FLAG.MESSAGE_TYPE.VIDEO.equals(chatMessageDto.messageType)) {
				chatMessageMap.put(ABookCommConstants.KEY.MESSAGE, getFileUrlPath(chatMessageDto.message));
			} else {
				chatMessageMap.put(ABookCommConstants.KEY.MESSAGE, chatMessageDto.message);
			}
			chatMessageMap.put(ABookCommConstants.KEY.MESSAGE_TYPE, chatMessageDto.messageType);
			chatMessageMap.put(ABookCommConstants.KEY.INSERT_DATE, chatMessageDto.insertDate);
			chatMessageMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(chatMessageDto.profileUrl));
			chatMessageMap.put(ABookCommConstants.KEY.SELF_FLG, chatMessageDto.selfFlg);
			chatMessageMap.put(ABookCommConstants.KEY.UNREAD_COUNT, chatMessageDto.unreadCount);
			JSONObject jsonObject = new JSONObject(chatMessageMap);
			resultJsonArray.put(jsonObject);
		}
		return resultJsonArray.toString();
	}

	public String getMessagesFromMessageId(Integer roomId, String messageId) {

		List<ChatMessageDto> chatMessageDtoList = chatMessageDao.getChatMessageFromMessageId(roomId, messageId);

		JSONArray resultJsonArray = new JSONArray();
		for (ChatMessageDto chatMessageDto : chatMessageDtoList) {
			Map<String, Object> chatMessageMap = new HashMap<String, Object>();
			chatMessageMap.put(ABookCommConstants.KEY.CHAT_ROOM_ID, chatMessageDto.chatRoomId);
			chatMessageMap.put(ABookCommConstants.KEY.LOGIN_ID, chatMessageDto.shopMemberName);
			chatMessageMap.put(ABookCommConstants.KEY.MESSAGE_ID, chatMessageDto.chatMessageId);
			chatMessageMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, chatMessageDto.shopMemberId);
			if (ABookCommConstants.FLAG.MESSAGE_TYPE.PHOTO.equals(chatMessageDto.messageType) || ABookCommConstants.FLAG.MESSAGE_TYPE.VIDEO.equals(chatMessageDto.messageType)) {
				chatMessageMap.put(ABookCommConstants.KEY.MESSAGE, getFileUrlPath(chatMessageDto.message));
			} else {
				chatMessageMap.put(ABookCommConstants.KEY.MESSAGE, chatMessageDto.message);
			}
			chatMessageMap.put(ABookCommConstants.KEY.MESSAGE_TYPE, chatMessageDto.messageType);
			chatMessageMap.put(ABookCommConstants.KEY.INSERT_DATE, chatMessageDto.insertDate);
			chatMessageMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(chatMessageDto.profileUrl));
			chatMessageMap.put(ABookCommConstants.KEY.SELF_FLG, chatMessageDto.selfFlg);
			chatMessageMap.put(ABookCommConstants.KEY.UNREAD_COUNT, chatMessageDto.unreadCount);
			JSONObject jsonObject = new JSONObject(chatMessageMap);
			resultJsonArray.put(jsonObject);
		}
		return resultJsonArray.toString();
	}

	public String searMessages(Integer roomId, String keyword, String userList) {
		String[] replacedKeyword = keyword.replaceAll("　", " ").split(" ");
		List<ChatMessageDto> chatMessageDtoList = chatMessageDao.searchChatMessage(roomId, replacedKeyword, userList);

		JSONArray resultJsonArray = new JSONArray();
		for (ChatMessageDto chatMessageDto : chatMessageDtoList) {
			Map<String, Object> chatMessageMap = new HashMap<String, Object>();
			chatMessageMap.put(ABookCommConstants.KEY.CHAT_ROOM_ID, chatMessageDto.chatRoomId);
			chatMessageMap.put(ABookCommConstants.KEY.LOGIN_ID, chatMessageDto.shopMemberName);
			chatMessageMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, chatMessageDto.shopMemberId);
			if (ABookCommConstants.FLAG.MESSAGE_TYPE.PHOTO.equals(chatMessageDto.messageType) || ABookCommConstants.FLAG.MESSAGE_TYPE.VIDEO.equals(chatMessageDto.messageType)) {
				chatMessageMap.put(ABookCommConstants.KEY.MESSAGE, getFileUrlPath(chatMessageDto.message));
			} else {
				chatMessageMap.put(ABookCommConstants.KEY.MESSAGE, chatMessageDto.message);
			}
			chatMessageMap.put(ABookCommConstants.KEY.MESSAGE_TYPE, chatMessageDto.messageType);
			chatMessageMap.put(ABookCommConstants.KEY.INSERT_DATE, chatMessageDto.insertDate);
			chatMessageMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(chatMessageDto.profileUrl));
			chatMessageMap.put(ABookCommConstants.KEY.SELF_FLG, chatMessageDto.selfFlg);
			chatMessageMap.put(ABookCommConstants.KEY.UNREAD_COUNT, chatMessageDto.unreadCount);
			JSONObject jsonObject = new JSONObject(chatMessageMap);
			resultJsonArray.put(jsonObject);
		}
		return resultJsonArray.toString();
	}

	public String getNameCardInfo(Integer shopMemberId) {
		JSONObject resultJson = new JSONObject();
		ShopMemberDto shopMemberDto = shopMemberDao.getShopMember(shopMemberId);
		List<ChatGroupDto> userGroupList = chatGroupDao.getUserGroups(shopMemberId);
		JSONArray groupPathJsonArray = new JSONArray();
		for (ChatGroupDto userGroup : userGroupList) {
			List<ChatGroupDto> gorupPathList = chatGroupDao.getGroupTree(userGroup.groupId);
			List<String> groupPathString = new ArrayList<String>();
			for (ChatGroupDto group : gorupPathList) {
				groupPathString.add(group.groupName);
			}
			Map<String, Object> groupMap = new HashMap<String, Object>();
			groupMap.put(ABookCommConstants.KEY.GROUP_ID, userGroup.groupId);
			groupMap.put(ABookCommConstants.KEY.GROUP_PATH, StringUtil.join(" / ", groupPathString));
			JSONObject jsonObject = new JSONObject(groupMap);
			groupPathJsonArray.put(jsonObject);
		}
		resultJson.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMemberDto.shopMemberId);
		resultJson.put(ABookCommConstants.KEY.LOGIN_ID, shopMemberDto.loginId);
		resultJson.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMemberDto.shopMemberName);
		resultJson.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMemberDto.profileUrl));
		resultJson.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathJsonArray);
		boolean hasRegisteredDate = !StringUtil.isNullOrEmpty(shopMemberDto.favoriteRegisterDate);
		resultJson.put(ABookCommConstants.KEY.IS_FAVORITE, hasRegisteredDate);

		return resultJson.toString();
	}

	public String getMyInfo() {
		ShopMemberDto myInfo = shopMemberDao.getMyInfo();
		List<String> groupPathList = chatGroupDao.getMyGroupPathList();
		JSONObject myInfoJson = new JSONObject();
		myInfoJson.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, myInfo.shopMemberId);
		myInfoJson.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, myInfo.shopMemberName);
		myInfoJson.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(myInfo.profileUrl));
		myInfoJson.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathList);
		return myInfoJson.toString();
	}

	public String getUserListByLoginId(String loginIdList) {
		if (loginIdList == null) {
			return null;
		}

		String[] userIdArray = loginIdList.split(",");
		List<ShopMemberDto> shopMemberDtoList = shopMemberDao.getUserListByLoginId(userIdArray);
		JSONArray joinedMemberArray = new JSONArray();
		for (ShopMemberDto dto : shopMemberDtoList) {
			Map<String, Object> joinedMemberMap = new HashMap<String, Object>();
			joinedMemberMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, dto.shopMemberId);
			joinedMemberMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, dto.shopMemberName);
			joinedMemberMap.put(ABookCommConstants.KEY.LOGIN_ID, dto.loginId);
			joinedMemberMap.put(ABookCommConstants.KEY.PROFILE_URL, dto.profileUrl);
			JSONObject myInfoJson = new JSONObject(joinedMemberMap);
			joinedMemberArray.put(myInfoJson);
		}

		return joinedMemberArray.toString();
	}

	public String getMyGroupUsers() {
		List<ChatGroupDto> myGroupIds = chatGroupDao.getMyGroups();
		JSONArray myGroupUsersJson = new JSONArray();
		for (ChatGroupDto chatGroupDto : myGroupIds) {
			JSONObject groupUserJson = new JSONObject();
			groupUserJson.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
			groupUserJson.put(ABookCommConstants.KEY.GROUP_NAME, makeGroupPath(chatGroupDto.groupId, " > "));
			List<ShopMemberDto> groupUsers = shopMemberDao.getUserListByGroupId(chatGroupDto.groupId);
			JSONArray groupUserArray = new JSONArray();
			for (ShopMemberDto groupUser : groupUsers) {
				Map<String, Object> myGroupUserMap = new HashMap<String, Object>();
				myGroupUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, groupUser.shopMemberId);
				myGroupUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, groupUser.shopMemberName);
				myGroupUserMap.put(ABookCommConstants.KEY.CHECKED, "");
				myGroupUserMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(groupUser.profileUrl));
				myGroupUserMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupUser);
				myGroupUserMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(groupUser.favoriteRegisterDate));
				JSONObject jsonObject = new JSONObject(myGroupUserMap);
				groupUserArray.put(jsonObject);
			}
			groupUserJson.put(ABookCommConstants.KEY.GROUP_USER_LIST, groupUserArray);
			myGroupUsersJson.put(groupUserJson);
		}
		return myGroupUsersJson.toString();
	}

	public String getMyGroupUsersNotInRoom(Integer roomId) {
		List<ChatGroupDto> myGroupIds = chatGroupDao.getMyGroups();
		JSONArray myGroupUsersJson = new JSONArray();
		for (ChatGroupDto chatGroupDto : myGroupIds) {
			JSONObject groupUserJson = new JSONObject();
			groupUserJson.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
			groupUserJson.put(ABookCommConstants.KEY.GROUP_NAME, chatGroupDto.groupName);
			List<ShopMemberDto> groupUsers = shopMemberDao.getUserListNotInRoomByGroupId(chatGroupDto.groupId, roomId);
			JSONArray groupUserArray = new JSONArray();
			for (ShopMemberDto groupUser : groupUsers) {
				Map<String, Object> myGroupUserMap = new HashMap<String, Object>();
				myGroupUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, groupUser.shopMemberId);
				myGroupUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, groupUser.shopMemberName);
				myGroupUserMap.put(ABookCommConstants.KEY.CHECKED, "");
				myGroupUserMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(groupUser.profileUrl));
				myGroupUserMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupUser);
				myGroupUserMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(groupUser.favoriteRegisterDate));
				JSONObject jsonObject = new JSONObject(myGroupUserMap);
				groupUserArray.put(jsonObject);
			}
			groupUserJson.put(ABookCommConstants.KEY.GROUP_USER_LIST, groupUserArray);
			myGroupUsersJson.put(groupUserJson);
		}
		return myGroupUsersJson.toString();
	}

	public String getMyGroupUserByName(String keyword) {

		String[] replacedKeyword = keyword.replaceAll("　", " ").split(" ");
		List<ShopMemberDto> selectedShopMembers = shopMemberDao.getUserInMyGroupByName(replacedKeyword);

		JSONArray resultJsonArray = new JSONArray();
		for (ShopMemberDto shopMember : selectedShopMembers) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			List<ChatGroupDto> groupPathList = chatGroupDao.getUserGroupPathList(shopMember.shopMemberId);
			JSONArray groupPathArray = new JSONArray();
			for (ChatGroupDto chatGroupDto : groupPathList){
				Map<String, Object> groupMap = new HashMap<String, Object>();
				groupMap.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
				groupMap.put(ABookCommConstants.KEY.GROUP_PATH, chatGroupDto.groupPath);
				JSONObject jsonObject = new JSONObject(groupMap);
				groupPathArray.put(jsonObject);
			}
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMember.shopMemberId);
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMember.shopMemberName);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMember.profileUrl));
			userMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathArray);
			userMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(shopMember.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(userMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getMyGroupUserNotInRoomByName(String keywords, Integer roomId) {

		String[] replacedKeyword = keywords.replaceAll("　", " ").split(" ");
		List<ShopMemberDto> selectedShopMembers = shopMemberDao.getUserInMyGroupNotInRoomByName(replacedKeyword, roomId);

		JSONArray resultJsonArray = new JSONArray();
		for (ShopMemberDto shopMember : selectedShopMembers) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			List<ChatGroupDto> groupPathList = chatGroupDao.getUserGroupPathList(shopMember.shopMemberId);
			JSONArray groupPathArray = new JSONArray();
			for (ChatGroupDto chatGroupDto : groupPathList){
				Map<String, Object> groupMap = new HashMap<String, Object>();
				groupMap.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
				groupMap.put(ABookCommConstants.KEY.GROUP_PATH, chatGroupDto.groupPath);
				JSONObject jsonObject = new JSONObject(groupMap);
				groupPathArray.put(jsonObject);
			}
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMember.shopMemberId);
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMember.shopMemberName);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMember.profileUrl));
			userMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathArray);
			userMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(shopMember.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(userMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getAllGroupShopMemberByName(String keyword) {
		String[] replacedKeyword = keyword.replaceAll("　", " ").split(" ");
		List<ShopMemberDto> selectedShopMembers = shopMemberDao.getUserInAllGroupByName(replacedKeyword);

		JSONArray resultJsonArray = new JSONArray();
		for (ShopMemberDto shopMember : selectedShopMembers) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			List<ChatGroupDto> groupPathList = chatGroupDao.getUserGroupPathList(shopMember.shopMemberId);
			JSONArray groupPathArray = new JSONArray();
			for (ChatGroupDto chatGroupDto : groupPathList){
				Map<String, Object> groupMap = new HashMap<String, Object>();
				groupMap.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
				groupMap.put(ABookCommConstants.KEY.GROUP_PATH, chatGroupDto.groupPath);
				JSONObject jsonObject = new JSONObject(groupMap);
				groupPathArray.put(jsonObject);
			}
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMember.shopMemberId);
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMember.shopMemberName);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMember.profileUrl));
			userMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathArray);
			userMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(shopMember.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(userMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getAllGroupShopMemberNotInRoomByName(String keywords, Integer roomId) {

		String[] replacedKeyword = keywords.replaceAll("　", " ").split(" ");
		List<ShopMemberDto> selectedShopMembers = shopMemberDao.getUserInAllGroupNotInRoomByName(replacedKeyword, roomId);


		JSONArray resultJsonArray = new JSONArray();
		for (ShopMemberDto shopMember : selectedShopMembers) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			List<ChatGroupDto> groupPathList = chatGroupDao.getUserGroupPathList(shopMember.shopMemberId);
			JSONArray groupPathArray = new JSONArray();
			for (ChatGroupDto chatGroupDto : groupPathList){
				Map<String, Object> groupMap = new HashMap<String, Object>();
				groupMap.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
				groupMap.put(ABookCommConstants.KEY.GROUP_PATH, chatGroupDto.groupPath);
				JSONObject jsonObject = new JSONObject(groupMap);
				groupPathArray.put(jsonObject);
			}
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMember.shopMemberId);
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMember.shopMemberName);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMember.profileUrl));
			userMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathArray);
			userMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(shopMember.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(userMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getGroupByName(String keyword) {
		String[] replacedKeyword = keyword.replaceAll("　", " ").split(" ");
		List<ChatGroupDto> searchGroups = chatGroupDao.getGroupByName(replacedKeyword);

		JSONArray resultJsonArray = new JSONArray();
		for (ChatGroupDto group : searchGroups) {
			Map<String, Object> groupMap = new HashMap<String, Object>();
			groupMap.put(ABookCommConstants.KEY.GROUP_NAME, group.groupName);
			groupMap.put(ABookCommConstants.KEY.GROUP_ID, group.groupId);
			groupMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(group.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(groupMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getMyGroupIds() {
		List<ChatGroupDto> myGroupList = chatGroupDao.getMyGroups();
		StringBuilder groupIds = new StringBuilder();
		for (ChatGroupDto myGroup : myGroupList) {
			if (!groupIds.toString().equals("")) {
				groupIds.append(",");
			}
			groupIds.append(myGroup.groupId);
		}
		return groupIds.toString();
	}

	public List<ChatGroupDto> getAllGroup() {
		return chatGroupDao.getAllGroups();
	}

	public ChatRoomDto getChatRoom(Integer roomId) {
		return chatRoomDao.getChatRoom(roomId);
	}
	public String getFavoriteUsers() {

		List<ShopMemberDto> favoriteUsers = shopMemberDao.getfavoriteUserList();

		JSONArray resultJsonArray = new JSONArray();
		for (ShopMemberDto favoriteUser : favoriteUsers) {
			Map<String, Object> favoriteUserMap = new HashMap<String, Object>();
			List<ChatGroupDto> groupPathList = chatGroupDao.getUserGroupPathList(favoriteUser.shopMemberId);
			JSONArray groupPathArray = new JSONArray();
			for (ChatGroupDto chatGroupDto : groupPathList){
				Map<String, Object> groupMap = new HashMap<String, Object>();
				groupMap.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
				groupMap.put(ABookCommConstants.KEY.GROUP_PATH, chatGroupDto.groupPath);
				JSONObject jsonObject = new JSONObject(groupMap);
				groupPathArray.put(jsonObject);
			}
			favoriteUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, favoriteUser.shopMemberId);
			favoriteUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, favoriteUser.shopMemberName);
			favoriteUserMap.put(ABookCommConstants.KEY.CHECKED, "");
			favoriteUserMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(favoriteUser.profileUrl));
			favoriteUserMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathArray);
			favoriteUserMap.put(ABookCommConstants.KEY.IS_FAVORITE, true);
			JSONObject jsonObject = new JSONObject(favoriteUserMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getFavoriteUsersNotInRoom(Integer roomId) {

		List<ShopMemberDto> favoriteUsers = shopMemberDao.getfavoriteUserListNotInRoom(roomId);

		JSONArray resultJsonArray = new JSONArray();
		for (ShopMemberDto favoriteUser : favoriteUsers) {
			Map<String, Object> favoriteUserMap = new HashMap<String, Object>();
			List<ChatGroupDto> groupPathList = chatGroupDao.getUserGroupPathList(favoriteUser.shopMemberId);
			JSONArray groupPathArray = new JSONArray();
			for (ChatGroupDto chatGroupDto : groupPathList){
				Map<String, Object> groupMap = new HashMap<String, Object>();
				groupMap.put(ABookCommConstants.KEY.GROUP_ID, chatGroupDto.groupId);
				groupMap.put(ABookCommConstants.KEY.GROUP_PATH, chatGroupDto.groupPath);
				JSONObject jsonObject = new JSONObject(groupMap);
				groupPathArray.put(jsonObject);
			}
			favoriteUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, favoriteUser.shopMemberId);
			favoriteUserMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, favoriteUser.shopMemberName);
			favoriteUserMap.put(ABookCommConstants.KEY.CHECKED, "");
			favoriteUserMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(favoriteUser.profileUrl));
			favoriteUserMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, groupPathArray);
			favoriteUserMap.put(ABookCommConstants.KEY.IS_FAVORITE, true);
			JSONObject jsonObject = new JSONObject(favoriteUserMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public String getFavoriteGroups() {

		List<ChatGroupDto> favoriteGroups = chatGroupDao.getFavoriteGroup();

		JSONArray resultJsonArray = new JSONArray();
		for (ChatGroupDto favoriteGroup : favoriteGroups) {
			Map<String, Object> favoriteGroupMap = new HashMap<String, Object>();
			favoriteGroupMap.put(ABookCommConstants.KEY.GROUP_NAME, favoriteGroup.groupName);
			favoriteGroupMap.put(ABookCommConstants.KEY.GROUP_ID, favoriteGroup.groupId);
			favoriteGroupMap.put(ABookCommConstants.KEY.IS_FAVORITE, true);
			JSONObject jsonObject = new JSONObject(favoriteGroupMap);
			resultJsonArray.put(jsonObject);
		}

		return resultJsonArray.toString();
	}

	public Integer getFavoriteCount() {
		List<ChatGroupDto> favoriteGroups = chatGroupDao.getFavoriteGroup();
		List<ShopMemberDto> favoriteUsers = shopMemberDao.getfavoriteUserList();
		return favoriteGroups.size() + favoriteUsers.size();
	}

	public String getGroupSearchData(Integer groupId) {

		//レスポンスになるJSON
		JSONObject groupSearchData = new JSONObject();

		//パラメータグループIDが0であれば自分のグループ情報を取得。
		ChatGroupDto myGroup = chatGroupDao.getMyGroups().get(0);
		ChatGroupDto rootGroup = chatGroupDao.getRootGroup();

		Integer targetGroupId;
		if (groupId == 0) {
			targetGroupId = rootGroup.groupId;
		} else {
			targetGroupId = groupId;
		}

		ChatGroupDto parentGroup = chatGroupDao.getParentGroup(targetGroupId);
		if (parentGroup != null) {
			groupSearchData.put(ABookCommConstants.KEY.PARENT_GROUP_ID, parentGroup.groupId);
		}
		if (rootGroup != null) {
			groupSearchData.put(ABookCommConstants.KEY.ROOT_GROUP_ID, rootGroup.groupId);
		}

		//グループ
		List<ChatGroupDto> groupPathList = chatGroupDao.getGroupTree(targetGroupId);
		JSONArray groupPathJSONArray = new JSONArray();
		for (ChatGroupDto group : groupPathList) {
			Map<String, Object> groupTreeMap = new HashMap<String, Object>();
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_NAME, group.groupName);
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_ID, group.groupId);
			JSONObject jsonObject = new JSONObject(groupTreeMap);
			groupPathJSONArray.put(jsonObject);
		}
		groupSearchData.put(ABookCommConstants.KEY.GROUP_PATH_LIST,groupPathJSONArray);

		List<ChatGroupDto> childGroupList = chatGroupDao.getGroupChildList(targetGroupId);
		JSONArray childGroupJSONArray = new JSONArray();
		for (ChatGroupDto group : childGroupList) {
			Map<String, Object> groupTreeMap = new HashMap<String, Object>();
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_NAME, group.groupName);
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_ID, group.groupId);
			groupTreeMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(group.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(groupTreeMap);
			childGroupJSONArray.put(jsonObject);
		}
		groupSearchData.put(ABookCommConstants.KEY.CHILD_GROUP_LIST,childGroupJSONArray);

		List<ShopMemberDto> shopMemberList = shopMemberDao.getUserListByGroupId(targetGroupId);

		JSONArray shopMemberJSONArray = new JSONArray();
		for (ShopMemberDto shopMember : shopMemberList) {
			Map<String, Object> shopMemberMap = new HashMap<String, Object>();
			shopMemberMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMember.shopMemberId);
			shopMemberMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMember.shopMemberName);
			shopMemberMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMember.profileUrl));
			shopMemberMap.put(ABookCommConstants.KEY.CHECKED, "");
			shopMemberMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, chatGroupDao.getUserGroupPathList(shopMember.shopMemberId));
			shopMemberMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(shopMember.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(shopMemberMap);
			shopMemberJSONArray.put(jsonObject);
		}
		groupSearchData.put(ABookCommConstants.KEY.GROUP_USER_LIST,shopMemberJSONArray);

		return groupSearchData.toString();
	}

	public String getGroupSearchDataForAddUser(Integer groupId, Integer roomId) {

		//レスポンスになるJSON
		JSONObject groupSearchData = new JSONObject();

		//パラメータグループIDが0であれば自分のグループ情報を取得。
		ChatGroupDto myGroup = chatGroupDao.getMyGroups().get(0);
		ChatGroupDto rootGroup = chatGroupDao.getRootGroup();

		Integer targetGroupId = groupId == 0 ? rootGroup.groupId : groupId;

		ChatGroupDto parentGroup = chatGroupDao.getParentGroup(targetGroupId);
		if (parentGroup != null) {
			groupSearchData.put(ABookCommConstants.KEY.PARENT_GROUP_ID, parentGroup.groupId);
		}
		if (rootGroup != null) {
			groupSearchData.put(ABookCommConstants.KEY.ROOT_GROUP_ID, rootGroup.groupId);
		}

		//グループ
		List<ChatGroupDto> groupPathList = chatGroupDao.getGroupTree(targetGroupId);
		JSONArray groupPathJSONArray = new JSONArray();
		for (ChatGroupDto group : groupPathList) {
			Map<String, Object> groupTreeMap = new HashMap<String, Object>();
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_NAME, group.groupName);
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_ID, group.groupId);
			JSONObject jsonObject = new JSONObject(groupTreeMap);
			groupPathJSONArray.put(jsonObject);
		}
		groupSearchData.put(ABookCommConstants.KEY.GROUP_PATH_LIST,groupPathJSONArray);

		List<ChatGroupDto> childGroupList = chatGroupDao.getGroupChildList(targetGroupId);
		JSONArray childGroupJSONArray = new JSONArray();
		for (ChatGroupDto group : childGroupList) {
			Map<String, Object> groupTreeMap = new HashMap<String, Object>();
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_NAME, group.groupName);
			groupTreeMap.put(ABookCommConstants.KEY.GROUP_ID, group.groupId);
			groupTreeMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(group.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(groupTreeMap);
			childGroupJSONArray.put(jsonObject);
		}
		groupSearchData.put(ABookCommConstants.KEY.CHILD_GROUP_LIST,childGroupJSONArray);

		List<ShopMemberDto> shopMemberList = shopMemberDao.getUserListNotInRoomByGroupId(targetGroupId, roomId);

		JSONArray shopMemberJSONArray = new JSONArray();
		for (ShopMemberDto shopMember : shopMemberList) {
			Map<String, Object> shopMemberMap = new HashMap<String, Object>();
			shopMemberMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMember.shopMemberId);
			shopMemberMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMember.shopMemberName);
			shopMemberMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(shopMember.profileUrl));
			shopMemberMap.put(ABookCommConstants.KEY.CHECKED, "");
			shopMemberMap.put(ABookCommConstants.KEY.GROUP_PATH_LIST, chatGroupDao.getUserGroupPathList(shopMember.shopMemberId));
			shopMemberMap.put(ABookCommConstants.KEY.IS_FAVORITE, !StringUtil.isNullOrEmpty(shopMember.favoriteRegisterDate));
			JSONObject jsonObject = new JSONObject(shopMemberMap);
			shopMemberJSONArray.put(jsonObject);
		}
		groupSearchData.put(ABookCommConstants.KEY.GROUP_USER_LIST,shopMemberJSONArray);

		return groupSearchData.toString();
	}

	public void insertChatRoomList(List<ChatRoomDto> roomList) {
		List<ChatMessageDto> insertMessageList = new ArrayList<ChatMessageDto>();
		for (ChatRoomDto chatRoomDto : roomList) {
			if (chatRoomDto.lastMessageInfo != null) {
				insertMessageList.add(chatRoomDto.lastMessageInfo);
			}
		}
		chatRoomDao.insertChatRoom(roomList);
		chatMessageDao.insertChatMessage(insertMessageList);
	}

	public void insertMessageList(List<ChatMessageDto> messageList) {
		chatMessageDao.insertChatMessage(messageList);
	}

	public void insertChatRoom(ChatRoomDto chatRoomDto) {
		chatRoomDao.insertChatRoom(chatRoomDto);
	}

	public void addUserInRoom(String userIds, Integer roomId) {
		String[] userIdArray = userIds.split(",");
		List<Integer> userIdList = new ArrayList<Integer>();
		for (String userId : userIdArray) {
			userIdList.add(Integer.parseInt(userId));
		}
		chatRoomDao.insertChatRoomUsers(userIdList, roomId);
	}

	public void updateChatRoomName(String roomName, Integer roomId) {
		chatRoomDao.updateChatRoomName(roomName, roomId);
	}

	public void insertShopMember(ShopMemberDto shopMemberDto) {
		ShopMemberDto myInfo = shopMemberDao.getMyInfo();
		if (myInfo.shopMemberId.equals(shopMemberDto.shopMemberId)) {
			shopMemberDto.selfFlg = 1;
		} else {
			shopMemberDto.selfFlg = 0;
		}
		shopMemberDao.insertShopMember(shopMemberDto);
	}


	public void insertShopMember(List<ShopMemberDto> shopMemberDtoList) {
		ShopMemberDto myInfo = shopMemberDao.getMyInfo();

		List<ShopMemberDto> insertList = new ArrayList<ShopMemberDto>();

		for (ShopMemberDto shopMemberDto : shopMemberDtoList) {
			if (myInfo.shopMemberId.equals(shopMemberDto.shopMemberId)) {
				shopMemberDto.selfFlg = 1;
			} else {
				shopMemberDto.selfFlg = 0;
			}
			insertList.add(shopMemberDto);
		}
		shopMemberDao.insertShopMember(insertList);
	}


	public void insertShopMemberByMap(HashMap<Integer, List<ShopMemberDto>> shopMemberDtoListMap) {
		ShopMemberDto myInfo = shopMemberDao.getMyInfo();
		List<ShopMemberDto> insertList = new ArrayList<ShopMemberDto>();
		shopMemberDao.beginTransaction();
		Set<Integer> groupIds = shopMemberDtoListMap.keySet();
		for (Integer groupId : groupIds) {
			List<ShopMemberDto> shopMemberDtoList = shopMemberDtoListMap.get(groupId);
			for (ShopMemberDto shopMemberDto : shopMemberDtoList) {
				if (myInfo.shopMemberId.equals(shopMemberDto.shopMemberId)) {
					shopMemberDto.selfFlg = 1;
				} else {
					shopMemberDto.selfFlg = 0;
				}
				insertList.add(shopMemberDto);
			}
			shopMemberDao.insertShopMemberGroupRelation(groupId, shopMemberDtoList);
		}
		shopMemberDao.commit();
		shopMemberDao.insertShopMember(insertList);
	}

	public void updateArchives(List<ArchiveDto> archiveDtoList) {
		List<ArchiveDto> insertList = new ArrayList<ArchiveDto>();
		List<ArchiveDto> deleteList = new ArrayList<ArchiveDto>();
		List<ArchiveDto> updateList = new ArrayList<ArchiveDto>();

		List<Integer> existIdList = archiveDao.getExistArchiveIds();
		for (ArchiveDto archiveDto : archiveDtoList) {
			if (archiveDto.delFlg == 1) {
				deleteList.add(archiveDto);
			} else {
				if (existIdList.contains(archiveDto.archiveId)) {
					updateList.add(archiveDto);
				} else {
					insertList.add(archiveDto);
				}
			}
		}
		archiveDao.insertArchiveList(insertList);
		archiveDao.updateArchiveInfoList(updateList);
		archiveDao.deleteArchiveList(deleteList);
	}

	public void updateArchiveDetial(ArchiveDto archiveDto) {
		archiveDao.updateArchiveDetailInfo(archiveDto);
	}

	public void updateGroup(List<ChatGroupDto> GroupList) {
		List<ChatGroupDto> existGroupList = chatGroupDao.getAllGroups();
		Map<Integer, ChatGroupDto> existGroupMap = convertGroupListToMap(existGroupList);
		ArrayList<ChatGroupDto> insertGroupList = new ArrayList<ChatGroupDto>();
		ArrayList<ChatGroupDto> deleteGroupList = new ArrayList<ChatGroupDto>();
		ArrayList<ChatGroupDto> updateGroupList = new ArrayList<ChatGroupDto>();
		HashMap<Integer, List<ShopMemberDto>> groupMembersMap = new HashMap<Integer, List<ShopMemberDto>>();
		for (ChatGroupDto chatGroupDto : GroupList) {
			if (chatGroupDto.delFlg == 1) {
				deleteGroupList.add(chatGroupDto);
				continue;
			}
			if (existGroupMap.get(chatGroupDto.groupId) == null) {
				insertGroupList.add(chatGroupDto);
			} else {
				updateGroupList.add(chatGroupDto);
			}
			if (chatGroupDto.groupMembers == null) {
				continue;
			}
			groupMembersMap.put(chatGroupDto.groupId, chatGroupDto.groupMembers);
		}
		chatGroupDao.insertGroupList(insertGroupList);
		chatGroupDao.updateGroupList(updateGroupList);
		chatGroupDao.deleteGroupList(deleteGroupList);
		insertShopMemberByMap(groupMembersMap);
	}

	public ShopMemberDto getMyShopMemberDto() {
		return shopMemberDao.getMyInfo();
	}

	public void insertMyInfo(ShopMemberDto shopMemberDto) {
		shopMemberDto.selfFlg = 1;
		shopMemberDao.resetSelfFlg();
		shopMemberDao.insertShopMember(shopMemberDto);
	}

	public void updateFavoriteUser(List<Integer> favoriteUserIds){
		shopMemberDao.updateFavoriteUserList(favoriteUserIds);
	}

	public void updateFavoriteGroup(List<Integer> favoriteGroupIds){
		List<ChatGroupDto> chatGroupDtos = new ArrayList<ChatGroupDto>();
		for (Integer favoriteGroupId : favoriteGroupIds) {
			ChatGroupDto chatGroupDto = new ChatGroupDto();
			chatGroupDto.groupId = favoriteGroupId;
			chatGroupDto.favoriteRegisterDate = DateTimeUtil.toString_yyyyMMddHHmmss_none(new Date(System.currentTimeMillis()));
			chatGroupDtos.add(chatGroupDto);
		}
		chatGroupDao.updateFavoriteGroupList(chatGroupDtos);
	}

	public void deleteavoriteUser(List<Integer> favoriteUserIds){
		for (Integer favoriteUserId : favoriteUserIds) {
			ShopMemberDto favoriteUserDto= new ShopMemberDto();
			favoriteUserDto.shopMemberId = favoriteUserId;
			favoriteUserDto.favoriteRegisterDate = null;
			shopMemberDao.updateFavoriteUser(favoriteUserDto);
		}
	}
	public void deleteavoriteGroup(List<Integer> favoriteGroupIds){
		List<ChatGroupDto> chatGroupDtos = new ArrayList<ChatGroupDto>();
		for (Integer favoriteGroupId : favoriteGroupIds) {
			ChatGroupDto chatGroupDto = new ChatGroupDto();
			chatGroupDto.favoriteRegisterDate = null;
			chatGroupDto.groupId = favoriteGroupId;
			chatGroupDtos.add(chatGroupDto);
		}
		chatGroupDao.updateFavoriteGroupList(chatGroupDtos);
	}

	public void clearAllData() {
		chatGroupDao.deleteChatGroup();
		chatMessageDao.deleteChatMessage();
		chatRoomDao.deleteChatRoom();
		shopMemberDao.deleteShopMember();
		archiveDao.deleteArchiveAllData();
	}

	public void roomDisplayOff(Integer roomId, Integer viewFlg) {
		chatRoomDao.changeRoomViewFlg(roomId, viewFlg);
	}

	public String getUserList(String userIdList) {
		List<ShopMemberDto> selectedUsers = shopMemberDao.getUserList(userIdList);
		JSONArray attendUsers = new JSONArray();
		for (ShopMemberDto dto : selectedUsers) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, dto.shopMemberId);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, getProfileUrlPath(dto.profileUrl));
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, dto.shopMemberName);
			JSONObject jsonObject = new JSONObject(userMap);
			attendUsers.put(jsonObject);
		}
		return attendUsers.toString();
	}

	public String getProfileUrlPath(String profileUrl) {
		if (profileUrl == null) {
			return null;
		}
		String[] filePaths = profileUrl.split("/");
		String fileName = filePaths[filePaths.length-1];
		String filePath = localFilePath + fileName;
		File file = new File(filePath);
		if (file.exists()) {
			return file.toURI().toString();
		} else {
			return profileUrl;
		}
	}

	public String getFileUrlPath(String fileUrlPath) {
		if (!fileUrlPath.contains("fileName=")) {
			return fileUrlPath;
		}
		String fileName = fileUrlPath.split("fileName=")[1].split("&amp;")[0];
		String filePath = localFilePath + fileName;
		File file = new File(filePath);
		if (file.exists()) {
			String replaceTarget = "https" + fileUrlPath.split("https")[1].split("\"")[0];
			String uri = file.toURI().toString();
			String resultPath = fileUrlPath;
			while (resultPath.contains(replaceTarget)) {
				resultPath = resultPath.replace(replaceTarget, uri);
			}
			return resultPath;
		} else {
			return fileUrlPath;
		}
	}

	public void setPackagePath(String localFilePath) {
		this.localFilePath = localFilePath;
	}

	public String getAllArchive() {
		List<ArchiveDto> archiveList = archiveDao.getAllArchive();
		JSONArray archiveArray = new JSONArray();
		for (ArchiveDto archive : archiveList) {
			Map<String, Object> userMap = new HashMap<String, Object>();
			userMap.put(ABookCommConstants.KEY.ARCHIVE_ID, archive.archiveId);
			userMap.put(ABookCommConstants.KEY.ARCHIVE_NAME, archive.archiveName);
			userMap.put(ABookCommConstants.KEY.ARCHIVE_DATE, archive.archiveDate);
			userMap.put(ABookCommConstants.KEY.ARCHIVE_TYPE, archive.archiveType);
			JSONObject jsonObject = new JSONObject(userMap);
			archiveArray.put(jsonObject);
		}
		return archiveArray.toString();
	}

	public String getArchiveListByName(String keywods) {
		String[] replacedKeyword = keywods.replaceAll("　", " ").split(" ");
		List<ArchiveDto> archiveList = archiveDao.getArchiveListByName(replacedKeyword);
		JSONArray archiveArray = new JSONArray();
		for (ArchiveDto archive : archiveList) {
			Map<String, Object> archiveMap = new HashMap<String, Object>();
			archiveMap.put(ABookCommConstants.KEY.ARCHIVE_ID, archive.archiveId);
			archiveMap.put(ABookCommConstants.KEY.ARCHIVE_NAME, archive.archiveName);
			archiveMap.put(ABookCommConstants.KEY.ARCHIVE_DATE, archive.archiveDate);
			archiveMap.put(ABookCommConstants.KEY.ARCHIVE_TYPE, archive.archiveType);
			JSONObject jsonObject = new JSONObject(archiveMap);
			archiveArray.put(jsonObject);
		}
		return archiveArray.toString();
	}

	public String getArchiveDetail(Integer archiveId) {
		ArchiveDto archiveDto = archiveDao.getArchive(archiveId);
		Map<String, Object> archiveMap = new HashMap<String, Object>();
		archiveMap.put(ABookCommConstants.KEY.ARCHIVE_ID, archiveDto.archiveId);
		archiveMap.put(ABookCommConstants.KEY.ARCHIVE_NAME, archiveDto.archiveName);
		archiveMap.put(ABookCommConstants.KEY.ARCHIVE_DATE, archiveDto.archiveDate);
		archiveMap.put(ABookCommConstants.KEY.ARCHIVE_TYPE, archiveDto.archiveType);
		archiveMap.put(ABookCommConstants.KEY.ROOM_ID, archiveDto.roomId);
		archiveMap.put(ABookCommConstants.KEY.ROOM_NAME, archiveDto.roomName);
		archiveMap.put(ABookCommConstants.KEY.SAVE_USER_ID, archiveDto.saveUserId);
		archiveMap.put(ABookCommConstants.KEY.ATTEND_USER_IDS, archiveDto.attendUserIds);
		archiveMap.put(ABookCommConstants.KEY.FILE_PATH, archiveDto.filePath);
		JSONObject jsonObject = new JSONObject(archiveMap);
		return jsonObject.toString();
	}

	public ArchiveDto getArchive(Integer archiveId) {
		return archiveDao.getArchive(archiveId);
	}

	public String getUserInfo(Integer shopMemberId) {
		ShopMemberDto shopMemberDto = shopMemberDao.getShopMember(shopMemberId);
		Map<String, Object> userMap = new HashMap<String, Object>();
		if (shopMemberDto == null) {
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, null);
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, "不明");
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, null);
		} else {
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_ID, shopMemberDto.shopMemberId);
			userMap.put(ABookCommConstants.KEY.SHOP_MEMBER_NAME, shopMemberDto.shopMemberName);
			userMap.put(ABookCommConstants.KEY.PROFILE_URL, shopMemberDto.profileUrl);
		}
		JSONObject jsonObject = new JSONObject(userMap);
		return jsonObject.toString();
	}

	public String makeGroupPath(Integer groupId, String seperator) {
		List<ChatGroupDto> gorupPathList = chatGroupDao.getGroupTree(groupId);
		List<String> groupPathString = new ArrayList<String>();
		for (ChatGroupDto group : gorupPathList) {
			groupPathString.add(group.groupName);
		}
		return StringUtil.join(seperator, groupPathString);
	}

	public String getContentList() {
		List<ContentDto> contentDtoList = contentDao.getDownloadedContents();
		JSONArray resultJsonArray = new JSONArray();
		for (ContentDto contentDto : contentDtoList) {
			Map<String, Object> contentMap = new HashMap<String, Object>();
			contentMap.put(ABookCommConstants.KEY.CONTENT_ID, contentDto.contentId);
			contentMap.put(ABookCommConstants.KEY.CONTENT_NAEM, contentDto.contentName);
			contentMap.put(ABookCommConstants.KEY.THUMBNAIL_NORMAL_PATH, contentDto.thumbnailNormalPath);
			contentMap.put(ABookCommConstants.KEY.THUMBNAIL_BIG_PATH, contentDto.thumbnailBigPath);
			contentMap.put(ABookCommConstants.KEY.RESOURCE_PATH, contentDto.resourcePath);
			JSONObject jsonObject = new JSONObject(contentMap);
			resultJsonArray.put(jsonObject);
		}
		return resultJsonArray.toString();
	}

	public Map<Integer, ChatGroupDto> convertGroupListToMap(List<ChatGroupDto> groupList) {
		Map<Integer, ChatGroupDto> map = new HashMap<Integer, ChatGroupDto>();
		for (ChatGroupDto group : groupList) {
			map.put(group.groupId, group);
		}
		return map;
	}

	public void updateUnreadCount(List<ChatMessageDto> messageList, String roomId) {
		chatMessageDao.updatUnreadCount(messageList, roomId);
	}

	/**
	 * 協業招待メッセージからCollaborationTypeを取得
	 * @param message プッシュメッセージ
	 * @return 協業タイプ
	 */
	public String getCollaborationType(String message) {
		try {
			if (message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_AUDIO.EN) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_AUDIO.JP) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_AUDIO.KR)
			) {
				return String.valueOf(ABookCommConstants.FLAG.COLLABORATION_TYPE.AUDIO);
			}else if (message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_CAMERA.EN) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_CAMERA.JP) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_CAMERA.KR)
			) {
				return String.valueOf(ABookCommConstants.FLAG.COLLABORATION_TYPE.CAMERA);
			}else if (message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_VIDEO.EN) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_VIDEO.JP) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_VIDEO.KR)
			) {
				return String.valueOf(ABookCommConstants.FLAG.COLLABORATION_TYPE.VIDEO);
			}else if (message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_DOCUMENT.EN) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_DOCUMENT.JP) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_DOCUMENT.KR)
			) {
				return String.valueOf(ABookCommConstants.FLAG.COLLABORATION_TYPE.DOCUMENT);
			}else if (message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_BOARD.EN) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_BOARD.JP) ||
					message.contains(ABookCommConstants.COLLABORATION.COLLABORATION_BOARD.KR)
			) {
				return String.valueOf(ABookCommConstants.FLAG.COLLABORATION_TYPE.BOARD);
			}
			return null;
		} catch (Exception e) {
			return null;
		}
	}
}
