package jp.agentec.abook.abv.launcher.android;

import android.app.DownloadManager;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.IBinder;
import androidx.core.content.FileProvider;

import java.io.File;

import jp.agentec.abook.abv.bl.acms.type.AcmsApis;
import jp.agentec.abook.abv.bl.common.ABVEnvironment;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.data.ABVDataCache;
import jp.agentec.adf.util.DateTimeFormat;
import jp.agentec.adf.util.DateTimeUtil;

public class BackgroundDownloadService  extends Service {
    private final static String LOG_TAG = BackgroundDownloadService.class.getSimpleName();
    private DownloadManager mDownloadManager;
    private long mDownloadedFileID;
    private String notificationChannelId;
    private String notificationChannelIdLow;
    private NotificationManager mNotificationManager;

    @Override
    public void onCreate() {
        super.onCreate();
        mDownloadManager = (DownloadManager) getSystemService(Context.DOWNLOAD_SERVICE);
        notificationChannelId = getApplicationContext().getPackageName();
        notificationChannelIdLow += ".Low";

        if (mNotificationManager == null) {
            mNotificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);

            NotificationChannel notificationChannel = new NotificationChannel(notificationChannelId, getString(R.string.app_name), NotificationManager.IMPORTANCE_DEFAULT);
            NotificationChannel notificationChannelLow = new NotificationChannel(notificationChannelIdLow, getString(R.string.app_name), NotificationManager.IMPORTANCE_LOW);

            if (mNotificationManager != null) {
                mNotificationManager.createNotificationChannel(notificationChannel);
                mNotificationManager.createNotificationChannel(notificationChannelLow);
            }
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        if (intent == null) {
            return START_STICKY;
        }

        BroadcastReceiver onDownloadComplete = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                unregisterReceiver(this);

                // Prevents the occasional unintentional call.
                if (mDownloadedFileID == -1) {
                    return;
                }

                try {
                    File file = new File(context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS), ABVEnvironment.APK_FILE_NAME);
                    if (file.exists()) {
                        Uri apkUri = FileProvider.getUriForFile(context, context.getPackageName() + ".provider", file);
                        Intent updateIntent = new Intent(Intent.ACTION_VIEW);
                        updateIntent.setDataAndType(apkUri, "application/vnd.android.package-archive");
                        updateIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                        updateIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);

                        // Android10(Q)
                        if (Build.VERSION.SDK_INT >= 29) {
                            sendNotify(updateIntent, getString(R.string.app_update), false);
                            Logger.d(LOG_TAG, "sendNotify(updateIntent...).");
                        } else {
                            stopForeground(true);
                            startActivity(updateIntent);
                            Logger.d(LOG_TAG, "startActivity(fileIntent).");
                        }
                    } else {
                        sendNotify(intent, getString(R.string.DOWNLOAD_ERROR), false);
                        Logger.w(LOG_TAG, "sendNotify(intent...).");
                    }
                } catch (Exception ex) {
                    sendNotify(intent, getString(R.string.DOWNLOAD_ERROR), false);
                    Logger.e(LOG_TAG, "startActivity(fileIntent).", ex);
                } finally {
                    // Sets up the prevention of an unintentional call.
                    mDownloadedFileID = -1;
                }
            }
        };

        try {
            ABVEnvironment abvEnvironment = ABVEnvironment.getInstance();
            ABVDataCache dataCache = ABVDataCache.getInstance();
            String currentDate = DateTimeUtil.toString(DateTimeUtil.getCurrentTimestamp(), DateTimeFormat.yyyyMMddHHmmss000_none);
            String downloadUrl = AcmsApis.getDownloadApplicationFileUrl(abvEnvironment.acmsAddress, dataCache.getUrlPath(), dataCache.getMemberInfo().sid, currentDate);
            Logger.d(LOG_TAG, "downloadUrl=%s", downloadUrl);
            DownloadManager.Request request = new DownloadManager.Request(Uri.parse(downloadUrl));
            request.setDescription("ABook Plus New Version File");
            //LANケーブル接続のタイプ(ETHERNET TYPE)がないため、セットしない
            //request.setAllowedNetworkTypes(DownloadManager.Request.NETWORK_MOBILE | DownloadManager.Request.NETWORK_WIFI);

            File file = new File(getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS), ABVEnvironment.APK_FILE_NAME);
            Logger.d(LOG_TAG, "download local file=%s", file.getAbsolutePath());
            File[] childs = file.getParentFile().listFiles();
            if (childs != null && childs.length > 0 && childs[0].exists()) {
                childs[0].delete();
            }

            request.setDestinationUri(Uri.fromFile(file));

            // Registers function to listen to the completion of the download.
            registerReceiver(onDownloadComplete, new IntentFilter(DownloadManager.ACTION_DOWNLOAD_COMPLETE));

            // Download 開始
            Logger.d(LOG_TAG, "mDownloadManager.enqueue(request).");
            mDownloadedFileID = mDownloadManager.enqueue(request);

            //５秒以内にstartForegroundを呼ばないとクラッシュ
            int uniqueId = (int) System.currentTimeMillis();
            Notification notification = getNotification(uniqueId, intent, getString(R.string.download_start), true);
            startForeground(uniqueId, notification);
        } catch (Exception ex) {
            Logger.e(LOG_TAG, "mDownloadManager.enqueue(request).", ex);
        }

        return super.onStartCommand(intent, flags, startId);
    }

    private void sendNotify(Intent intent, String message, boolean isImportanceLow) {
        stopForeground(true);

        int uniqueId = (int) System.currentTimeMillis();
        Notification notification = getNotification(uniqueId, intent, message, isImportanceLow);

        if (mNotificationManager != null) {
            mNotificationManager.notify(uniqueId, notification);
            Logger.d(LOG_TAG, "sendNotify()");
        }
    }

    private Notification getNotification(int uniqueId, Intent intent, String message, boolean isImportanceLow) {
        PendingIntent pendingIntent;
        if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            pendingIntent = PendingIntent.getActivity(this, uniqueId, intent, PendingIntent.FLAG_UPDATE_CURRENT|PendingIntent.FLAG_IMMUTABLE);
        } else {
            pendingIntent = PendingIntent.getActivity(this, uniqueId, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        }
        return new Notification.Builder(this)
                .setDefaults(Notification.DEFAULT_ALL)
                .setSmallIcon(R.drawable.icon)
                .setWhen(System.currentTimeMillis())
                .setAutoCancel(true)
                .setContentTitle(getString(R.string.app_name))
                .setContentText(message)
                .setContentIntent(pendingIntent)
                .setChannelId(isImportanceLow ? notificationChannelIdLow : notificationChannelId)
                .build();
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }
}
