package jp.agentec.abook.abv.bl.common.exception;

import java.security.NoSuchAlgorithmException;

import jp.agentec.abook.abv.bl.logic.UserAuthenticateLogic;

/**
 * UIに渡すエラーコードです。<br>
 * エラーコードの体系は以下の通りです。<br>
 * <em>{C or S}</em>_<em>{W or E}</em>_<em>{ACMS or ABV}</em>_<em>{コード番号4桁}</em><br>
 * <li>{C or S} C : クライアントから判断した警告又はエラー、 S : サーバからのエラー</li>
 * <li>{W or E} W : 警告、 E : エラー</li>
 * <li>{ACMS or ...} ACMS : サーバとの通信関係</li>
 * <li>{コード番号4桁} 連番</li>
 * @author Taejin Hong
 * @version 1.1.2
 */
public enum ABVExceptionCode {
	/**
	 * クライアント側の不明なエラーです。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0001,
	/**
	 * {@link NoSuchAlgorithmException} によるエラーです。つまり、システムで、暗号化アルゴリズムが使えません。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0002,
	/**
	 * {@link ClassNotFoundException} によるエラーです。OSに依存するライブラリの初期化に失敗した可能性が高いです。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0003,
	/**
	 * WI-FIアダプターの電源が入っていません。初期ログイン時は、WI-FIをONにしなければなりません。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0004,
	/**
	 * コンテンツフォルダにアクセスできません。デバイスの再起動をお勧めします。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0101,
	/**
	 * 内部ストレージの容量が不足しています。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0102,
	/**
	 * 外部ストレージの容量が不足しています。
	 * @since 1.0.0
	 */
	C_E_SYSTEM_0103,
	/**
	 * API {@link UserAuthenticateLogic#newAppStoreLogin} の応答で、resultがfailedだったのに、requirePasswordChangeの値が0である。
	 * @since 1.0.0
	 */
	C_E_SECURITY_1001,
	/**
	 * ログインパスワードのAES暗号化が失敗しました。
	 * @since 1.0.0
	 */
	C_E_SECURITY_1002,
	/**
	 * ログインパスワードのAES復号化が失敗しました。
	 * @since 1.0.0 
	 */
	C_E_SECURITY_1003,
	/**
	 * ログイン中だと認識されている状態だが、キャッシュにユーザ情報がありません。
	 * @since 1.0.0
	 */
	C_E_SECURITY_1004,
	/**
	 * 認証失敗です。パスワードが違います。
	 * @since 1.0.0
	 */
	C_E_SECURITY_1005,
	/**
	 * ユーザ切替です。既存データを削除するか確認します。
	 * @since 1.0.0
	 */
	C_E_SECURITY_1006,
	/**
	 * 新着コンテンツの更新が失敗しました。
	 * @since 1.0.0
	 */
	C_E_CONTENT_1001,
	/**
	 * コンテンツのダウンロードができません。存在しないコンテンツIDです。
	 * @since 1.0.0
	 */
	C_E_CONTENT_1002,
	/**
	 * ZIP解凍が失敗しました。
	 * @since 1.1.2
	 */
	C_E_CONTENT_1003,
	/**
	 * ダウンロードキューのクリアに失敗しました。
	 * @since 1.0.0
	 */
	C_E_CONTENT_1004,
	/**
	 * JSONデータが不正です。
	 * @since plus 1.1.0
	 */
	C_E_CONTENT_2001,
	/**
	 * 既に新着コンテンツの更新中です。
	 * @since 1.0.0
	 */
	C_I_CONTENT_0001,
	/**
	 * コンテンツをダウンロードしているため、自動更新は行いません。
	 * @since 1.1.2
	 */
	C_I_CONTENT_0002,
	/**
	 * ACMSのAPIの応答で、HTTP status codeが定義済みではありません。不明なエラーです。
	 * @since 1.0.0
	 */
	S_E_ACMS_0001,
	/**
	 * ACMSのAPIの応答で、HTTP status codeが400です。バリデータエラーです。
	 * @since 1.0.0
	 */
	S_E_ACMS_0400,
	/**
	 * ACMSのAPIの応答で、HTTP status codeが403です。認証エラーです。
	 * @since 1.0.0
	 */
	S_E_ACMS_0401,
	/**
	 * ACMSのAPIの応答で、HTTP status codeが401です。認証要求エラーです。
	 * @since 1.0.0
	 */
	S_E_ACMS_0403,
	/**
	 * ACMSのAPIの応答で、HTTP status codeが404です。APIが存在しません。
	 * @since 1.0.0
	 */
	S_E_ACMS_0404,
	/**
	 * ACMSのAPIの応答で、HTTP status codeが500です。システムエラーです。
	 * @since 1.0.0
	 */
	S_E_ACMS_0500,
	/**
	 * ACMSのAPIの応答で、不正な値が返ってきた場合のエラーです。システムエラーです。（エラーページが返ってきた場合やサーバ側のバグ）
	 * @since 1.0.0
	 */
	S_E_ACMS_INVALID_RESPONSE,
	/**
	 * ログインではないAPIの利用で、認証失敗（403）になった場合です。再ログインが必要です。
	 * @since 1.1.1
	 */
	S_E_ACMS_1403,
	/**
	 * ユーザ認証失敗です。ユーザＩＤ又はパスワードが違います。
	 * @since 1.0.0
	 */
	S_E_ACMS_L001,
	/**
	 * ユーザ認証失敗です。別のユーザが端末を利用中です。
	 * @since 1.0.0
	 */
	S_E_ACMS_L002,
	/**
	 * ユーザ認証失敗です。別のユーザが別の端末を使用中です。
	 * @since 1.0.0
	 */
	S_E_ACMS_L004,
	/**
	 * ユーザ認証失敗です。ユーザが属する事業者がサービス利用中ではありません。
	 * @since 1.0.0
	 */
	S_E_ACMS_L005,
	/**
	 * 端末認証失敗です。端末が未登録（自動登録無し）状態です。
	 * @since 1.0.0
	 */
	S_E_ACMS_L006,
	/**
	 * この事業者は現在利用できません。
	 * @since 1.1.0
	 */
	S_E_ACMS_L008,
	/**
	 * ご利用のユーザアカウントに別の端末のIDが登録されています。
	 * @since 1.1.0
	 */
	S_E_ACMS_L009,
	/**
	 * 端末IDが特定できません。
	 * @since 1.1.0
	 */
	S_E_ACMS_L010,
	/**
	 * 端末IDが特定できません。
	 * @since 4.0.0
	 */
	S_E_ACMS_L012,
	/**
	 * コンテンツ名に半角カタカナは使用できません。
	 */
	P_E_ACMS_P001,
	/**
	 *  既にコンテンツが登録されています。
	 */
	P_E_ACMS_P002,
	/**
	 *  このプロジェクトの作業は新規登録・修正・削除出来ません。
	 */
	P_E_ACMS_P003,
	/**
	 *  このプロジェクトの作業はサーバで閲覧中です。しばらく経ってから再度同期処理を実行してください。
	 */
	P_E_ACMS_P004,
    /**
     * 定期点検、既に点検済みです。点検データ送信できません。
     */
    P_E_ACMS_P005,
	/**
	 * 既に作業報告を行う権限がありません。
	 */
	P_E_ACMS_P006,
	/**
	* ロック中である場合
	*/
	P_E_ACMS_P007;

	/**
	 * HTTPレスポンスコードから対応するコードを返す
	 * 
	 * @param responseCode
	 * @return
	 */
	public static ABVExceptionCode fromResponseCode(int responseCode) {
		switch (responseCode) {
		case 400:
			return S_E_ACMS_0400;
		case 401:
			return S_E_ACMS_0401;
		case 403:
			return S_E_ACMS_0403;
		case 404:
			return S_E_ACMS_0404;
		case 500:
			return S_E_ACMS_0500;
		}
		return S_E_ACMS_0001;
	}
}
