package jp.agentec.abook.abv.bl.common.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;

import jp.agentec.abook.abv.bl.acms.client.json.DownloadedContentInfoJSON;
import jp.agentec.abook.abv.bl.common.ABVEnvironment;
import jp.agentec.abook.abv.bl.common.exception.AcmsException;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.adf.util.FileUtil;
import jp.agentec.adf.util.NumericUtil;
import jp.agentec.adf.util.StringUtil;

public class ContentFileUtil {
	private static final String TAG = "ContentFileUtil";
	
	public static final String SIZE_L = "l";
	public static final String SIZE_M = "m";
	public static final String SIZE_S = "s";
	
	public static File[] getBigThumbnails(File dir) {
		//	大サムネール（nnnn...nnn__THUMBIG.jpg）のみ取得
		File[] bigThumbnail = dir.listFiles(new FilenameFilter() {
			@Override
			public boolean accept(File dir, String name) {
				return name.endsWith("THUMBIG.jpg");
			}
		});
		
		
		return bigThumbnail;
	}
	
	public static File[] getPageThumbnails(File dir) {
		//	ページのサムネール(page_n.jpg)のみ取得
		File[] pageThumbnails = dir.listFiles(new FilenameFilter() {
			@Override
			public boolean accept(File dir, String name) {
				return (FileUtil.getExtension(name).equals("jpg") && FileUtil.getFilenameWithoutExt(name).startsWith("page_"));
			}
		});
		
		return pageThumbnails;
	}
	
	public static File[] getPageTexts(File dir) {
		//	ページテキストファイル（pageText_n.txt）のみ取得
		File[] pageTexts = dir.listFiles(new FilenameFilter() {
			@Override
			public boolean accept(File dir, String name) {
				return (FileUtil.getExtension(name).equals("txt") && FileUtil.getFilenameWithoutExt(name).startsWith("pageText_"));
			}
		});
		
		return pageTexts;
	}
	
	public static int getPageNumFromFileName(String fileName) {
		int delimiterIndex = fileName.lastIndexOf("_");
		
		if (delimiterIndex >= 0 && fileName.length() > delimiterIndex) {
			return NumericUtil.parseInt(fileName.substring(delimiterIndex + 1));
		} else {
			return -1;
		}
	}
	
	public static String getStringFromPageTextFile(File f) throws IOException {
		StringBuffer sb = new StringBuffer();
		
		BufferedReader reader = null;
		
		try {
			reader = new BufferedReader(new FileReader(f));
			String line;
						
			while ((line = reader.readLine()) != null) {
				//	form-feed(0x0c -> \f)は無視する。
				//	おそらく、PDFからテキストをとる時に、form-feedまで入れられたようだ。
				if (line.equals(new String(new byte[]{0x0c}))) {
					continue;
				} else {
					StringUtil.appendLine(sb, line);
				}
			}
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (Exception e) {}
			}
		}
		
		return sb.toString();
	}
	
	/**
	 * 指定したコンテンツIDのjson.txtの情報を取得します。
	 * @param contentId コンテンツID
	 * @return
	 * @throws IOException 
	 * @throws AcmsException 
	 */
	public static DownloadedContentInfoJSON getDownloadedContentInfoJSON(long contentId) throws IOException, AcmsException {
		String contentInfoFilePath = ABVEnvironment.getInstance().getContentDirectoryPath(contentId, false) + File.separator + ABVEnvironment.ContentVersionInfoJsonName;
		DownloadedContentInfoJSON json = getDownloadedContentInfoJSON(contentInfoFilePath);

		return json;
	}
	
	
	/**
	 * 指定したファイル(ファイル名までのフルパス)を読み込んでDownloadedContentInfoJSONとして返します。
	 * @param contentInfoFilePath ファイルパス(ファイル名まで)
	 * @return
	 * @throws IOException 
	 * @throws AcmsException 
	 * @throws Exception
	 */
	public static DownloadedContentInfoJSON getDownloadedContentInfoJSON(String contentInfoFilePath) throws IOException, AcmsException {
		String jsonText = FileUtil.readTextFile(contentInfoFilePath);
		Logger.d(TAG, "%s : %s", contentInfoFilePath, jsonText);
		return new DownloadedContentInfoJSON(jsonText);
	}
	
	/**
	 * 指定したコンテンツのサムネイルファイルを全て削除します。
	 * 
	 * @param contentId コンテンツID
	 * @throws Exception その他、例外です。
	 */
	public static void deleteContentThumbnailFile(long contentId) {
		String thumPath = ABVEnvironment.getInstance().getPdfThumbnailDirectory(contentId);
		FileUtil.deleteChilds(thumPath);
	}
	
	/**
	 * サムネイルのパスを返す
	 * 
	 * @param contentId コンテンツID
	 * @param pageNum ページナンバー
	 * @return
	 * @throws Exception その他。例外です。
	 */
	public static String getPdfThumbnailPath(long contentId, int pageNum, String sizeType) {
		return ABVEnvironment.getInstance().getPdfThumbnailDirectory(contentId) + "/" + pageNum + "_" + sizeType + ".jpg";
	}
	
	/**
	 * PDFイメージのパスを渡す
	 * 
	 * @param contentId
	 * @param pageNum
	 * @return
	 */
	public static String getPdfImagePath(long contentId, int pageNum) {
		return ABVEnvironment.getInstance().getContentCacheDirectory(contentId) + "/pdfPageImage_" + pageNum + ".jpg";
	}
	
	/**
	 * 動画サムネイルのパスを返す(最初)
	 * 
	 * @param moviePath
	 * @return
	 */
	public static String getMovieImageStartPath(String moviePath) {
		return moviePath.substring(0, moviePath.lastIndexOf(".")) + "_THUM_S.jpg";
	}

	/**
	 * 動画サムネイルのパスを返す(最後)。
	 * 
	 * @param moviePath
	 * @return
	 */
	public static String getMovieImageEndPath(String moviePath) {
		return moviePath.substring(0, moviePath.lastIndexOf(".")) + "_THUM_E.jpg";
	}


}
