package jp.agentec.abook.abv.ui.common.helper;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.Display;
import android.view.LayoutInflater;
import android.view.View;
import android.view.WindowManager;
import android.widget.TextView;

import androidx.annotation.NonNull;

import jp.agentec.abook.abv.bl.acms.client.json.AgreementToTermsJSON;
import jp.agentec.abook.abv.bl.common.CommonExecutor;
import jp.agentec.abook.abv.bl.common.constant.ABookKeys;
import jp.agentec.abook.abv.bl.common.exception.NetworkDisconnectedException;
import jp.agentec.abook.abv.bl.common.log.Logger;
import jp.agentec.abook.abv.bl.data.ABVDataCache;
import jp.agentec.abook.abv.bl.logic.AbstractLogic;
import jp.agentec.abook.abv.bl.logic.AgreementToTermsLogic;
import jp.agentec.abook.abv.cl.util.PreferenceUtil;
import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.common.activity.ABVActivity;
import jp.agentec.abook.abv.ui.common.util.AlertDialogUtil;

/**
 * 規約同意ダイアログを表示するヘルパー
 */
public class AgreementToTermsHelper {
    private static final String TAG = "AgreementToTermsHelper";

    /**
     * 規約同意状態に応じて動作するリスナー
     */
    public interface AgreementListener {
        // 規約同意オプションがNoだった
        void onDisabled();
        // 規約内容やバージョンが空だった
        void onEmpty();
        // 規約情報の取得に失敗した
        void onFailed(Exception e);
        // すでに同意済みだった
        void onAgreed();
        // 同意するまたはしないをタップされた
        void onConfirmed(boolean isAgreed);
        // 規約情報の取得以外でエラーがあった
        void onError(Exception e);
    }

    private static AlertDialog currentDialog = null;

    private static synchronized void setCurrentDialog(AlertDialog dialog) {
        currentDialog = dialog;
    }

    public static synchronized AlertDialog getCurrentDialog() {
        return currentDialog;
    }


    /**
     * 同意規約情報を読み込んで、バージョンが異なっていたら、ダイアログを表示
     * バージョンが同じだったら何もしない
     */
    public static void confirmAgreement(final ABVActivity context, final boolean isNomalSize, @NonNull final AgreementListener listener) {
        if (! ABVDataCache.getInstance().serviceOption.isUsableAgreementToTerms()) {
            // 規約同意オプションが無効な場合、表示しない
            listener.onDisabled();
            return;
        }

        context.showProgressPopup();
        CommonExecutor.execute(new Runnable() {
            @Override
            public void run() {
                AgreementToTermsJSON termsJSON;
                try {
                    AgreementToTermsLogic logic = AbstractLogic.getLogic(AgreementToTermsLogic.class);
                    termsJSON = logic.getTerms();
                    if (termsJSON == null || termsJSON.getTerms().isEmpty() || termsJSON.getTermsVersion().isEmpty()) {
                        // 規約内容やバージョンが空の場合は表示しない
                        listener.onEmpty();
                        return;
                    }
                    if (getLatestVersion(context).equals(termsJSON.getTermsVersion())) {
                        // 規約のバージョンが、すでに同意しているものと同じときは表示しない
                        listener.onAgreed();
                        return;
                    }
                    final AgreementToTermsJSON termsJSON1 = termsJSON;
                    context.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            // 同意画面表示
                            showAgreementDialog(context, termsJSON1, isNomalSize, listener);
                        }
                    });
                } catch (Exception e) {
                    Logger.e(TAG, e);
                    // 取得に失敗した場合
                    listener.onFailed(e);
                } finally {
                    context.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            context.closeProgressPopup();
                        }
                    });
                }
            }
        });
    }

    /**
     * 実際にダイアログを表示するメソッド
     *
     * @param activity      土台になるActivity
     * @param termsJSON     規約情報
     * @param isNomalSize   画面サイズがスマホならtrue
     * @param listener      状態に応じてコールバックを受け取るリスナー
     */
    private static void showAgreementDialog(final ABVActivity activity, final AgreementToTermsJSON termsJSON, final boolean isNomalSize, @NonNull final AgreementListener listener) {
        try {
            AlertDialog dialog = makeDialog(activity, termsJSON, isNomalSize, listener);
            dialog.show();
            WindowManager.LayoutParams params = dialog.getWindow().getAttributes();
            WindowManager wm = (WindowManager)activity.getSystemService(Context.WINDOW_SERVICE);
            Display disp = wm.getDefaultDisplay();
            if (isNomalSize) {
                params.width = disp.getWidth() * 49 / 50;
            } else {
                params.width = disp.getWidth() * 4 / 5;
            }
            dialog.getWindow().setAttributes(params);
            dialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    setCurrentDialog(null);
                }
            });
            setCurrentDialog(dialog);
        } catch (Exception e) {
            listener.onError(e);
        }
    }

    /**
     * ダイアログを組み立てるメソッド
     *
     * @param activity      土台になるActivity
     * @param termsJSON     規約情報
     * @param isNomalSize   画面サイズがスマホならtrue
     * @param listener      状態に応じてコールバックを受け取るリスナー
     * @return  規約同意画面になるダイアログ
     */
    private static AlertDialog makeDialog(final ABVActivity activity, final AgreementToTermsJSON termsJSON, final boolean isNomalSize, @NonNull final AgreementListener listener) {
        // ダイアログ表示
        AlertDialog.Builder builder = new AlertDialog.Builder(activity, R.style.Theme_MyDialog);
        builder.setTitle(R.string.terms_of_service);

        LayoutInflater inflater = activity.getLayoutInflater();
        View layout = inflater.inflate(R.layout.dlg_agreement_to_terms, null, false);

        TextView versionText = layout.findViewById(R.id.textVersion);
        TextView termsText = layout.findViewById(R.id.textTerms);
        versionText.setText(termsJSON.getTermsVersion());
        termsText.setText(termsJSON.getTerms());

        builder.setView(layout);
        builder.setCancelable(false);

        final AlertDialog dialog = builder.create();

        layout.findViewById(R.id.agreeButton).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                agree(activity, dialog, termsJSON.getTermsVersion(), isNomalSize, listener);
            }
        });
        layout.findViewById(R.id.disagreeButton).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                disagree(activity, dialog, isNomalSize, listener);
            }
        });
        return dialog;
    }

    /**
     * 同意するをタップしたときの処理
     *
     * @param activity      土台となるActivity
     * @param dialog        規約同意画面のダイアログ
     * @param currentVersion    同意する規約のバージョン
     * @param isNomalSize   画面サイズがスマホならtrue
     * @param listener      状態に応じてコールバックを受け取るリスナー
     */
    private static void agree(final ABVActivity activity, final AlertDialog dialog, final String currentVersion, final boolean isNomalSize, final AgreementListener listener) {
        activity.showProgressPopup();
        CommonExecutor.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    // 同意したことをCMSに送信
                    AgreementToTermsLogic logic = AbstractLogic.getLogic(AgreementToTermsLogic.class);
                    logic.agreeTerms(currentVersion);
                    // 同意したバージョンを保存して閉じる
                    activity.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            setLatestVersion(activity, currentVersion);
                            dialog.dismiss();
                            listener.onConfirmed(true);
                        }
                    });
                } catch (final Exception e) {
                    Logger.e(TAG, e);
                    // 失敗したらアラート表示
                    activity.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            int messageId;
                            if (e instanceof NetworkDisconnectedException) {
                                messageId = R.string.msg_network_offline;
                            } else {
                                messageId = R.string.failed_to_send_agreement;
                            }
                            AlertDialogUtil.showAlertDialog(activity, R.string.app_name, messageId);
                        }
                    });
                } finally {
                    activity.closeProgressPopup();
                }
            }
        });
    }

    /**
     * 同意しないをタップしたときの処理
     *
     * @param activity      土台となるActivity
     * @param dialog        規約同意画面のダイアログ
     * @param isNomalSize   画面サイズがスマホならtrue
     * @param listener      状態に応じてコールバックを受け取るリスナー
     */
    private static void disagree(final ABVActivity activity, final AlertDialog dialog, final boolean isNomalSize, final AgreementListener listener) {
        AlertDialogUtil.showAlertDialog(activity, R.string.app_name, R.string.logout_by_disagree, false,
                new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        // ログアウトしてよし
                        listener.onConfirmed(false);
                        dialog.dismiss();
                    }
                }
        );
    }

    /**
     * 最後に同意した規約のバージョンの取得
     *
     * @param context   コンテクスト
     * @return  最後に同意した規約のバージョン
     */
    private static String getLatestVersion(Context context) {
        return PreferenceUtil.getUserPref(context, ABookKeys.AGREEMENT_TO_TERMS_VERSION, "");
    }

    /**
     * 最後に同意した規約のバージョンの保存
     *
     * @param context   コンテクスト
     * @param version   規約のバージョン
     */
    private static void setLatestVersion(Context context, String version) {
        PreferenceUtil.putUserPref(context, ABookKeys.AGREEMENT_TO_TERMS_VERSION, version);
    }
}
