package jp.agentec.abook.abv.ui.viewer.adapter;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import jp.agentec.abook.abv.launcher.android.R;
import jp.agentec.abook.abv.ui.common.dialog.ABookAlertDialog;
import jp.agentec.abook.abv.ui.common.util.AlertDialogUtil;
import jp.agentec.abook.abv.ui.viewer.activity.ContentViewActivity;

import org.json.adf.JSONArray;
import org.json.adf.JSONObject;

import android.content.DialogInterface;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

public class PDFIndexDialog {

	private static final String TAG = "PDFIndexDialog";
	private final int ROOT_ID = 0; // ルートID
	private List<ContentPDFIndexDto> dataList; // データリスト
	private ContentViewActivity activity; // 呼出元アクティビティ
	private Stack<Integer> rid; // 戻る用IDスタック

	public PDFIndexDialog(ContentViewActivity activity, JSONArray json) {
		this.activity = activity;
		makeDataList(json);
		rid = new Stack<>();
	}

	/**
	 * Jsonデータをもとにリストを作成して設定します。
	 * 
	 * @param json
	 */
	private void makeDataList(JSONArray json) {
		dataList = new ArrayList<>();

		if (null != json) {
			for (int i = 0; json.length() > i; i++) {
				JSONObject obj = json.getJSONObject(i);
				ContentPDFIndexDto dto = new ContentPDFIndexDto();
				dto.setId(obj.getInt("ID"));
				dto.setDestPageNumber(obj.getInt("destPageNumber"));
				dto.setParentID(obj.getInt("parentID"));
				dto.setTitle(obj.getString("title"));
				dto.setParent(checkIsParent(dto.getId(), json));
				dataList.add(dto);
			}
		}
	}

	
	/**
	 * 指定したIDが子を持っているか判定して結果を返します。
	 * 
	 * @param id ID
	 * @param json Jsonデータ
	 * @return boolean 判定結果
	 */
	private boolean checkIsParent(int id, JSONArray json) {
		boolean result = false;

		if (null != json) {
			for (int i = 0; json.length() > i; i++) {
				JSONObject obj = json.getJSONObject(i);
				if (obj.getInt("parentID") == id) {
					result = true;
					break;
				}
			}
		}
		return result;
	}
	
	/**
	 * ルートIDを親に持つインデックスの一覧をダイアログとして表示します。
	 */
	public void showDialog() {
		showDialog(ROOT_ID);
	}

	/**
	 * 指定したIDを親に持つインデックスの一覧をダイアログとして表示します。
	 * 
	 * @param pid 親ID
	 */
	public void showDialog(int pid) {
		final ABookAlertDialog alert = AlertDialogUtil.createAlertDialog(activity, R.string.index);

		LayoutInflater inflater = LayoutInflater.from(activity);
		View layout = inflater.inflate(R.layout.pdfindex_render, null);

		// 戻るボタン
		ImageButton rbtn = (ImageButton) layout.findViewById(R.id.pdfindex_btn_return);
		rbtn.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				try {
					showDialog(rid.pop());
				} finally {
					alert.dismiss();
				}
			}
		});
		
		// 戻る領域拡張
		LinearLayout rlin = (LinearLayout) layout.findViewById(R.id.pdfindex_linear_return);
		rlin.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				try {
					showDialog(rid.pop());
				} finally {
					alert.dismiss();
				}
			}
		});
		
		// 親IDが0の場合、戻るボタン非表示
		if (pid == 0) {
			rlin.setVisibility(View.GONE);
		}

		// インデックスリスト
		ListView lv = (ListView) layout.findViewById(R.id.pdfindex_list);
		final ABVPDFIndexAdapter adapter = new ABVPDFIndexAdapter(activity, makeListItem(pid), alert, this);

		lv.setAdapter(adapter);
		lv.setOnItemClickListener(new AdapterView.OnItemClickListener() {
			@Override
			public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
				// ページジャンプ
				activity.jumpToPage(adapter.getItem(position).getDestPageNumber() - 1);
				alert.dismiss();
			}
		});

		TextView etv = (TextView) layout.findViewById(R.id.txt_empty);

		if (adapter.getCount() > 0) {
			// リストが空でない場合、空通知テキスト非表示
			etv.setVisibility(View.GONE);
		} else {
			// リストが空の場合、リスト非表示
			lv.setVisibility(View.GONE);
		}

		// キャンセルボタン
		alert.setNegativeButton(R.string.cancel, new DialogInterface.OnClickListener() {
			@Override
			public void onClick(DialogInterface dialog, int which) {
				dialog.dismiss();
			}
		});

		// ダイアログ表示
		alert.setView(layout);
		alert.show();
	}

	/**
	 * 指定したIDを親に持つインデックスを抽出してリストを返します。
	 * 
	 * @param pid 親ID
	 * @return List
	 */
	private List<ContentPDFIndexDto> makeListItem(int pid) {
		List<ContentPDFIndexDto> listItem = new ArrayList<>();

		for (ContentPDFIndexDto dto : dataList) {
			if (dto.getParentID() == pid) {
				listItem.add(dto);
			}
		}
		return listItem;
	}

	/**
	 * 戻るボタンを押したときに表示するリストの親IDをスタックにプッシュします。
	 * 
	 * @param id ID
	 */
	public void pushRid(int id) {
		rid.push(id);
	}
}
